#include "gamewidget.h"

#include <QGraphicsScene>
#include <QCoreApplication>
#include <QMouseEvent>
#include <QKeyEvent>
#include <QGraphicsRectItem>
#include <QGraphicsTextItem>
#include <QTimer>
#include <QDebug>
#include <QGraphicsPixmapItem>
#include "mygraphicsview.h"
#include "mickymouse.h"
#include "egg.h"
#include "hen.h"
#include "remainingtrialitem.h"
#include "baseitem.h"
#include "textbutton.h"
#include "pausedmenu.h"
#include "scorecardscreen.h"
#include "mainmenuscreen.h"
#include "hatchingegg.h"

GameWidget::GameWidget(QWidget *parent) :
    QMainWindow(parent),mMaxEggCount(3),mPoints(0),mMinSpeed(3),mMaxSpeed(5),mCurrentScreen(0)
{
    this->setWindowTitle("Crazy Chickens");

    mScene = new QGraphicsScene( 0, 0, 800, 480, this );
    MyGraphicsView* view = new MyGraphicsView(this);
    view->setRenderHint(QPainter::Antialiasing);
    view->setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    view->setScene( mScene );
    view->show();
    this->setCentralWidget( view );

    QGraphicsPixmapItem* background = new QGraphicsPixmapItem(QPixmap(":/image/gamebg.png"),0,mScene);

    mTrialItem = new RemainingTrialItem( 4 );
    mTrialItem->setPos(800 - mTrialItem->boundingRect().width() - 100,5);
    mScene->addItem( mTrialItem );

    mScoreItem = new QGraphicsTextItem();
    QFont font = mScoreItem->font();
    font.setPointSize( font.pointSize() + 5 );
    mScoreItem->setFont( font);
    mScene->addItem( mScoreItem );
    mScoreItem->setPos(0,5);

    mMickyMouse = new MickyMouse;
    mScene->addItem( mMickyMouse );    
    mMickyMouse->catchEgg(UpperRight);

    createBase();

    createHens();

    createButtons();

    createEgg();

    mGameTimer = new QTimer(this);

    connect(mGameTimer,SIGNAL(timeout()),this,SLOT(gameLoop()));
    connect(view,SIGNAL(keyReleased(QKeyEvent*)),this,SLOT(keyReleased(QKeyEvent*)));

    showMenu();
}

void GameWidget::createBase()
{
    class CreateBaseItem
    {
    public:
        QGraphicsRectItem* operator()( QPoint pos,QGraphicsScene* scene)
        {
          QGraphicsRectItem* rect = new QGraphicsRectItem(QRect(0,0,200,5));
          rect->setPen(Qt::NoPen);
          rect->setPos( pos);
          scene->addItem(rect);
          return rect;
        }
    } createBaseItem;

    QGraphicsRectItem* upperLeftBase = createBaseItem(QPoint(0,150),mScene);
    mBases << upperLeftBase;

    QGraphicsRectItem* lowerLeftBase = createBaseItem(QPoint(0,280),mScene);
    mBases << lowerLeftBase;

    QGraphicsRectItem* upperRightBase = createBaseItem(QPoint(600,150),mScene);
    mBases << upperRightBase;

    QGraphicsRectItem* lowerRightBase = createBaseItem(QPoint(600,280),mScene);
    mBases << lowerRightBase;
}

void GameWidget::createButtons()
{
    mPauseBtn = new TextButton("Pause");
    mPauseBtn->setPos(715,5);
    mScene->addItem( mPauseBtn );
    connect(mPauseBtn,SIGNAL(clicked()),this,SLOT(pauseGame()));
}

void GameWidget::createHens()
{
    for( int pos = UpperLeft ; pos <= LowerRight ;  pos++){
        Hen* hen = new Hen((Position)pos);
        mScene->addItem( hen );
        connect(hen,SIGNAL(touched(Position)),this,SLOT(henTouched(Position)));
        mHens << hen;
    }
}

void GameWidget::newGame()
{
    deleteOldScreen();

    mMaxEggCount = 3;
    mPoints = 0;
    mMinSpeed = 3;
    mMaxSpeed = 5;
    mTrialItem->setRemainigTrial(4);
    mScoreItem->setPlainText(QString("Score:%1").arg( mPoints));

    mLastTimeEggDelivery = QTime();

    foreach( Egg* egg, mEggs) {
        delete egg;
    }
    mEggs.clear();

    foreach( HatchingEgg* egg, mHatchingEggs) {
        delete egg;
    }
    mHatchingEggs.clear();

    createEgg();
    mGameTimer->start( 1000/30 );
}

void GameWidget::pauseGame()
{
    deleteOldScreen();

    mGameTimer->stop();
    PausedMenu* pauseMenu =  new PausedMenu( mScene );
    connect(pauseMenu,SIGNAL(resumeGame()),this,SLOT(resumeGame()));
    connect(pauseMenu,SIGNAL(showMainMenu()),this,SLOT(showMenu()));
    mCurrentScreen = pauseMenu;
}

void GameWidget::showMenu()
{
    deleteOldScreen();

    MainMenuScreen* mainMenu = new MainMenuScreen(mScene);
    connect(mainMenu,SIGNAL(startNewGame()),this,SLOT(newGame()));
    connect(mainMenu,SIGNAL(exitGame()),this,SLOT(exitGame()));
    mCurrentScreen = mainMenu;
}

void GameWidget::exitGame()
{
    deleteOldScreen();
    QCoreApplication::exit(0);
}

void GameWidget::resumeGame()
{
    deleteOldScreen();
    mGameTimer->start( 1000/30 );
}

void GameWidget::gameLoop()
{
    mMickyMouse->update();

    foreach( HatchingEgg* hatchingEgg, mHatchingEggs) {
        hatchingEgg->nextFrame();
    }

    foreach( Egg* egg, mEggs ) {
        bool cleanUpEgg = false;
        egg->move();
        if( mMickyMouse->isFallingInBasket( egg )) {
            //qDebug() << "Egg caught ...";
            cleanUpEgg = true;
            increasePoints();
            //increate points
        } else if( egg->isBroken()) {
            //qDebug() << "Egg broken ...";
            cleanUpEgg = true;            
            mTrialItem->decreaseTrial();

            HatchingEgg* hatchingEgg = new HatchingEgg(egg,mScene);
            mHatchingEggs << hatchingEgg;
            connect(hatchingEgg,SIGNAL(hatched(HatchingEgg*)),this,SLOT(eggHatched(HatchingEgg*)));
        }

        if( cleanUpEgg ) {
            mEggs.removeOne( egg);
            delete egg;

            if( mTrialItem->isTrialAvailable() )
                createEgg();
            else{
                gameOver();
            }
        }
    }
}

void GameWidget::eggHatched(HatchingEgg* hatchedEgg)
{
    mHatchingEggs.removeOne( hatchedEgg );
    hatchedEgg->deleteLater();
}

void GameWidget::gameOver()
{
    mGameTimer->stop();
    deleteOldScreen();
    ScorecardScreen* scoreScreen =  new ScorecardScreen( mScene,mPoints );
    connect(scoreScreen,SIGNAL(showMainMenu()),this,SLOT(showMenu()));
    connect(scoreScreen,SIGNAL(startNewGame()),this,SLOT(newGame()));
    mCurrentScreen = scoreScreen;
}

void GameWidget::deleteOldScreen()
{
    if(mCurrentScreen) {
        delete mCurrentScreen;
    }
    mCurrentScreen = 0;
}

void GameWidget::eggAboutToFall()
{
    createEgg();
}

void GameWidget::createEgg()
{
    if( mEggs.count() > mMaxEggCount || mLastTimeEggDelivery.elapsed() < 450 ) //this can be changed as per dufficulty paramater
        return;

    //qDebug() << "elapsed:"<<mLastTimeEggDelivery.elapsed();
    int random = qrand() % 4 ;
    int speed = qrand() % mMaxSpeed + mMinSpeed ; //this can be changed as per diffuculty
    Egg* egg = new Egg( mBases.at(random), speed );
    connect(egg,SIGNAL(aboutToFall()),this,SLOT(eggAboutToFall()));
    mScene->addItem( egg );
    mEggs << egg;
    mLastTimeEggDelivery.start();
}

void GameWidget::keyReleased(QKeyEvent *event)
{    
    if(mCurrentScreen)
        return;

    switch( event->key() ) {
    case Qt::Key_E:        
        henTouched( UpperRight);
        break;
    case Qt::Key_D:        
        henTouched( LowerRight);
        break;
    case Qt::Key_W:
        henTouched( UpperLeft);
        break;
    case Qt::Key_S:
        henTouched( LowerLeft);
        break;
    }
}

void GameWidget::henTouched( Position position)
{
     mMickyMouse->catchEgg(position);
     mScene->update();
}

void GameWidget::increasePoints()
{
    mPoints++;    
    QString score = QString("Score:%1").arg( mPoints);
    mScoreItem->setPlainText( score);

    increaseDifficulty();
}

void GameWidget::increaseDifficulty()
{
    //increase egg count
    if( mPoints % 100 == 0 ) {
        qDebug() << "increasing egg count";
        mMaxEggCount ++;
    }

    //increase egg speed
    if( mPoints % 65 == 0 ) {
        qDebug() << "increasing egg speed";
        //mMinSpeed ++;
        mMaxSpeed ++;
    }
}
