/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <string.h>
#include <libebook/e-book-util.h>
#include "debug.h"
#include "merger.h"


/* MatchToken */

typedef struct
{
    gchar *string;
    OssoABookContact *contact;
    gchar *description;
    gboolean partial;
} MatchToken;

static MatchToken *
match_token_new (const gchar      *string,
                 OssoABookContact *contact,
                 const gchar      *description,
                 gboolean          partial)
{
    MatchToken *token;

    g_return_val_if_fail (string, NULL);
    g_return_val_if_fail (contact, NULL);
    g_return_val_if_fail (description, NULL);

    token = g_new0 (MatchToken, 1);
    token->string = g_strdup (string);
    token->contact = g_object_ref (contact);
    token->description = g_strdup (description);
    token->partial = partial;

    return token;
}

static void
match_token_free (MatchToken *token)
{
    if (!token)
        return;

    g_free (token->string);
    g_object_unref (token->contact);
    g_free (token->description);
}


/* MatchTokens table */

static void
free_value_queue (GQueue *queue)
{
    if (!queue)
        return;

    g_queue_foreach (queue, (GFunc) match_token_free, NULL);
    g_queue_free (queue);
}

static GHashTable *
match_tokens_table_new (void)
{
    return g_hash_table_new_full (g_str_hash, g_str_equal, g_free,
            (GDestroyNotify) free_value_queue);
}

static void
match_tokens_table_add (GHashTable       *table,
                        const gchar      *string,
                        OssoABookContact *contact,
                        const gchar      *description,
                        gboolean          partial)
{
    gchar *simplified_string;
    MatchToken *token;
    GQueue *queue;

    g_return_if_fail (table);
    g_return_if_fail (string);
    g_return_if_fail (contact);
    g_return_if_fail (OSSO_ABOOK_IS_CONTACT (contact));
    g_return_if_fail (description);

    simplified_string = string_simplify (string);

    queue = g_hash_table_lookup (table, simplified_string);
    if (!queue) {
        queue = g_queue_new ();
        /* Leave ownership of the simplified string */
        g_hash_table_insert (table, simplified_string, queue);
    } else {
        g_free (simplified_string);
    }

    token = match_token_new (string, contact, description, partial);
    g_queue_push_head (queue, token);
}


/* Token generation */

typedef struct
{
    GHashTable *full_ids; /* Emails and IM names */
    GHashTable *partial_ids; /* Local part of emails and IM names */
    GHashTable *phones; /* Phone numbers */
    GHashTable *full_names; /* First + last names and last + first names */
    GHashTable *nicknames; /* Unparsed nicknames */

    GHashTable *im_field_quarks; /* quark of vcard fields -> TRUE */
} SuggestionsData;

static gboolean
inster_generic_name (SuggestionsData    *data,
                     OssoABookContact   *contact,
                     const EContactName *name,
                     const gchar        *description)
{
    gchar *tmp;

    if (IS_EMPTY (name->given) || IS_EMPTY (name->family))
        return FALSE;

    tmp = g_strdup_printf ("%s %s", name->given, name->family);
    DEBUG ("      Add token for full name %s", tmp);
    match_tokens_table_add (data->full_names, tmp, contact,
            description, FALSE);
    g_free (tmp);

    tmp = g_strdup_printf ("%s %s", name->family, name->given);
    DEBUG ("      Add token for reversed full name %s", tmp);
    match_tokens_table_add (data->full_names, tmp, contact,
            description, FALSE);
    g_free (tmp);

    if (!IS_EMPTY (name->additional)) {
        EContactName *name_no_middle;

        /* Bah, consts... */
        name_no_middle = e_contact_name_copy ((EContactName*) name);
        name_no_middle->additional[0] = '\0';
        g_free (name_no_middle->given);
        name_no_middle->given = g_strdup_printf ("%s %s", name->given,
                name->additional);

        DEBUG ("      Retrying with the middle name %s too",
            name->additional);
        inster_generic_name (data, contact, name_no_middle, description);
    }

    return TRUE;
}

static void
insert_full_name (SuggestionsData    *data,
                  OssoABookContact   *contact,
                  const EContactName *name)
{
    DEBUG ("    Add tokens for name:", name->given, name->family);

    inster_generic_name (data, contact, name, _("name"));
}

static void
insert_nickname (SuggestionsData    *data,
                 OssoABookContact   *contact,
                 const gchar        *nickname)
{
    EContactName *name;

    DEBUG ("    Add tokens for nickname %s:", nickname);

    name = e_contact_name_from_string (nickname);

    if (!inster_generic_name (data, contact, name, _("nickname"))) {
        /* If the name doesn't parse as a full name then it still makes sense
         * to try to still match it with other nicknames. The chance of false
         * positives seems lower than the one for considering first and last
         * names by themselves */
        DEBUG ("      Add token for unparsed nickname %s", nickname);
        match_tokens_table_add (data->nicknames, nickname, contact,
                _("nickname"), FALSE);
    }

    e_contact_name_free (name);
}

static void
insert_id (SuggestionsData  *data,
           OssoABookContact *contact,
           const gchar      *id,
           const gchar      *description)
{
    gchar *modified_value;
    gchar *tmp;
    EContactName *ename;

    DEBUG ("    Add token for ID %s", id);
    match_tokens_table_add (data->full_ids, id, contact, description, FALSE);

    /* If you have the same local part of your EMAIL or IM name across
     * multiple services there are good chances you are the same person */
    modified_value = g_strdup (id);
    tmp = strpbrk (modified_value, "@%");
    if (tmp)
        *tmp = '\0';
    /* FIXME: maybe we should skip admin, info, jobs and webmaster? */
    DEBUG ("    Add token for partial ID %s", modified_value);
    match_tokens_table_add (data->partial_ids, modified_value, contact,
            description, TRUE);

    /* Try to use the local part of the ID as a name */
    g_strdelimit (modified_value, "-_.?/+", ' ');
    DEBUG ("    Add tokens for parsed ID %s:", modified_value);
    ename = e_contact_name_from_string (modified_value);
    inster_generic_name (data, contact, ename, description);

    e_contact_name_free (ename);
    g_free (modified_value);
}

static gchar *
normalize_phone (const gchar *phone)
{
    gchar *normalized;
    gint offset;

    g_return_val_if_fail (phone != NULL, FALSE);

    /* Strip spaces, dashes, etc. */
    normalized = e_normalize_phone_number (phone);

    /* Strip DTMF chars */
    offset = strcspn (normalized, OSSO_ABOOK_DTMF_CHARS);
    normalized[offset] = '\0';

    /* Only consider the last 7 digits of the number */
    if (offset > 7) {
        /* strcpy and similar functions don't handle overlapping
         * areas */
        gint offset_src = offset - 7;
        gint offset_dest = 0;
        while (normalized[offset_dest] != '\0')
            normalized[offset_dest++] = normalized[offset_src++];
        normalized[offset_dest] = '\0';
    }

    return normalized;
}

static void
insert_phone (SuggestionsData  *data,
              OssoABookContact *contact,
              const gchar      *phone)
{
    gchar *normalized;

    normalized = normalize_phone (phone);
    DEBUG ("    Add token for phone %s (%s)", phone, normalized);
    match_tokens_table_add (data->phones, normalized, contact,
            _("phone"), FALSE);
    g_free (normalized);
}

static void
insert_attr (SuggestionsData  *data,
             OssoABookContact *original_contact,
             OssoABookContact *flat_contact,
             EVCardAttribute  *attr)
{
    const gchar *attr_name;
    GList *values;
    GQuark name_quark;

    attr_name = e_vcard_attribute_get_name (attr);
    values = e_vcard_attribute_get_values (attr);
    if (!values || IS_EMPTY (values->data))
        return;

    name_quark = g_quark_from_string (attr_name);

    if (name_quark == OSSO_ABOOK_QUARK_VCA_EMAIL) {
        insert_id (data, original_contact, values->data, _("email"));
    } else if (name_quark == OSSO_ABOOK_QUARK_VCA_TEL) {
        insert_phone (data, original_contact, values->data);
    } else if (name_quark == OSSO_ABOOK_QUARK_VCA_N) {
        const EContactName *name;
        /* There's no way to convert the GList in a EContactName... */
        name = e_contact_get_const (E_CONTACT (flat_contact), E_CONTACT_NAME);
        insert_full_name (data, original_contact, name);
    } else if (name_quark == OSSO_ABOOK_QUARK_VCA_FN) {
        if (!e_contact_get_const (E_CONTACT (flat_contact), E_CONTACT_NAME)) {
            /* There is a FN but not an N, so let's parse FN */
            EContactName *name;
            name = e_contact_name_from_string (values->data);
            insert_full_name (data, original_contact, name);
            e_contact_name_free (name);
        }
    } else if (name_quark == OSSO_ABOOK_QUARK_VCA_NICKNAME) {
        insert_nickname (data, original_contact, values->data);
    } else if (g_hash_table_lookup (data->im_field_quarks,
                    GUINT_TO_POINTER (name_quark))) {
        insert_id (data, original_contact, values->data, _("IM name"));
    }
}


/* ContactPair */

typedef struct {
    OssoABookContact *contact1;
    OssoABookContact *contact2;
} ContactPair;

static ContactPair *
contact_pair_new (OssoABookContact *contact1,
                  OssoABookContact *contact2)
{
    ContactPair *pair;

    g_return_val_if_fail (contact1, NULL);
    g_return_val_if_fail (contact2, NULL);
    g_return_val_if_fail (contact1 != contact2, NULL);

    pair = g_new0 (ContactPair, 1);

    /* Keep pair->contact1 < pair->contact2 to make contact_pair_equal work */
    pair->contact1 = MIN (contact1, contact2);
    pair->contact2 = MAX (contact1, contact2);

    g_object_ref (pair->contact1);
    g_object_ref (pair->contact2);

    return pair;
}

static void
contact_pair_free (ContactPair *pair)
{
    if (!pair)
        return;

    g_object_unref (pair->contact1);
    g_object_unref (pair->contact2);
    g_free (pair);
}

static guint
contact_pair_hash (gconstpointer p)
{
    const ContactPair *pair = p;

    g_return_val_if_fail (pair, 0);

    return GPOINTER_TO_UINT (pair->contact1) +
           GPOINTER_TO_UINT (pair->contact2);
}

static gboolean
contact_pair_equal (gconstpointer p1,
                    gconstpointer p2)
{
    const ContactPair *pair1 = p1;
    const ContactPair *pair2 = p2;

    if (pair1 == NULL && pair2 == NULL)
        return TRUE;
    else if (pair1 == NULL || pair2 == NULL)
        return FALSE;
    else
        /* This works as we always have contact1 < contact2 */
        return pair1->contact1 == pair2->contact1 &&
               pair1->contact2 == pair2->contact2;
}


/* MatchInfo */

typedef struct {
    gint score;
    gboolean partial;
} MatchData;

typedef struct {
    gint total_score;
    GHashTable *descriptions; /* gchar* -> MatchData* */
}  MatchInfo;

static MatchInfo *
match_info_new (void)
{
    MatchInfo *info;

    info = g_new0 (MatchInfo, 1);
    info->descriptions = g_hash_table_new_full (g_str_hash, g_str_equal,
            g_free, g_free);

    return info;
}

static void
match_info_free (MatchInfo *info)
{
    if (!info)
        return;

    g_hash_table_unref (info->descriptions);
    g_free (info);
}

static void
match_info_add_description (MatchInfo   *info,
                            const gchar *description,
                            gint         score,
                            gboolean     partial)
{
    MatchData *data;

    data = g_hash_table_lookup (info->descriptions, description);
    if (!data) {
        data = g_new0 (MatchData, 1);
        data->partial = TRUE;
        g_hash_table_insert (info->descriptions, g_strdup (description), data);
    }

    data->score += score;
    /* If "description" matches at least something in a full way then it's
     * not reported as a partial match */
    data->partial &= partial;
}

static void
match_info_add (MatchInfo   *info,
                const gchar *description1,
                const gchar *description2,
                gint         score,
                gboolean     partial)
{
    g_return_if_fail (info);

    info->total_score += score;
    match_info_add_description (info, description1, score, partial);
    match_info_add_description (info, description2, score, partial);
}


/* Matches hash table */

static GHashTable *
match_table_new (void)
{
    return g_hash_table_new_full (contact_pair_hash, contact_pair_equal,
            (GDestroyNotify) contact_pair_free,
            (GDestroyNotify) match_info_free);
}

static void
match_table_add (GHashTable       *table,
                 OssoABookContact *contact1,
                 OssoABookContact *contact2,
                 const gchar      *description1,
                 const gchar      *description2,
                 gint              score,
                 gboolean          partial)
{
    ContactPair *pair;
    MatchInfo *info;

    if (contact1 == contact2)
        return;

    pair = contact_pair_new (contact1, contact2);

    info = g_hash_table_lookup (table, pair);
    if (!info) {
        info = match_info_new ();
        g_hash_table_insert (table, pair, info); /* Leave ownership */
    } else {
        contact_pair_free (pair);
    }

    match_info_add (info, description1, description2, score, partial);
}


/* Match */

struct _Match {
    guint ref_count;
    ContactPair *pair;
    MatchInfo *info;
    gchar *description;
};

static Match *
match_new (ContactPair *pair,
           MatchInfo   *info)
{
    Match *match;

    g_return_val_if_fail (pair, NULL);
    g_return_val_if_fail (info, NULL);

    match = g_new0 (Match, 1);
    match->ref_count = 1;
    match->pair = pair;
    match->info = info;

    return match;
}

Match *
match_ref (Match *match)
{
    g_return_val_if_fail (match, NULL);

    match->ref_count++;

    return match;
}

void
match_unref (Match *match)
{
    g_return_if_fail (match);
    g_return_if_fail (match->ref_count > 0);

    match->ref_count--;

    if (match->ref_count == 0) {
        contact_pair_free (match->pair);
        match_info_free (match->info);
        g_free (match->description);
        g_free (match);
    }
}

GType
match_get_type (void)
{
    static GType our_type = 0;

    if (G_UNLIKELY (our_type == 0))
        our_type = g_boxed_type_register_static ("Match",
                (GBoxedCopyFunc) match_ref, (GBoxedFreeFunc) match_unref);

    return our_type;
}

static gint
match_compare (gconstpointer a,
               gconstpointer b)
{
    const Match *ma = a;
    const Match *mb = b;

    g_return_val_if_fail (ma, -1);
    g_return_val_if_fail (mb, 1);

    return mb->info->total_score - ma->info->total_score;
}

typedef struct {
    gchar *description;
    MatchData *data;
} DescriptionMatchDataPair;

static gint
description_match_data_compare (gconstpointer a,
                                gconstpointer b)
{
    const DescriptionMatchDataPair *pa = a;
    const DescriptionMatchDataPair *pb = b;

    g_return_val_if_fail (pa, 1);
    g_return_val_if_fail (pb, -1);

    return pb->data->score - pa->data->score;
}

void
match_get_contacts (Match             *match,
                    OssoABookContact **c1,
                    OssoABookContact **c2)
{
    g_return_if_fail (match);

    if (c1)
        *c1 = match->pair->contact1;

    if (c2)
        *c2 = match->pair->contact2;
}

const gchar *
match_get_description (Match *match)
{
    g_return_val_if_fail (match, NULL);

    if (!match->description) {
        GHashTableIter iter;
        gpointer key, value;
        GList *descriptions_list = NULL;
        DescriptionMatchDataPair *pair;
        GString *s;
        GList *l;

        g_hash_table_iter_init (&iter, match->info->descriptions);
        while (g_hash_table_iter_next (&iter, &key, &value)) {
            pair = g_new0 (DescriptionMatchDataPair, 1);
            pair->description = key;
            pair->data = value;
            descriptions_list = g_list_prepend (descriptions_list, pair);
        }

        descriptions_list = g_list_sort (descriptions_list,
                description_match_data_compare);

        s = g_string_new (_("Matches: "));

        for (l = descriptions_list; l; l = l->next) {
            pair = l->data;

            if (!debug_is_enabled ())
                g_string_append_printf (s, "%s%s", pair->description,
                        pair->data->partial ? " (partial)" : "");
            else
                g_string_append_printf (s, "%s (%s%d)", pair->description,
                        pair->data->partial ? "partial, " : "",
                        pair->data->score);

            if (l->next)
                g_string_append (s, ", ");

            g_free (pair);
        }

        match->description = g_string_free (s, FALSE);

        g_list_free (descriptions_list);
    }

    return match->description;
}

gint
match_get_score (Match *match)
{
    return match->info->total_score;
}


/* Match list */

static GList *
match_list_from_table_steal (GHashTable *table)
{
    GHashTableIter iter;
    gpointer key, value;
    Match *match;
    GList *matches = NULL;

    g_return_val_if_fail (table, NULL);

    g_hash_table_iter_init (&iter, table);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        match = match_new (key, value);
        matches = g_list_prepend (matches, match);
    }

    matches = g_list_sort (matches, match_compare);

    g_hash_table_steal_all (table);

    return matches;
}

void
match_list_free (GList *matches)
{
    g_list_foreach (matches, (GFunc) match_unref, NULL);
    g_list_free (matches);
}

GList *
match_list_copy (GList *matches)
{
    GList *copy;

    copy = g_list_copy (matches);
    g_list_foreach (copy, (GFunc) match_ref, NULL);

    return copy;
}

GType
match_list_get_type (void)
{
    static GType our_type = 0;

    if (G_UNLIKELY (our_type == 0))
        our_type = g_boxed_type_register_static ("MatchList",
                (GBoxedCopyFunc) match_list_copy,
                (GBoxedFreeFunc) match_list_free);

    return our_type;
}


/* Suggestions */

/* Matched both the full ID and partial one and also the ones coming from
 * parsing the ID as a name */
#define SCORE_IDS          30
#define SCORE_PARTIAL_IDS  10
#define SCORE_PHONES       15
/* Matches twice because of the reverse combination */
#define SCORE_FULL_NAME    10
#define SCORE_NICKNAMES    10

static void
generate_same_kind_suggestions (GHashTable *matches,
                                GHashTable *tokens_table,
                                gint        score)
{
    GHashTableIter iter;
    gpointer value;
    GQueue *queue;
    GList *l1, *l2;
    MatchToken *t1, *t2;

    g_return_if_fail (matches);
    g_return_if_fail (tokens_table);

    g_hash_table_iter_init (&iter, tokens_table);
    while (g_hash_table_iter_next (&iter, NULL, &value)) {
        queue = value;

        for (l1 = queue->head; l1; l1 = l1->next) {
            t1 = l1->data;
            for (l2 = l1->next; l2; l2 = l2->next) {
                t2 = l2->data;
                DEBUG ("    %s (uid: %s) matches %s (uid: %s)\n"
                    "      as %s (%s) matches %s (%s)",
                    osso_abook_contact_get_display_name (t1->contact),
                    osso_abook_contact_get_uid (t1->contact),
                    osso_abook_contact_get_display_name (t2->contact),
                    osso_abook_contact_get_uid (t2->contact),
                    t1->string, t1->description,
                    t2->string, t2->description);
                match_table_add (matches,
                    t1->contact, t2->contact,
                    t1->description, t2->description,
                    score, t1->partial || t2->partial);
            }
        }
    }
}


/* Merge suggestions generation */

GList * /* List of Match* */
generate_merge_suggestions (GList *contacts,
                            GList *im_fields)
{
    SuggestionsData *data;
    GHashTable *match_table;
    GList *matches;
    GList *l;

    data = g_new0 (SuggestionsData, 1);
    data->full_ids = match_tokens_table_new ();
    data->partial_ids = match_tokens_table_new ();
    data->phones = match_tokens_table_new ();
    data->full_names = match_tokens_table_new ();
    data->nicknames = match_tokens_table_new ();
    data->im_field_quarks = g_hash_table_new (g_direct_hash, g_direct_equal);
    for (l = im_fields; l; l = l->next) {
        /* Skip phone numbers are they are handled separately */
        if (strcmp (l->data, "TEL") != 0)
            g_hash_table_insert (data->im_field_quarks,
                    GUINT_TO_POINTER (g_quark_from_string (l->data)),
                    GINT_TO_POINTER (TRUE));
    }

    DEBUG ("Analysing contacts:");

    for (; contacts; contacts = contacts->next) {
        OssoABookContact *original_contact = contacts->data;
        OssoABookContact *flat_contact;
        GList *attrs;

        DEBUG ("  Analysing %s (uid: %s):",
            osso_abook_contact_get_display_name (original_contact),
            osso_abook_contact_get_uid (original_contact));

        /* We care about the flattened contact as it's also easier to analyse */
        flat_contact = osso_abook_contact_merge_roster_info (original_contact);

        attrs = e_vcard_get_attributes (E_VCARD (flat_contact));
        for (; attrs; attrs = attrs->next) {
            insert_attr (data, original_contact, flat_contact, attrs->data);
        }

        g_object_unref (flat_contact);
    }

    DEBUG ("Generating all the matches:");
    match_table = match_table_new ();
    DEBUG ("  Full IDs:");
    generate_same_kind_suggestions (match_table, data->full_ids, SCORE_IDS);
    DEBUG ("  Partial IDs:");
    generate_same_kind_suggestions (match_table, data->partial_ids,
            SCORE_PARTIAL_IDS);
    DEBUG ("  Phone numbers:");
    generate_same_kind_suggestions (match_table, data->phones, SCORE_PHONES);
    DEBUG ("  Full names:");
    generate_same_kind_suggestions (match_table, data->full_names,
            SCORE_FULL_NAME);
    DEBUG ("  Nicknames:");
    generate_same_kind_suggestions (match_table, data->nicknames,
            SCORE_NICKNAMES);

    DEBUG ("Matches are:");
    matches = match_list_from_table_steal (match_table);

    g_hash_table_unref (match_table);
    g_hash_table_unref (data->im_field_quarks);
    g_hash_table_unref (data->nicknames);
    g_hash_table_unref (data->full_names);
    g_hash_table_unref (data->phones);
    g_hash_table_unref (data->partial_ids);
    g_hash_table_unref (data->full_ids);
    g_free (data);

    return matches;
}


/* Generate list of contacts to merge */

static GHashTable *
contacts_set_new ()
{
    return g_hash_table_new (g_direct_hash, g_direct_equal);
}

static void
contacts_set_add (GHashTable       *set,
                  OssoABookContact *contact)
{
    g_hash_table_insert (set, contact, GUINT_TO_POINTER (TRUE));
}

static void
contacts_set_extend (GHashTable *dest,
                     GHashTable *src)
{
    GHashTableIter iter;
    gpointer key;

    g_hash_table_iter_init (&iter, src);
    while (g_hash_table_iter_next (&iter, &key, NULL))
        contacts_set_add (dest, key);
}

static void
merge_table_set (GHashTable       *table,
                 OssoABookContact *contact,
                 GHashTable       *set)
{
    g_hash_table_insert (table, contact, g_hash_table_ref (set));
}

static void
merge_table_replace (GHashTable *table,
                     GHashTable *old,
                     GHashTable *new)
{
    GList *keys;

    keys = g_hash_table_get_keys (table);
    while (keys) {
        OssoABookContact *c = keys->data;

        if (g_hash_table_lookup (table, c) == old)
            merge_table_set (table, c, new);

        keys = g_list_delete_link (keys, keys);
    }
}

static GList * /* of GList of OssoABookContact* */
generate_merge_lists (GList *matches)
{
    GHashTable *table;
    GList *l;
    OssoABookContact *c1;
    OssoABookContact *c2;
    GHashTable *set_c1;
    GHashTable *set_c2;
    GHashTableIter iter;
    gpointer value;
    GList *all_merges = NULL;

    /* We have a list of pairs of contacts, with the same contact possibly
     * in multiple pairs, and we have to transform this is a list of lists
     * of contacts to merge.
     * If, for example, we have (A, B), (C, D), (E, F) and (A, F) we need to
     * get (A, B, E, F) and (C, D). */

    /* OssoABookContact -> Set<OssoABookContact>
     * Each contact points to a set of contacts. The set contains the
     * contact itself plus the other contacts that should be merged with
     * it */
    table = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL,
                (GDestroyNotify) g_hash_table_unref);

    for (l = matches; l; l = l->next) {
        match_get_contacts (l->data, &c1, &c2);

        set_c1 = g_hash_table_lookup (table, c1);
        set_c2 = g_hash_table_lookup (table, c2);

        if (set_c1 == NULL && set_c2 == NULL) {
            /* Neither of the contacts is in the table yet, create a
             * set for both of them */
            set_c1 = contacts_set_new ();
            contacts_set_add (set_c1, c1);
            contacts_set_add (set_c1, c2);
            merge_table_set (table, c1, set_c1);
            merge_table_set (table, c2, set_c1);
            g_hash_table_unref (set_c1);
        } else if (set_c1 != NULL && set_c2 != NULL) {
            /* Both of the contacts are already in the table, we
             * copy all the contacts from the second set to the
             * first one. Then we replace the second set in the
             * table with the first one (that now contains the
             * union of the two sets) */
            contacts_set_extend (set_c1, set_c2);
            merge_table_replace (table, set_c2, set_c1);
        } else if (set_c1 != NULL) {
            /* We already have a set, so we just add the other
             * contact to it */
            contacts_set_add (set_c1, c2);
            merge_table_set (table, c2, set_c1);
        } else {
            /* Ditto */
            contacts_set_add (set_c2, c1);
            merge_table_set (table, c1, set_c2);
        }
    }

    /* Now we get the list of merges from the table */
    g_hash_table_iter_init (&iter, table);
    while (g_hash_table_iter_next (&iter, NULL, &value)) {
        GHashTable *set = value;
        GList *contacts;

        contacts = g_hash_table_get_keys (set);

        /* This set was already added */
        if (!contacts)
            continue;

        g_list_foreach (contacts, (GFunc) g_object_ref, NULL);
        all_merges = g_list_prepend (all_merges, contacts);

        /* There are multiple contacts referring to the same set.
         * We empty the set so that the next time we find it we can
         * skip it. */
        g_hash_table_remove_all (set);
    }

    g_hash_table_unref (table);

    return all_merges;
}


/* Merge contacts */

/* We cannot use more for now or the dialogs for merge resolution
 * could come up while other ones are visible :( */
#define MAX_PENDING 1

typedef struct
{
    gint pending_count;
    gint merged_count;
    gint failed_count;
    GList *contacts_to_merge;
    GList *failed;
    GtkWindow *parent;
    MergeFinishedCb cb;
    gpointer user_data;
} MergeData;

typedef struct
{
    MergeData *merge_data;
    GList *contacts;
} MergeCbData;

static void merge_continue (MergeData *data);

static void
free_contacts_list (GList *contacts)
{
    g_list_foreach (contacts, (GFunc) g_object_unref, NULL);
    g_list_free (contacts);
}

static gboolean
merge_continue_from_idle_cb (gpointer user_data)
{
    MergeData *data = user_data;

    merge_continue (data);

    return FALSE;
}

static void
contacts_merged_cb (const gchar *uid,
                    gpointer     user_data)
{
    MergeCbData *merge_cb_data = user_data;
    MergeData *data = merge_cb_data->merge_data;

    if (uid) {
        DEBUG ("Merge of %d contacts succeeded, new UID: %s",
            g_list_length (merge_cb_data->contacts), uid);
        data->merged_count += g_list_length (merge_cb_data->contacts);
        free_contacts_list (merge_cb_data->contacts);
    } else {
        DEBUG ("Merge of %d contacts failed",
            g_list_length (merge_cb_data->contacts));
        data->failed_count += g_list_length (merge_cb_data->contacts);
        data->failed = g_list_prepend (data->failed, merge_cb_data->contacts);
    }

    g_slice_free (MergeCbData, merge_cb_data);

    data->pending_count--;

    /* The merge function can show a dialog with gtk_dialog_run()
     * and this can make us lose some D-Bus method returns, see
     * https://bugs.freedesktop.org/show_bug.cgi?id=14581 */
    gdk_threads_add_idle_full (G_PRIORITY_HIGH_IDLE,
            merge_continue_from_idle_cb, data, NULL);
}

static void
merge_continue (MergeData *data)
{
    if (data->pending_count >= MAX_PENDING)
        return;

    if (data->pending_count == 0 && data->contacts_to_merge == NULL) {
        /* We are done */
        DEBUG ("Done merging %d contacts, %d failed",
            data->merged_count, data->failed_count);

        if (data->cb)
            data->cb (data->merged_count, data->failed_count,
                    data->failed, data->user_data);

        g_list_foreach (data->failed, (GFunc) free_contacts_list, NULL);
        g_list_free (data->failed);
        g_slice_free (MergeData, data);
        return;
    }

    while (data->pending_count < MAX_PENDING && data->contacts_to_merge) {
        GList *contacts = data->contacts_to_merge->data;
        MergeCbData *merge_cb_data;

        DEBUG ("Scheduling merge of %d contacts", g_list_length (contacts));

        merge_cb_data = g_slice_new0 (MergeCbData);
        merge_cb_data->merge_data = data;
        merge_cb_data->contacts = contacts;

        data->pending_count++;
        data->contacts_to_merge = g_list_delete_link (
                data->contacts_to_merge, data->contacts_to_merge);

        osso_abook_merge_contacts_and_save (contacts, data->parent,
                contacts_merged_cb, merge_cb_data);
    }
}

void
merge_contacts (GList           *matches,
                GtkWindow       *parent,
                MergeFinishedCb  cb,
                gpointer         user_data)
{
    GList *contacts_to_merge;
    GList *l;
    GList *contacts;
    GList *m;
    MergeData *data;

    g_return_if_fail (matches);

    contacts_to_merge = generate_merge_lists (matches);

    if (debug_is_enabled ()) {
        for (l = contacts_to_merge; l; l = l->next) {
            contacts = l->data;
            DEBUG ("Going to merge:");
            for (m = contacts; m; m = m->next) {
                DEBUG ("  %s (%s)",
                        osso_abook_contact_get_display_name (m->data),
                        osso_abook_contact_get_uid (m->data));
            }
        }
    }

    data = g_slice_new0 (MergeData);
    data->contacts_to_merge = contacts_to_merge;
    data->parent = parent;
    data->cb = cb;
    data->user_data = user_data;

    merge_continue (data);
}
