/*
 * This file is part of contactinfos
 *
 * Copyright (C) 2007 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <contactinfos.h>
#include <backend/backend.h>
#include <ui/interface.h>
#include <ui/callbacks.h>
#include <dbus/dbus-error.h>

#include <gtk/gtk.h>
#include <libintl.h>

#include <config.h>

/* strlen needed from string.h */
#include <string.h>

#if HILDON == 1
#include <hildon/hildon.h>
#else
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-font-selection-dialog.h>
#endif

/* Privates: */

GtkWidget* create_editarea( AppUIData *main, gchar* searchText );
gboolean validate(GtkWindow *parent, GSList* pathList, GtkTreeStore *store, gboolean getAll, guint upperLimit, GPtrArray **pPtrArray, guint *count, GError **error);

AppUIData* newAppUIData() {
	/* Zero memory with g_new0 */
	return g_new0( AppUIData, 1 );
}


/* Creates and initialises a dialog */
gboolean interface_dialog_new(
					AppUIData* result, 
					GtkWindow *parent, 
					gboolean multiSel, 
					gboolean showGui, 
					gboolean confirm, 
					gchar* searchText, 
					GPtrArray ** resultArray, 
					gpointer backendData, 
					guint requestedInfos, 
					guint upperLimit, 
					GError ** error ) {

	GtkWidget* table;
	gint resultCode;
	guint count;
	gchar * textSearch;
	GError *localError=NULL;

#ifdef DEBUG
	g_printf("interface_dialog_new(\n  AppUIData* result=%p, \n GtkWindow *parent=%p,\n gboolean multiSel=%d,\n gboolean showGui=%d,\n gchar* searchText=%s,\n GSList ** resultArray=%p,\n gpointer backendData=%p,\n guint requestedInfos=%d )\n",
			result, parent, multiSel, showGui, searchText, * resultArray, backendData, requestedInfos);
#endif	
	/* mode multiSel */
	result->multipleSelection = multiSel;
	result->pathList = NULL;
	result->backendData = backendData;
	result->requestedInfos = requestedInfos;
	result->error = error;
	
	textSearch = g_strdup(searchText);
	
	clearPathList(result);
	result->storeSearch = gtk_tree_store_new (N_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_UINT);

	if (!showGui || strlen(textSearch)>0) {
		count = search(parent, textSearch, GTK_TREE_STORE(result->storeSearch), backendData, requestedInfos, &localError);
		if (localError) {
			g_free(textSearch);
			g_propagate_error(error, localError);
			return TRUE;
		}
		if (count == 0) {
			g_free(textSearch);
			g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_NOT_FOUND,_("ContactInfos : No answser."));
			return TRUE;
		}
		if ((!showGui && multiSel) || (!multiSel && !confirm && count == 1)) {
			validate(parent, NULL, result->storeSearch, TRUE, upperLimit, resultArray, &count, &localError);
			if (localError) {
				g_propagate_error(error, localError);
			}
			g_free(textSearch);
			return TRUE;
		}
		if ((!showGui && !multiSel && count > 1)) {
			g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_FOUND_BUT_NO_DISPLAY,_("ContactInfos : More than 1 answer."));
			g_free(textSearch);
			return TRUE;
		}
	}
	
	result->dialog = gtk_dialog_new_with_buttons (_("ContactInfos"),
				        parent,
					GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, 
					NULL);
	
	result->buttonOK = gtk_dialog_add_button(GTK_DIALOG(result->dialog), GTK_STOCK_OK, GTK_RESPONSE_ACCEPT);
	gtk_widget_set_sensitive(result->buttonOK, FALSE);
	gtk_dialog_add_button(GTK_DIALOG(result->dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);
	
	/* Create all our necessary parts */
	table = create_editarea( result, textSearch);
	/* Put GTK table to app's view */
	gtk_container_add (GTK_CONTAINER(((GtkDialog *)result->dialog)->vbox), table);
	/* Focus on search Entry */
	gtk_widget_grab_focus(GTK_WIDGET(result->searchEntry ));

	gtk_window_set_default_size(GTK_WINDOW(result->dialog), 600, 300);
	gtk_widget_show_all (GTK_WIDGET(result->dialog));

	setExitTimer(result, TRUE);
	resultCode = gtk_dialog_run (GTK_DIALOG (result->dialog));
	removeExitTimer(result);
	switch (resultCode) {
		case GTK_RESPONSE_ACCEPT:
			result->pathList = g_slist_reverse(result->pathList);
			validate(parent, result->pathList, result->storeSearch, FALSE, upperLimit, resultArray, &count, &localError);
			if (localError) {
				g_propagate_error(error, localError);
			}
			break;
		default:
			break;
	}
	gtk_widget_destroy (result->dialog);
	result->dialog = NULL;
	gtk_tree_store_clear(result->storeSearch);
	clearPathList(result);
	g_free(textSearch);

	/* Return  */
	if (count == 0) {
		g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_NOT_FOUND,_("ContactInfos : No answser."));
		return TRUE;
	}
	return TRUE;
}


/* Create the edit area */
GtkWidget* create_editarea( AppUIData *main, gchar *searchText ) {
	GtkWidget *button;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;
	GtkWidget *scrolled_window;
	GtkWidget* table;

	/* la boite gnrale */
	table = gtk_table_new(4, 5, TRUE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	
	/* Add a field & 2 buttons */
	main->searchEntry = gtk_entry_new();
	if (searchText != NULL) {
		 gtk_entry_set_text(GTK_ENTRY(main->searchEntry), searchText);
	}
	gtk_table_attach_defaults( GTK_TABLE(table), main->searchEntry, 0, 4, 0, 1);

	/* Buttons */
	button = gtk_button_new_with_label (_("Search"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	g_signal_connect (G_OBJECT (button), "clicked",
	                  G_CALLBACK (search_cb), (gpointer) main);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 4, 5, 0, 1);

	/* la liste des entres */
	if (main->storeSearch == NULL) {
		main->storeSearch = gtk_tree_store_new (N_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
	}

#if MAEMO_VERSION_MAJOR == 5
	scrolled_window = hildon_pannable_area_new();
#else
	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#endif
	
	main->treeSearch = gtk_tree_view_new_with_model (GTK_TREE_MODEL (main->storeSearch));
#if HILDON == 0
	g_object_set(G_OBJECT(main->treeSearch), "allow-checkbox-mode", FALSE, NULL);
	g_object_set(G_OBJECT(main->treeSearch), "force-list-kludge", TRUE, NULL);
#endif
	
	gtk_container_add(GTK_CONTAINER(scrolled_window), main->treeSearch);
	gtk_table_attach_defaults( GTK_TABLE(table), scrolled_window, 0, 5, 1, 4);

	renderer = gtk_cell_renderer_toggle_new ();
	g_signal_connect (renderer, "toggled", G_CALLBACK (setRowChecked_cb), main);
	column = gtk_tree_view_column_new_with_attributes (_("Check"), renderer, "active", CHECK_COLUMN, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (main->treeSearch), column);
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Type"), renderer, "text", TYPE_COLUMN, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (main->treeSearch), column);
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Name"), renderer, "text", LABEL_COLUMN, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (main->treeSearch), column);

	/* Setup the selection handler */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (main->treeSearch));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

	gtk_tree_view_columns_autosize(GTK_TREE_VIEW(main->treeSearch));
	
	return table;

}

void fillElement(GtkTreeModel *model, GtkTreeIter *iter, ConstructRepData *pConstructRepData) {
	gchar* type;
	gchar* label;
	gchar* name;
	gchar* company;
	gchar* value;
	guint typeInfo;
	ContactInfosElement *pContactInfosElement;
	ContactInfosValue *pContactInfosValue;
	
	pConstructRepData->index++;
	if (pConstructRepData->limit > 0 && pConstructRepData->index > pConstructRepData->limit) {
		return;
	}

	gtk_tree_model_get(model, iter,
	    TYPE_COLUMN, &type,
	    LABEL_COLUMN, &label,
	    NAME_COLUMN, &name,
	    COMPANY_COLUMN, &company,
	    REQUESTED_INFO_COLUMN, &typeInfo,
	    -1);

	value = g_strdup_printf("%s-%s",name, company);
	pContactInfosElement = (ContactInfosElement *) g_hash_table_lookup(pConstructRepData->hashTable, value);
	if (!pContactInfosElement) {
		pContactInfosElement = g_new0(ContactInfosElement, 1);
		if (name) {
			pContactInfosElement->name = g_strdup(name);
		}
		if (company) {
			pContactInfosElement->company = g_strdup(company);
		}
		pContactInfosElement->valueArray = g_ptr_array_new();
		g_ptr_array_add(pConstructRepData->ptrArray, pContactInfosElement);
		g_hash_table_insert(pConstructRepData->hashTable, value, pContactInfosElement);
	}
	g_free(value);
	
	pContactInfosValue = g_new0(ContactInfosValue, 1);
	pContactInfosValue->typeInfo = typeInfo;
	pContactInfosValue->type = g_strdup(type);
	pContactInfosValue->value = g_strdup(label);
	g_ptr_array_add(pContactInfosElement->valueArray, pContactInfosValue);
	
}

gboolean fillList (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data) {
	ConstructRepData *pConstructRepData = (ConstructRepData *) data;
	
	if (!gtk_tree_model_iter_has_child(model, iter)) {
		fillElement(model, iter, pConstructRepData);
	}
	return FALSE;
}

gboolean validate(GtkWindow *parent, GSList* pathList, GtkTreeStore *store, gboolean getAll, guint upperLimit, GPtrArray **pPtrArray, guint *count, GError **error) {

	GtkTreeModel *model;
	GtkTreeIter iter;
	GSList *entry;
/*
	gboolean rowChecked;
*/
	ConstructRepData * pConstructRepData;
	
	pConstructRepData = g_new0(ConstructRepData, 1);
	pConstructRepData->ptrArray = g_ptr_array_new();
	pConstructRepData->hashTable = g_hash_table_new(g_str_hash, g_str_equal);
	pConstructRepData->limit = upperLimit;

	model = GTK_TREE_MODEL(store);
	if (getAll) {
		gtk_tree_model_foreach(model, (GtkTreeModelForeachFunc) fillList, (gpointer) pConstructRepData);
	} else {
		for (entry=pathList; entry; entry=g_slist_next(entry)) {
			if (gtk_tree_model_get_iter_from_string(model, &iter, entry->data) ) {
/*
				gtk_tree_model_get(model, &iter,
				    CHECK_COLUMN, &rowChecked,
				    -1);
				if (rowChecked) {
*/
				fillElement(model, &iter, pConstructRepData);
/*
				}
*/			
			}
		}
	}
	
	*count = pConstructRepData->index;
	if (pConstructRepData->limit > 0 && pConstructRepData->index >= pConstructRepData->limit) {
		*count = pConstructRepData->limit;
		g_set_error(error, CONTACTINFOS_DBUS_ERROR, CI_FOUND_BUT_UPPER_LIMIT_REACHED,_("ContactInfos : upper limit reached."));
	}
	
	(*pPtrArray) = pConstructRepData->ptrArray;

	/* Clean construct data */
#if HILDON==1
	g_hash_table_steal_all(pConstructRepData->hashTable);
	g_hash_table_unref(pConstructRepData->hashTable);
#else
	g_hash_table_destroy (pConstructRepData->hashTable);
#endif
	
	g_free(pConstructRepData);
	
	/* Return true */
	return TRUE;
}
	
