/*
 * This file is part of contactinfos
 *
 * Copyright (C) 2007 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "contactinfos.h"
#include <backend/backend.h>

#include "util/structures.h"

#include "ui/interface.h"

#include "dbus/dbus-service.h"
#include "dbus/dbus-error.h"
#include "dbus/server-contactinfos-dbus.h"
#include "util/timer.h"

#include <glib.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>

G_DEFINE_TYPE (ContactinfosService, contactinfos_service, G_TYPE_OBJECT)

ContactinfosService *the_service = NULL;


gboolean send_and_unref(DBusConnection *connection, DBusMessage *message) {
	if (!dbus_connection_send(connection, message, NULL)) {
#ifdef DEBUG
		g_printf("send_and_unref Failure ! (\nDBusConnection *connection=%p, \nDBusMessage *message=%p \n", connection, message);
#endif
		dbus_message_unref(message);
		return FALSE;
	}

	dbus_connection_flush(connection);
	dbus_message_unref(message);

#ifdef DEBUG
	g_printf("send_and_unref Success ! (\nDBusConnection *connection=%p, \nDBusMessage *message=%p \n", connection, message);
#endif
	return TRUE;
}

void append_dbus_args(DBusMessage *message, int first_arg_type, ...) {
	dbus_bool_t ret;
	va_list ap;

	va_start(ap, first_arg_type);
	ret = dbus_message_append_args_valist(message, first_arg_type, ap);
	va_end(ap);

	if (ret == FALSE) {
		g_printf("dbus_message_append_args failed");
	}
}

/*
gboolean sendSignal(DBusGConnection *gConnection, GPtrArray *response) {

	DBusMessage *signal;
	DBusConnection *connection;
	
#ifdef DEBUG
	ContactInfosElement *pContactInfosElement;
	g_printf("sendSignal(\nDBusConnection *connection=%p, \nGPtrArray *response=%p \n", connection, response);
	if (response && response->len>0) {
		pContactInfosElement = (ContactInfosElement *) g_ptr_array_index(response,0);
		g_printf("guint length=%d)\n First reponse=%s\n", response->len, response->len?pContactInfosElement->name:"no response");
	}
#endif
	
	connection = dbus_g_connection_get_connection(gConnection);

	signal = dbus_message_new_signal(CONTACTINFOS_DBUS_PATH, CONTACTINFOS_DBUS_INTERFACE, CONTACTINFOS_DBUS_RESULTS_SIGNAL);
	if (signal == NULL) {
		g_printf("Out of memory during dbus_message_new_signal()");
		return FALSE;
	}

	if (response) {
		append_dbus_args(signal,
			dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_STRING, G_TYPE_STRING, dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID)), G_TYPE_INVALID)),
			&response,
			DBUS_TYPE_INVALID);
	}
	
	return send_and_unref(connection, signal);
}
*/

gboolean sendElementarySignal(DBusGConnection *gConnection, const gchar *name, const gchar *company, const guint typeInfo, const gchar *type, const gchar *value, const guint elementId, const guint valueId, const gint retCode) {

	DBusMessage *signal;
	DBusConnection *connection;
	gboolean ret;
	
	gchar *_name;
	gchar *_company;
	gchar *_type;
	gchar *_value;
	
#ifdef DEBUG
	g_printf("sendElementarySignal(\nDBusConnection *connection=%p, \nname=%s \nCompany=%s \ntypeInfo=%d\n, type=%s,\nvalue=%s,\n valueId=%d\nretCode=%d\n", connection, name, company, typeInfo, type, value, valueId, retCode);
#endif
	
	connection = dbus_g_connection_get_connection(gConnection);

	signal = dbus_message_new_signal(CONTACTINFOS_DBUS_PATH, CONTACTINFOS_DBUS_INTERFACE, CONTACTINFOS_DBUS_RESULTS_SIGNAL);
	if (signal == NULL) {
		g_printf("Out of memory during dbus_message_new_signal()");
		return FALSE;
	}

#ifdef DEBUG
	g_printf("sendElementarySignal before append_dbus_args\n");
#endif
	
	if (name) {
		_name = g_strdup(name);
	} else {
		_name = g_strdup("");
	}
	if (company) {
		_company = g_strdup(company);
	} else {
		_company = g_strdup("");
	}
	if (type) {
		_type = g_strdup(type);
	} else {
		_type = g_strdup("");
	}
	if (value) {
		_value = g_strdup(value);
	} else {
		_value = g_strdup("");
	}
	
	
	dbus_message_append_args(signal,
		DBUS_TYPE_STRING, &_name,
		DBUS_TYPE_STRING, &_company,
		DBUS_TYPE_UINT32, &typeInfo,
		DBUS_TYPE_STRING, &_type,
		DBUS_TYPE_STRING, &_value,
		DBUS_TYPE_UINT32, &elementId,
		DBUS_TYPE_UINT32, &valueId,
		DBUS_TYPE_INT32, &retCode,
		DBUS_TYPE_INVALID);
	
#ifdef DEBUG
	g_printf("sendElementarySignal before send and unref\n");
#endif
	ret = send_and_unref(connection, signal);
	
	g_free(_name);
	g_free(_company);
	g_free(_type);
	g_free(_value);
	
	return ret;
}


void declareDbusServices() {
	DBusGProxy *driver_proxy;
	GError *err = NULL;
	guint request_name_result;
	DBusGConnection *connection = NULL;
	
	g_return_if_fail (connection == NULL);
	g_return_if_fail (the_service == NULL);

	connection = dbus_g_bus_get (DBUS_BUS_STARTER, &err);
	if (connection == NULL) {
		g_warning ("DBUS Service registration failed: %s", err ? err->message : "");
		g_error_free (err);
		gtk_exit(1);
	}
	/*
	dbus_connection_set_exit_on_disconnect (dbus_g_connection_get_connection (connection), FALSE);
	*/
	driver_proxy = dbus_g_proxy_new_for_name (connection, DBUS_SERVICE_DBUS,
					      DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

	if (!org_freedesktop_DBus_request_name (driver_proxy, CONTACTINFOS_DBUS_NAME,
					    0, &request_name_result, &err)) {
		g_warning ("DBUS Service name request failed.");
		g_clear_error (&err);
		gtk_exit(1);
	}

	if (request_name_result == DBUS_REQUEST_NAME_REPLY_EXISTS) {
		g_warning ("DBUS Service already started elsewhere");
		gtk_exit(1);
	}


	dbus_g_object_type_install_info (CONTACTINFOS_TYPE_SERVICE, 
				     &dbus_glib_contactinfos_object_info);

	the_service = g_object_new (CONTACTINFOS_TYPE_SERVICE, NULL);
	the_service->connection = connection;
	dbus_g_connection_register_g_object (connection, CONTACTINFOS_DBUS_PATH,
					 G_OBJECT (the_service));

}

/* -----------------------------------------------------------------------------
 * OBJECT 
 */

static void contactinfos_service_dispose (GObject *gobject)
{
	ContactinfosService *svc = CONTACTINFOS_SERVICE (gobject);
	GError *error = NULL;
    
	if (!terminateBackend(svc->backendData, &error)) {
#ifdef DEBUG
		g_printf("contactinfos_service_dispose retCode=%d\n msg=%s\n backEnd = %p\n", error->code, error->message, svc->backendData);
#endif
	}
	
	g_free(svc->searchText);
	
	G_OBJECT_CLASS (contactinfos_service_parent_class)->dispose (gobject);
}

static void contactinfos_service_class_init (ContactinfosServiceClass *klass)
{
	GObjectClass *gobject_class;

	contactinfos_service_parent_class = g_type_class_peek_parent (klass);
	gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->dispose = contactinfos_service_dispose;
	
}

static void contactinfos_service_init (ContactinfosService *svc)
{
	svc->backendData = NULL;
	svc->appData = newAppUIData();
#ifdef DEBUG
	g_printf("contactinfos_service_init (ContactinfosService *svc=%p)\n", svc);
#endif
	if (!initBackend(&(svc->backendData), &(svc->error))) {
		g_printf("contactinfos_service_init retCode=%d\n msg=%s\n backEnd = %p\n", svc->error->code, svc->error->message, svc->backendData);
	}
	setExitTimer(svc->appData, FALSE);
}

/* Signal interne pour dcoupler l'appel du traitement */

gboolean startSearching_cb(ContactinfosService *svc) {
	gboolean ret;
	GPtrArray *response = NULL;
	GError *localError = NULL;
	guint retCode = CI_NO_ERROR;
	
	/* See DBus cannot handle struct and arrays */
	guint index;
	guint index2;
	gchar *name;
	gchar *company;
	ContactInfosElement *pContactInfosElement;
	ContactInfosValue *pContactInfosValue;
	GPtrArray *pValueArray;

#ifdef DEBUG	
	g_printf("startSearching_cb(\n  ContactinfosService *svc %p\n", svc);
#endif
	if (!svc) {
		g_printf("No Object !!!\n");
		return FALSE;
	}
	
	if (svc->error) {
		g_printf("Earlier Error %d : %s\n", svc->error->code, svc->error->message);
		sendElementarySignal(svc->connection, "", "", 0, "", "", CI_LAST_ELEMENT_ID, CI_LAST_VALUE_ID, svc->error->code);
		return FALSE;
	}
	
	removeExitTimer(svc->appData);

#ifdef DEBUG
	g_printf("startSearching_cb :\n requestedInfos=%d,\n searchText=%s,\n showGui=%d,\n multipleReponse=%d\n upperLimit=%d\n", svc->requestedInfos, svc->searchText, svc->showGui, svc->multipleReponse, svc->upperLimit);
#endif
	ret = interface_dialog_new(
			svc->appData, 
			NULL, 
			svc->multipleReponse, 
			svc->showGui, 
			svc->confirm, 
			svc->searchText, 
			&response, 
			svc->backendData, 
			svc->requestedInfos, 
			svc->upperLimit, 
			&localError);

#ifdef DEBUG
	g_printf("ret %d startSearching_cb \n", ret);
#endif
	if (localError) {
#ifdef DEBUG
		g_printf("startSearching_cb error code=%d, msg=%s\n", localError->code, localError->message);
#endif
		retCode = localError->code;
	}

#ifdef DEBUG
	g_printf("startSearching_cb before sending Dbus signal\n");
#endif

	/* send DBus-Signal */
	/* DBus cannot handle complexe Struct and Array yet */
/*
	sendSignal(svc->connection, response);
*/
	if (response) {
		for (index=0; index<response->len; index++) {
			pContactInfosElement = (ContactInfosElement *) response->pdata[index];
			name = pContactInfosElement->name;
			company = pContactInfosElement->company;
			pValueArray = pContactInfosElement->valueArray;
			if (pValueArray) {
				for (index2=0; index2<pValueArray->len; index2++) {
					pContactInfosValue = (ContactInfosValue *) pValueArray->pdata[index2];
					sendElementarySignal(
							svc->connection, 
							name, 
							company, 
							pContactInfosValue->typeInfo, 
							pContactInfosValue->type, 
							pContactInfosValue->value, 
							(response->len)-index-1, 
							(pValueArray->len)-index2-1, 
							retCode);
				}
			}
			freeContactInfosElement(pContactInfosElement);
		}
		g_ptr_array_free(response, TRUE);
	} else {
		if (retCode == CI_NO_ERROR) {
			retCode = CI_NOT_FOUND;
		}
		sendElementarySignal(svc->connection, "", "", 0, "", "", CI_LAST_ELEMENT_ID, CI_LAST_VALUE_ID, retCode);
	}
	
#ifdef DEBUG
	g_printf("startSearching_cb after sending Dbus signal\n");
#endif
	setExitTimer(svc->appData, FALSE);

	return FALSE;
}

/* -----------------------------------------------------------------------------
 * DBUS METHODS 
 */

gboolean server_is_available (ContactinfosService *svc, gboolean* availability, GError **error) {
	
	(*availability) = TRUE;
	if (svc->error) {
		g_propagate_error(error, svc->error);
		(*availability) = FALSE;
	}
	setExitTimer(svc->appData, FALSE);
	return TRUE;
}

gboolean server_get_available_contact_infos (ContactinfosService *svc, guint requestedInfos, guint* availableInfos, GError **error) {
	
	GError *localError=NULL;
	gboolean ret;
	if (svc->error) {
		g_propagate_error(error, svc->error);
		return FALSE;
	}
	removeExitTimer(svc->appData);

	ret = backendGetAvailableContactInfos (svc->backendData, requestedInfos, availableInfos, &localError);
	g_propagate_error(error, localError);
	
	setExitTimer(svc->appData, FALSE);
	return ret;
}

gboolean server_get_contact_infos (ContactinfosService *svc, guint requestedInfos, gchar * searchText, gboolean showGui, gboolean confirm, gboolean multipleReponse, guint upperLimit, gboolean* availability, GError **error) {

	if (svc->error) {
		g_propagate_error(error, svc->error);
		return TRUE;
	}

	removeExitTimer(svc->appData);
	svc->requestedInfos=requestedInfos;
	svc->searchText=g_strdup(searchText);
	svc->showGui=showGui;
	svc->confirm=confirm;
	svc->multipleReponse=multipleReponse;
	svc->upperLimit=upperLimit;

#ifdef DEBUG
	g_printf("server_get_contact_infos (\n ContactinfosService *svc=%p,\n guint requestedInfos=%d,\n gchar * searchText=%s,\n gboolean showGui=%d,\n gboolean multipleReponse=%d\nguint upperLimit=%d", svc, requestedInfos, searchText, showGui, multipleReponse, upperLimit);
	g_printf("server_get_contact_infos before sending local signal\n");
#endif

	g_idle_add ((GSourceFunc)startSearching_cb, svc);
	*availability = TRUE;
#ifdef DEBUG
	g_printf("server_get_contact_infos after sending local signal\n");
	g_printf("server_get_contact_infos before returning\n");
#endif
/*
	setExitTimer(svc->appData, FALSE);
*/
	return TRUE;
}



void undeclareDbusServices() {
}

