/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Xavier Claessens <xavier.claessens@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "contact-window.h"

struct _ContactWindowPrivate
{
  OssoABookContact *contact;

  OssoABookContactDetailStore *details;
  guint n_rows;

  GtkWidget *alignment;
  GtkWidget *no_contact_label;
  GtkWidget *info_pa;
  GtkWidget *info_table;
};

static const OssoABookMessageMapping msgid_table[] =
{
  { "", "addr_fi_mecard" }, /* fallback prefix */
  { NULL, NULL }
};

G_DEFINE_TYPE (ContactWindow, contact_window, HILDON_TYPE_STACKABLE_WINDOW);

enum
{
  PROP_CONTACT = 1,
  N_PROPS
};

static void
add_info_field (ContactWindow *self,
    const gchar *title,
    const gchar *value)
{
  GtkWidget *title_label;
  GtkWidget *value_label;

  /* If title or value is empty, skip this field */
  if (!title || title[0] == '\0' ||
      !value || value[0] == '\0')
    return;

  /* Add title to the info table */
  title_label = gtk_label_new (title);
  gtk_misc_set_alignment (GTK_MISC (title_label), 0, 0);
  gtk_label_set_justify (GTK_LABEL (title_label), GTK_JUSTIFY_LEFT);
  gtk_table_attach (GTK_TABLE (self->priv->info_table), title_label,
      0, 1, self->priv->n_rows, self->priv->n_rows + 1,
      GTK_FILL, GTK_FILL,
      0, 0);
  hildon_helper_set_logical_color (title_label, GTK_RC_FG, GTK_STATE_NORMAL,
      "SecondaryTextColor");
  gtk_widget_show (title_label);

  /* Add value to the info table */
  value_label = gtk_label_new (value);
  gtk_misc_set_alignment (GTK_MISC (value_label), 0, 0);
  gtk_table_attach (GTK_TABLE (self->priv->info_table), value_label,
      1, 2, self->priv->n_rows, self->priv->n_rows + 1,
      GTK_FILL | GTK_EXPAND, GTK_FILL,
      0, 0);
  gtk_label_set_line_wrap_mode (GTK_LABEL (value_label), PANGO_WRAP_WORD_CHAR);
  gtk_label_set_line_wrap (GTK_LABEL (value_label), TRUE);
  hildon_helper_set_logical_font (value_label, "LargeSystemFont");
  gtk_widget_show (value_label);

  self->priv->n_rows++;
}

static void
update_info_table (ContactWindow *self)
{
  GSequence *seq;
  GSequenceIter *iter;
  gint n_details;

  /* Destroy the current widgets if any */
  if (self->priv->info_pa != NULL)
    {
      gtk_widget_destroy (self->priv->info_pa);
      self->priv->info_pa = NULL;
    }
  if (self->priv->no_contact_label != NULL)
    {
      gtk_widget_destroy (self->priv->no_contact_label);
      self->priv->no_contact_label = NULL;
    }

  /* Display "no detail" if we have no detail */
  seq = osso_abook_contact_detail_store_get_fields (self->priv->details);
  if (seq == NULL)
    {
      self->priv->no_contact_label = gtk_label_new (g_dgettext (
          "osso-addressbook", "addr_ia_no_details"));
      gtk_misc_set_alignment (GTK_MISC (self->priv->no_contact_label),
          0.5, 0.5);
      hildon_helper_set_logical_font (self->priv->no_contact_label,
          "LargeSystemFont");
      gtk_container_add (GTK_CONTAINER (self->priv->alignment),
          self->priv->no_contact_label);
      gtk_widget_show (self->priv->no_contact_label);
      return;
    }

  /* Create a new info table */
  n_details = g_sequence_get_length (seq);
  self->priv->info_pa = hildon_pannable_area_new ();

  g_object_set (self->priv->info_pa,
      "hscrollbar-policy", GTK_POLICY_NEVER,
      "mov-mode", HILDON_MOVEMENT_MODE_VERT,
      NULL);

  gtk_container_add (GTK_CONTAINER (self->priv->alignment),
      self->priv->info_pa);
  gtk_widget_show (self->priv->info_pa);

  self->priv->info_table = gtk_table_new (n_details, 2, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (self->priv->info_table),
      HILDON_MARGIN_TRIPLE);
  gtk_table_set_row_spacings (GTK_TABLE (self->priv->info_table),
      HILDON_MARGIN_DEFAULT);

  hildon_pannable_area_add_with_viewport (
      HILDON_PANNABLE_AREA (self->priv->info_pa), self->priv->info_table);
  gtk_widget_show (self->priv->info_table);

  /* Fill the table */
  self->priv->n_rows = 0;
  iter = g_sequence_get_begin_iter (seq);
  while (!g_sequence_iter_is_end (iter))
    {
      OssoABookContactField *field;

      field = g_sequence_get (iter);
      add_info_field (self,
          osso_abook_contact_field_get_display_title (field),
          osso_abook_contact_field_get_display_value (field));
      iter = g_sequence_iter_next (iter);
    }
}

static void
constructed (GObject *object)
{
  ContactWindow *self = CONTACT_WINDOW (object);

  /* Create a detail store for the contact */
  self->priv->details = osso_abook_contact_detail_store_new (self->priv->contact,
      OSSO_ABOOK_CONTACT_DETAIL_ALL);
  osso_abook_contact_detail_store_set_message_map (self->priv->details,
      msgid_table);

  /* Top/Left/Right border */
  self->priv->alignment = gtk_alignment_new (0.5, 0.5, 1, 1);
  gtk_alignment_set_padding (GTK_ALIGNMENT (self->priv->alignment),
      HILDON_MARGIN_DOUBLE, 0, HILDON_MARGIN_DOUBLE, HILDON_MARGIN_DOUBLE);
  gtk_container_add (GTK_CONTAINER (self), self->priv->alignment);
  gtk_widget_show (self->priv->alignment);

  /* Create the info table */
  update_info_table (self);
  g_signal_connect_swapped (self->priv->details, "changed",
      G_CALLBACK (update_info_table), self);

  ((GObjectClass *) contact_window_parent_class)->constructed (object);
}

static void
get_property (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec)
{
  ContactWindow *self = CONTACT_WINDOW (object);

  switch (property_id)
    {
    case PROP_CONTACT:
      g_value_set_object (value, self->priv->contact);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
set_property (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec)
{
  ContactWindow *self = CONTACT_WINDOW (object);

  switch (property_id)
    {
    case PROP_CONTACT:
      g_assert (self->priv->contact == NULL);
      self->priv->contact = g_value_dup_object (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
dispose (GObject *object)
{
  ContactWindow *self = CONTACT_WINDOW (object);

  if (self->priv->contact != NULL)
    g_object_unref (self->priv->contact);
  self->priv->contact = NULL;

  if (self->priv->details != NULL)
    g_object_unref (self->priv->details);
  self->priv->details = NULL;

  ((GObjectClass *) contact_window_parent_class)->dispose (object);
}

static void
contact_window_init (ContactWindow *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, CONTACT_TYPE_WINDOW,
      ContactWindowPrivate);
}

static void
contact_window_class_init (ContactWindowClass *klass)
{
  GObjectClass *object_class = (GObjectClass *) klass;
  GParamSpec *param_spec;

  g_type_class_add_private (klass, sizeof (ContactWindowPrivate));
  object_class->constructed = constructed;
  object_class->get_property = get_property;
  object_class->set_property = set_property;
  object_class->dispose = dispose;

  param_spec = g_param_spec_object ("contact", "OssoABookContact object",
      "Contact to display",
      OSSO_ABOOK_TYPE_CONTACT,
      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (object_class, PROP_CONTACT, param_spec);
}

GtkWidget *
contact_window_new (OssoABookContact *contact)
{
  g_return_val_if_fail (OSSO_ABOOK_IS_CONTACT (contact), NULL);

  return g_object_new (CONTACT_TYPE_WINDOW, "title", "Contact View",
      "contact", contact, NULL);
}

