/*
* Copyright (C) 2008-2010 J-P Nurmi jpnurmi@gmail.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
* $Id$
*/

#include "mainwindow.h"
#include "application.h"
#include "connectionwizard.h"
#include "sessiontabwidget.h"
#include "shortcutmanager.h"
#include "maintabwidget.h"
#include "hometabwidget.h"
#include "connection.h"
#include "session.h"
#include <QtGui>

MainWindow::MainWindow(QWidget* parent) : QMainWindow(parent), trayIcon(0)
{
    tabWidget = new MainTabWidget(this);
    connect(tabWidget, SIGNAL(currentChanged(int)), this, SLOT(tabActivated(int)));
    connect(tabWidget, SIGNAL(alertStatusChanged(bool)), this, SLOT(activateAlert(bool)));
    setCentralWidget(tabWidget);

    if (QSystemTrayIcon::isSystemTrayAvailable())
    {
        trayIcon = new TrayIcon(this);
        trayIcon->setIcon(QApplication::windowIcon());
        trayIcon->setVisible(true);
        connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
            this, SLOT(trayIconActivated(QSystemTrayIcon::ActivationReason)));
    }

    ShortcutManager* manager = ShortcutManager::instance(this);
    connect(manager->shortcut(ShortcutManager::NavigateUp), SIGNAL(activated()), tabWidget, SLOT(moveToPrevTab()));
    connect(manager->shortcut(ShortcutManager::NavigateDown), SIGNAL(activated()), tabWidget, SLOT(moveToNextTab()));
    connect(manager->shortcut(ShortcutManager::NavigateLeft), SIGNAL(activated()), tabWidget, SLOT(moveToPrevSubTab()));
    connect(manager->shortcut(ShortcutManager::NavigateRight), SIGNAL(activated()), tabWidget, SLOT(moveToNextSubTab()));

    QShortcut* shortcut = new QShortcut(QKeySequence(tr("Ctrl+Q")), this);
    connect(shortcut, SIGNAL(activated()), this, SLOT(close()));

#if defined(Q_WS_MAC) || defined(Q_WS_MAEMO_5)
    QMenu* menu = new QMenu(this);
    menuBar()->addMenu(menu);

    QAction* action = new QAction(tr("Settings"), this);
    action->setMenuRole(QAction::PreferencesRole);
    connect(action, SIGNAL(triggered()), qApp, SLOT(showSettings()));
    menu->addAction(action);

    action = new QAction(tr("About %1").arg(Application::applicationName()), this);
    action->setMenuRole(QAction::AboutRole);
    connect(action, SIGNAL(triggered()), qApp, SLOT(aboutApplication()));
    menu->addAction(action);

#ifdef Q_WS_MAEMO_5
    action = new QAction(tr("About Oxygen"), this);
    connect(action, SIGNAL(triggered()), qApp, SLOT(aboutOxygen()));
    menu->addAction(action);
#endif // Q_WS_MAEMO_5

    action = new QAction(tr("About Qt"), this);
    action->setMenuRole(QAction::AboutQtRole);
    connect(action, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
    menu->addAction(action);
#endif // Q_WS_MAC || Q_WS_MAEMO_5

    QSettings settings;
    if (settings.contains("geometry"))
        restoreGeometry(settings.value("geometry").toByteArray());

    HomeTabWidget* homeTab = new HomeTabWidget(tabWidget);
    tabWidget->addTab(homeTab, tr("Home"));
    connect(homeTab, SIGNAL(connectTo(Connection)), this, SLOT(connectToImpl(Connection)));

    connect(qApp, SIGNAL(connectionChanged(int, Connection)), this, SLOT(updateConnection(int, Connection)));

    QTimer::singleShot(0, this, SLOT(initialize()));
}

QSize MainWindow::sizeHint() const
{
    return QSize(800, 600);
}

void MainWindow::connectTo(const QString& host, quint16 port, const QString& nick, const QString& password)
{
    Connection conn;
    conn.host = host;
    conn.port = port;
    conn.nick = nick;
    conn.pass = password;
    connectTo(conn);
}

void MainWindow::connectTo(const Connection& connection)
{
    ConnectionWizard wizard;
    wizard.setConnection(connection);

    if (!connection.host.isEmpty() && !connection.nick.isEmpty())
    {
        connectToImpl(connection);
    }
    else if (wizard.exec())
    {
        Connection connection = wizard.connection();
        if (!connection.name.isEmpty())
        {
            Application::setConnections(Application::connections() << connection);
        }
        connectToImpl(connection);
    }
}

void MainWindow::connectToImpl(const Connection& connection)
{
    for (int i = 1; i < tabWidget->count(); ++i)
    {
        SessionTabWidget* tab  = qobject_cast<SessionTabWidget*>(tabWidget->widget(i));
        if (tab)
        {
            Session* session = tab->session();
            if (session && session->connection() == connection)
            {
                tabWidget->setCurrentIndex(i);
                return;
            }
        }
    }

    Session* session = new Session(this);
    session->connectTo(connection);

    SessionTabWidget* tab = new SessionTabWidget(session, tabWidget);
    connect(tab, SIGNAL(disconnectFrom(QString)), this, SLOT(disconnectFrom(QString)));
    if (connection.name.isEmpty())
        connect(tab, SIGNAL(titleChanged(QString)), tabWidget, SLOT(setSessionTitle(QString)));
    connect(tab, SIGNAL(alertStatusChanged(bool)), tabWidget, SLOT(activateAlert(bool)));
    connect(tab, SIGNAL(highlightStatusChanged(bool)), tabWidget, SLOT(activateHighlight(bool)));
    tabWidget->addTab(tab, connection.name.isEmpty() ? session->host() : connection.name);

    session->addBuffer(session->host());
}

void MainWindow::disconnectFrom(const QString& message)
{
    SessionTabWidget* tab = qobject_cast<SessionTabWidget*>(sender());
    if (!tab)
        tab = qobject_cast<SessionTabWidget*>(tabWidget->currentWidget());
    Q_ASSERT(tab);
    QString reason = message.trimmed();
    if (reason.isEmpty())
        reason = tr("%1 %2 - %3").arg(Application::applicationName())
        .arg(Application::applicationVersion())
        .arg(Application::organizationDomain());
    tab->session()->quit(reason);
    tab->session()->disconnectFromServer();
    // automatically rejoin channels when reconnected
    tab->session()->setAutoJoinChannels(tab->channels());
}

void MainWindow::quit(const QString& message)
{
    // TODO: confirm?
    for (int i = 0; i < tabWidget->count(); ++i)
    {
        SessionTabWidget* tab = qobject_cast<SessionTabWidget*>(tabWidget->widget(i));
        QString reason = message.trimmed();
        if (reason.isEmpty())
            reason = tr("%1 %2 - %3").arg(Application::applicationName())
                                     .arg(Application::applicationVersion())
                                     .arg(Application::organizationDomain());
        tab->session()->quit(reason);
        tab->session()->disconnectFromServer();
    }
    close();
}

void MainWindow::closeEvent(QCloseEvent* event)
{
    QSettings settings;
    settings.setValue("geometry", saveGeometry());

    QMainWindow::closeEvent(event);
}

void MainWindow::changeEvent(QEvent* event)
{
    QMainWindow::changeEvent(event);
    if (event->type() == QEvent::ActivationChange)
    {
        if (trayIcon && isActiveWindow())
            trayIcon->unalert();
    }
}

void MainWindow::initialize()
{
    foreach (const Connection& connection, Application::connections())
    {
        if (connection.autoConnect)
            connectToImpl(connection);
    }
    if (Application::connections().isEmpty())
        connectTo(Connection());
}

void MainWindow::trayIconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason)
    {
        case QSystemTrayIcon::DoubleClick:
            setVisible(!isVisible());
            break;
        case QSystemTrayIcon::Trigger:
            raise();
            activateWindow();
            break;
        default:
            break;
    }
}

void MainWindow::activateAlert(bool activate)
{
    if (!isActiveWindow())
    {
        if (activate)
        {
            if (trayIcon)
                trayIcon->alert();
            QApplication::alert(this);
        }
        else
        {
            if (trayIcon)
                trayIcon->unalert();
        }
    }
}

void MainWindow::tabActivated(int index)
{
    if (QTabWidget* tab = qobject_cast<QTabWidget*>(tabWidget->widget(index)))
    {
        setWindowFilePath(tab->tabText(tab->currentIndex()));
        if (index > 0)
            QMetaObject::invokeMethod(tab, "delayedTabReset");
    }
}

void MainWindow::updateConnection(int index, const Connection& connection)
{
    index++; // skip home tab
    SessionTabWidget* tab  = qobject_cast<SessionTabWidget*>(tabWidget->widget(index));
    if (tab)
    {
        disconnect(tab, SIGNAL(titleChanged(QString)), tabWidget, SLOT(setSessionTitle(QString)));

        if (connection.name.isEmpty())
        {
            connect(tab, SIGNAL(titleChanged(QString)), tabWidget, SLOT(setSessionTitle(QString)));
            tabWidget->setTabText(index, tab->tabText(0));
        }
        else
        {
            tabWidget->setTabText(index, connection.name);
        }
    }
}
