#include "TrackView.h"

#include "CompassPrivate.h"

#include <math.h>

#include <QDebug>
#include <QPainter>

class TrackViewPrivate
{
public:
    ColumbusModel *model;
    ColumbusWaypointModel *waypoints;

    qreal zoom;
    qreal rotation;
    QPointF position;
};

TrackView::TrackView(QWidget *parent) : QWidget(parent)
{
    this->d = new TrackViewPrivate;

    d->model = NULL;
    d->waypoints = NULL;

    d->zoom  = 20.0f;
    d->rotation = 0.0f;
    d->position = QPointF(0.0f, 0.0f);
}

TrackView::~TrackView()
{
    delete this->d;
}

ColumbusModel* TrackView::model()
{
    return d->model;
}

void TrackView::setModel(ColumbusModel *model)
{
    d->model = model;
    this->update();
    emit this->modelUpdated(d->model);
}

ColumbusWaypointModel* TrackView::waypoints()
{
    return d->waypoints;
}

void TrackView::setWaypoints(ColumbusWaypointModel *waypoints)
{
    d->waypoints = waypoints;
    this->update();
}

qreal TrackView::zoom() const
{
    return d->zoom;
}

void TrackView::setZoom(qreal zoom)
{
    d->zoom = zoom;
    this->update();
    emit this->zoomUpdated(d->zoom);
}

qreal TrackView::rotation() const
{
    return d->rotation;
}

void TrackView::setRotation(double rotation)
{
    d->rotation = rotation;
    this->update();
    emit this->rotationUpdated((qreal)rotation);
}

QPointF TrackView::position() const
{
    return d->position;
}

void TrackView::setPosition(QPointF position)
{
    d->position = position;
    this->update();
    emit this->positionUpdated(d->position);
}

void TrackView::paintEvent(QPaintEvent*)
{
    const int width     = this->width();
    const int height    = this->height();
    const int hwidth    = width / 2;
    const int hheight   = height / 2;

    qreal dim           = 1000 * d->zoom;

    QPointF p1(0, 0);
    QPointF p2(0, width);

    QGeoCoordinate c1(p1.y() / dim, p1.x() / dim);
    QGeoCoordinate c2(p2.y() / dim, p2.x() / dim);

    qreal wrd = c1.distanceTo(c2);
    qreal mpp = wrd / width;

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing, true);

    painter.setViewport(0, 0, width, height);
    painter.setWindow(-(width / 2), -(height / 2), width, height); // To make center (0, 0) units.

    painter.setPen(QPen(this->palette().foreground(), 1.0f));
    painter.setBrush(QBrush(this->palette().foreground()));

    painter.drawPoint(0, 0);

    painter.setMatrix(QMatrix());
    painter.rotate(-d->rotation);

    int px = 0;
    int py = 0;

    if(d->model != NULL)
    {
        qreal lat = d->position.y();
        qreal lon = d->position.x();

        if(d->model->tracklog()->count() > 0)
        {
            lat = d->model->tracklog()->last().latitude();
            lon = d->model->tracklog()->last().longitude();

            d->position = QPointF(lat, lon);

            for(int i = d->model->tracklog()->count() - 1; i > 0; i--)
            {
                qreal rLat = d->model->tracklog()->value(i).latitude();
                qreal rLon = d->model->tracklog()->value(i).longitude();

                int x = (lon - rLon) * dim;
                int y = (lat - rLat) * dim;

                painter.drawLine(-px, py, -x, y);

                px = x; py = y;
            }
        }

        if(d->waypoints != NULL)
        {
            for(int i = 0; i < d->waypoints->rowCount(); i++)
            {
                QColor wpColor = this->palette().foreground().color();
                ColumbusWaypoint wpt = d->waypoints->waypoint(d->waypoints->index(i, 0));

                int x = (lon - wpt.coordinate().longitude()) * dim;
                int y = (lat - wpt.coordinate().latitude()) * dim;

                if(d->waypoints->index(i, 0).row() == d->waypoints->destination().row())
                {
                    wpColor = this->palette().highlight().color();
                }

                painter.setMatrix(QMatrix());
                painter.rotate(-d->rotation);
                painter.translate(-x, y);

                // Reverse rotation so we render marker and textstraight.
                painter.rotate(d->rotation);

                QTextOption opt;
                opt.setWrapMode(QTextOption::NoWrap);

                painter.setPen(this->palette().highlight().color());
                painter.setBrush(this->palette().highlight());
                painter.setFont(QFont(painter.font().family(), 10));

                QRectF rect = painter.boundingRect(QRect(8, -20, 0, 0), wpt.name(), opt);
                painter.drawText(rect, wpt.name(), opt);

                QString distance = "-.-- " + d->model->distanceUnits();

                if(d->model->datalog()->length() > 0)
                {
                    distance = QString::number(ColumbusModel::convertDistanceToUnits(d->model->displayUnits(), d->model->currentPosition().distanceTo(wpt.coordinate())), 'f', 2) + " " + d->model->distanceUnits();
                }

                rect = painter.boundingRect(QRect(8, 0, 0, 0), distance, opt);
                painter.drawText(rect, distance, opt);

                painter.setPen(wpColor);
                wpColor.setAlphaF(0.5f);
                painter.setBrush(QBrush(wpColor));

                painter.scale(6.0f, 20.0f);
                painter.drawPolygon(QPolygon(4, &BearingMarker[0][0]));
            }
        }
    }
}
