#include "ColumbusStatusView.h"

#include "config.h"

#include "widgets/CompassSlide.h"
#include "widgets/SatelliteCompassDial.h"
#include "widgets/AccuracyGraph.h"
#include "widgets/SignalGraph.h"
#include "widgets/LCDIndicator.h"

#include <qnumeric.h>
#include <QDebug>
#include <QPainter>
#include <QBoxLayout>
#include <QFormLayout>
#include <QLabel>
#include <QPropertyAnimation>

class ColumbusStatusViewPrivate
{
public:
    // Application controller reference.
    ApplicationController   *controller;

    // View widget components.
    CompassSlide            *compass;
    SatelliteCompassDial    *satelliteView;
    AccuracyGraph           *accuracyGraph;
    SignalGraph             *signalGraph;

    // Heading & Speed telemetry indicators.
    LCDIndicator  *speedIndicator;
    LCDIndicator  *headingIndicator;

    // Positional information.
    QLabel          *time;
    QLabel          *altitude;
    QLabel          *latitude;
    QLabel          *longitude;
    QLabel          *hvacc;

    // Eye candy.
    QPropertyAnimation  *compassAnimation;
};

ColumbusStatusView::ColumbusStatusView(QWidget *parent)
    : ApplicationView(parent)
{
    this->d = new ColumbusStatusViewPrivate;

    QBoxLayout  *mbox   = new QBoxLayout(QBoxLayout::LeftToRight, this);
    mbox->setMargin(0);

    QVBoxLayout *vboxl  = new QVBoxLayout();
    QVBoxLayout *vboxr  = new QVBoxLayout();

    vboxl->setMargin(0);
    vboxl->setContentsMargins(0, 0, 0, 0);

    d->controller = NULL;

    // Compass Slide
    d->compass = new CompassSlide(this);
    vboxl->addWidget(d->compass);

    // Satellite Tracking Compass
    d->satelliteView = new SatelliteCompassDial(this);
    vboxl->addWidget(d->satelliteView);

    // Setup heading animation.
    d->compassAnimation = new QPropertyAnimation(d->compass, "heading", this);
    d->compassAnimation->setDuration(750);
    d->compassAnimation->setEasingCurve(QEasingCurve::InOutQuad);

    QObject::connect(d->compass, SIGNAL(headingChanged(double)), d->satelliteView, SLOT(setHeading(double)));

    // Heading & Speed Detail.
    QVBoxLayout *svbox  = new QVBoxLayout();
    QHBoxLayout *hboxl  = new QHBoxLayout();

    svbox->setMargin(0);
    svbox->setContentsMargins(0, 0, 0, 0);
    hboxl->setMargin(0);
    hboxl->setContentsMargins(0, 0, 0, 0);

    d->headingIndicator = new LCDIndicator(this);
    d->headingIndicator->setFixedWidth(125);
    d->headingIndicator->setUnitIndicator("deg");

    d->speedIndicator = new LCDIndicator(this);
    d->speedIndicator->setFixedWidth(125);

    hboxl->addWidget(d->headingIndicator, 0, Qt::AlignLeft | Qt::AlignTop);
    hboxl->addWidget(d->speedIndicator, 0, Qt::AlignRight | Qt::AlignTop);

    svbox->addLayout(hboxl);

    d->satelliteView->setLayout(svbox);

    // Position & Fix Information.
    QFormLayout *posl = new QFormLayout();
    posl->setLabelAlignment(Qt::AlignRight | Qt::AlignVCenter);

    d->latitude  = ValueLabel("-");
    d->longitude = ValueLabel("-");
    d->altitude = ValueLabel("-");
    d->hvacc = ValueLabel("-");

    QLabel *lLatitude = new QLabel(tr("Latitude"), this);
    lLatitude->setMinimumWidth(150);
    lLatitude->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    posl->addRow(lLatitude, d->latitude);
    posl->addRow(tr("Longitude"), d->longitude);
    posl->addRow(tr("Altitude"), d->altitude);
    posl->addRow(tr("H/V-Acc."), d->hvacc);

    vboxr->addSpacing(10);
    vboxr->addLayout(posl);

    // DOP Accuracy Graph
    d->accuracyGraph = new AccuracyGraph(this);
    vboxr->addWidget(d->accuracyGraph);

    // Satellite Signal Graph
    d->signalGraph = new SignalGraph(this);
    vboxr->addWidget(d->signalGraph);

    mbox->addLayout(vboxl);
    mbox->addLayout(vboxr);

    QObject::connect(parent, SIGNAL(onLandscape()), this, SLOT(onLandscape()));
    QObject::connect(parent, SIGNAL(onPortrait()), this, SLOT(onPortrait()));
}

ColumbusStatusView::~ColumbusStatusView()
{
    delete this->d;
}

QString ColumbusStatusView::friendlyName() const
{
    static const QString friendlyName = QString("Status");
    return friendlyName;
}

void ColumbusStatusView::setController(ApplicationController *controller)
{
    if(d->controller != NULL)
    {
        QObject::disconnect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                            this, SLOT(onPositionUpdated(QGeoPositionInfo)));

        QObject::disconnect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                            d->accuracyGraph, SLOT(update()));

        QObject::disconnect(d->controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                            d->signalGraph, SLOT(setSatsInUse(QList<QGeoSatelliteInfo>)));
        QObject::disconnect(d->controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                            d->signalGraph, SLOT(setSatsInView(QList<QGeoSatelliteInfo>)));

        QObject::disconnect(d->controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                            d->satelliteView, SLOT(setSatsInUse(QList<QGeoSatelliteInfo>)));
        QObject::disconnect(d->controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                            d->satelliteView, SLOT(setSatsInView(QList<QGeoSatelliteInfo>)));
    }

    d->controller = controller;

    if(controller != NULL)
    {
        QObject::connect(controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                         this, SLOT(onPositionUpdated(QGeoPositionInfo)));

        QObject::connect(controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                         d->signalGraph, SLOT(setSatsInUse(QList<QGeoSatelliteInfo>)));

        QObject::connect(controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                         d->signalGraph, SLOT(setSatsInView(QList<QGeoSatelliteInfo>)));

        QObject::connect(controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                         d->satelliteView, SLOT(setSatsInUse(QList<QGeoSatelliteInfo>)));

        QObject::connect(controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                         d->satelliteView, SLOT(setSatsInView(QList<QGeoSatelliteInfo>)));

        d->accuracyGraph->setModel(controller->tripModel());
        QObject::connect(controller, SIGNAL(positionUpdated(QGeoPositionInfo)), d->accuracyGraph, SLOT(update()));
    }
}

QLabel* ColumbusStatusView::ValueLabel(const QString &text) const
{
    QLabel *result = new QLabel(text, (QWidget*)this);
    QPalette palette = result->palette();
    QFont font;
    font.setPixelSize(24);

    palette.setColor(result->foregroundRole(), QColor(0xff, 0xff, 0xff));

    result->setAlignment(Qt::AlignLeft);
    result->setFont(font);
    result->setPalette(palette);

    return result;
}

void ColumbusStatusView::onLandscape()
{
    d->accuracyGraph->show();
    d->signalGraph->show();

    ((QBoxLayout*)this->layout())->setDirection(QBoxLayout::LeftToRight);
}

void ColumbusStatusView::onPortrait()
{
    d->accuracyGraph->hide();
    d->signalGraph->hide();

    ((QBoxLayout*)this->layout())->setDirection(QBoxLayout::TopToBottom);
}

void ColumbusStatusView::onPositionUpdated(const QGeoPositionInfo &posinfo)
{
    const TripModel *model = d->controller->tripModel();

    qreal speed = model->currentSpeed();
    qreal altitude = model->currentAltitude();
    qreal direction = model->currentHeading();

    QString latitude = model->currentLatitude();
    QString longitude = model->currentLongitude();

    if(direction != d->headingIndicator->value())
    {
        if(d->compassAnimation->state() == QPropertyAnimation::Running) d->compassAnimation->stop();

        d->compassAnimation->setStartValue(d->compassAnimation->targetObject()->property(d->compassAnimation->propertyName()));
        d->compassAnimation->setEndValue(direction);
        d->compassAnimation->start();
    }

    d->speedIndicator->setUnitIndicator(model->speedUnits());
    d->speedIndicator->setValue(speed);

    d->headingIndicator->setValue(direction);

    if(posinfo.hasAttribute(QGeoPositionInfo::HorizontalAccuracy))
    {
        qreal hdop = posinfo.attribute(QGeoPositionInfo::HorizontalAccuracy);

        if(!qIsNaN(hdop) && hdop < 500)
        {
            d->hvacc->setText(QString::number(hdop, 'f', 2) + "/");
        }
        else
        {
            d->hvacc->setText("---.--/");
        }
    }
    else
    {
        d->hvacc->setText("---.--/");
    }

    if(posinfo.hasAttribute(QGeoPositionInfo::VerticalAccuracy))
    {
        qreal vdop = posinfo.attribute(QGeoPositionInfo::VerticalAccuracy);

        if(!qIsNaN(vdop) && vdop < 500)
        {
            d->hvacc->setText(d->hvacc->text() + QString::number(vdop, 'f', 2));
        }
        else
        {
            d->hvacc->setText(d->hvacc->text() + "---.--");
        }
    }
    else
    {
        d->hvacc->setText(d->hvacc->text() + "---.--");
    }

    d->latitude->setText(latitude);
    d->longitude->setText(longitude);
    d->altitude->setText(QString::number(altitude, 'f', 2) + " <sub>" + d->controller->tripModel()->altitudeUnits() + "</sub>");
}
