#include <QDebug>

#include <QDir>
#include <QDesktopServices>

#include "config.h"
#include "ApplicationController.h"

#include <QGeoPositionInfoSource>
#include <QGeoSatelliteInfoSource>

class ApplicationControllerPrivate
{
public:
    ConfigManager   *config;
    PluginManager   *plugins;
    LoggingManager  *logger;

    TripModel       tripModel;

    QGeoPositionInfoSource  *positionSource;
    QGeoSatelliteInfoSource *satelliteSource;
};

ApplicationController::ApplicationController(QObject *parent)
    : QObject(parent)
{
    this->d = new ApplicationControllerPrivate;

    qDebug() << "ApplicationController:  Initializing configuration manager.";
    d->config = new ConfigManager(this);
    this->setupConfig();

    // Force loading of configuration data.
    qDebug() << "ApplicationController:  Loading application configuration.";
    this->onConfigUpdated(CONFIG_KEY_DISPLAY_POSITION_FORMAT, QVariant());
    this->onConfigUpdated(CONFIG_KEY_DISPLAY_UNITS, QVariant());

    qDebug() << "ApplicationController: Initializing logging manager.";
    d->logger = new LoggingManager(this);
    d->logger->initialize(this);

    qDebug() << "ApplicationController: Initializing plugin management system.";
    d->plugins = new PluginManager(this);
    d->plugins->initialize(this);

    QObject::connect(d->config,
                     SIGNAL(updated(QString,QVariant)),
                     this,
                     SLOT(onConfigUpdated(QString,QVariant)));

    qDebug() << "ApplicationController: Creating telemetry source objects.";
    d->positionSource = QGeoPositionInfoSource::createDefaultSource(this);
    d->satelliteSource = QGeoSatelliteInfoSource::createDefaultSource(this);

    qDebug() << "ApplicationController: Connecting telemetry source signals.";
    QObject::connect(d->positionSource,
                     SIGNAL(positionUpdated(QGeoPositionInfo)),
                     this,
                     SLOT(onPositionUpdated(QGeoPositionInfo)));

    QObject::connect(d->satelliteSource,
                     SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                     this,
                     SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)));

    QObject::connect(d->satelliteSource,
                     SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                     this,
                     SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)));

    qDebug() << "ApplicationController: Initiating GPS device.";
    d->positionSource->setUpdateInterval(1000);
    d->positionSource->startUpdates();
    d->satelliteSource->startUpdates();
}

ApplicationController::~ApplicationController()
{
    //   Need to destroy the plugin manager and its' plugins first, to
    // ensure resources it may rely on are still available.
    delete d->plugins;

    delete this->d;
}

TripModel* ApplicationController::tripModel() const {return &d->tripModel;}
ConfigManager*  ApplicationController::config() const {return d->config;}
PluginManager*  ApplicationController::plugins() const {return d->plugins;}
LoggingManager* ApplicationController::logger() const {return d->logger;}

void ApplicationController::setupConfig()
{
    this->config()->addToggle(CONFIG_KEY_DISPLAY_KEEP_ALIVE, tr("Keep-Alive Backlight"), true);

    this->config()->addOption(CONFIG_KEY_DISPLAY_ORIENTATION,
                              tr("Orientation"),
                              QStringList() << tr("Automatic") << tr("Landscape") << tr("Portrait"));

    // Aligned with TripModel::DisplayUnits enumeration.
    this->config()->addOption(CONFIG_KEY_DISPLAY_UNITS,
                              tr("Numerical Units"),
                              QStringList() << tr("Metric") << tr("Imperial") << tr("Nautical"));

    // Organised to be inline with QGeoCoordinate enumeration.
    this->config()->addOption(CONFIG_KEY_DISPLAY_POSITION_FORMAT,
                              tr("Position Format"),
                              QStringList()
                              << QString::fromUtf8("[-] dd.ddddd°")             // QGeoCoordinate::Degrees
                              << QString::fromUtf8("dd.ddddd° [N/S]")           // QGeoCoordinate::DegreesWithHemisphere
                              << QString::fromUtf8("[-] dd° mm.mmm'")           // QGeoCoordinate::DegreesMinutes
                              << QString::fromUtf8("dd° mm.mmm' [N/S]")         // QGeoCoordinate::DegreesMinutesWithHemisphere
                              << QString::fromUtf8("[-] dd° mm' ss.s\"")        // QGeoCoordinate::DegreesMinutesSeconds
                              << QString::fromUtf8("dd° mm' ss.s\" [N/S]"));    // QGeoCoordinate::DegreesMinutesSecondsWithHemisphere
}

void ApplicationController::onConfigUpdated(const QString &key, QVariant)
{
    if(key == CONFIG_KEY_DISPLAY_POSITION_FORMAT)
    {
        d->tripModel.setPositionFormat(((QGeoCoordinate::CoordinateFormat)d->config->valueIndex(CONFIG_KEY_DISPLAY_POSITION_FORMAT)));
    }
    else if(key == CONFIG_KEY_DISPLAY_UNITS)
    {
        d->tripModel.setDisplayUnits((TripModel::DisplayUnits)d->config->valueIndex(CONFIG_KEY_DISPLAY_UNITS));
    }
}

void ApplicationController::onPositionUpdated(const QGeoPositionInfo &position)
{
    QGeoPositionInfo nPosition(position);

    if(!position.isValid() || position.attribute(QGeoPositionInfo::HorizontalAccuracy) > 400.0f)
    {
        // This means we've not got a GPS fix or is so bad we might as well ignore it...
        return;
    }

// Hack for buggy Qt Mobility liblocation wrapper
#if defined(Q_WS_MAEMO_5) && QTM_VERSION <= 0x010100
    if(nPosition.hasAttribute(QGeoPositionInfo::GroundSpeed))
    {
        nPosition.setAttribute(QGeoPositionInfo::GroundSpeed, nPosition.attribute(QGeoPositionInfo::GroundSpeed) / 3.6);
    }

    if(nPosition.hasAttribute(QGeoPositionInfo::VerticalAccuracy))
    {
        nPosition.setAttribute(QGeoPositionInfo::VerticalAccuracy, nPosition.attribute(QGeoPositionInfo::VerticalAccuracy) / 3.6);
    }
#endif

    d->tripModel.onPositionUpdated(nPosition);

    emit this->positionUpdated(nPosition);
}
