#include <QPainter>
#include "SignalGraph.h"

class SignalGraphPrivate
{
public:
    QList<int>                      satsInUse;
    QHash<int,QGeoSatelliteInfo>    satsInView;
};

SignalGraph::SignalGraph(QWidget *parent)
    : AbstractChart(parent)
{
    this->d = new SignalGraphPrivate;

    QSizePolicy policy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    policy.setHeightForWidth(true);
    this->setSizePolicy(policy);

    this->setLabel(tr("Signal Graph (db)"));
}

SignalGraph::~SignalGraph()
{
    delete this->d;
}

void SignalGraph::setSatsInUse(const QList<QGeoSatelliteInfo> &sats)
{
    d->satsInUse.clear();

    foreach(QGeoSatelliteInfo satinfo, sats)
    {
        d->satsInUse.append(satinfo.prnNumber());
    }

    this->update();
}

void SignalGraph::setSatsInView(const QList<QGeoSatelliteInfo> &sats)
{
    d->satsInView.clear();

    foreach(QGeoSatelliteInfo satinfo, sats)
    {
        d->satsInView.insert(satinfo.prnNumber(), satinfo);
    }

    this->update();
}

void SignalGraph::paintEvent(QPaintEvent*)
{
    const QPen  pThin(this->palette().foreground(), 1.000f);

    const int   width   = this->width();
    int   height  = this->height();

    const float bAlloc  = width / 12.0f;
    const int   bPad    = 4;
    const int   bWidth  = bAlloc - bPad * 2;

    QColor fgColor = this->palette().foreground().color();
    fgColor.setAlphaF(0.3);

    QColor bgColor = this->palette().background().color();
    bgColor.setAlphaF(0.3);

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing, true);

    painter.setPen(QPen(QColor(0xff, 0xff, 0xff), 1.0f));
    painter.setFont(QFont(painter.font().family(), 10));
    QRect rect = painter.boundingRect(0, 0, 0, 0, 0, this->label());
    painter.drawText(width / 2 - (rect.width() / 2), height - rect.height(), rect.width(), rect.height(), 0, this->label());

    height = height - rect.height() - 20;

    QColor sColor = this->palette().foreground().color();
    painter.setPen(QPen(sColor, 0.5f));

    int m = 10;
    for(int i = 1; i < m; i++)
    {
        painter.drawLine(0, (height / m) * i + height % m, width, (height / m) * i  + height % m);
    }

    QFont font = painter.font();
    font.setPixelSize(16);

    QList<int> prns = d->satsInView.keys();
    qSort(prns);

    painter.setPen(pThin);
    painter.setFont(font);

    foreach(int prn, prns)
    {
        QGeoSatelliteInfo satinfo = d->satsInView.value(prn);
        int column = prns.indexOf(prn);

        painter.setMatrix(QMatrix());
        painter.setBrush(d->satsInUse.contains(prn) ? QBrush(fgColor) : QBrush(bgColor));
        painter.drawRect((bAlloc * column) + bPad, height, bWidth, -(satinfo.signalStrength() * 2));

        painter.setBrush(QBrush(fgColor));
        painter.setMatrix(QMatrix());
        QRect bRect = painter.boundingRect(bAlloc * column, height, 0, 0, 0, QString::number(prn));

        painter.translate((bAlloc - bRect.width()) / 2, 0);
        painter.drawText(bRect, QString::number(prn));
    }

    painter.setMatrix(QMatrix());
    painter.drawLine(0, height, bAlloc * 12, height);
}
