#include "LoggingManager.h"

#include "ApplicationController.h"

#include "data/NMEADataFormatter.h"

#include "data/FileDataLogger.h"
#include "data/FileDataLoggerConfig.h"

#include <QDebug>
#include <QtCore>
#include <QDesktopServices>

class LoggingManagerPrivate
{
public:
    ApplicationController *controller;

    QHash<QString,AbstractDataFormatter*>               formatters;
    QHash<AbstractDataLogger*,AbstractDataFormatter*>   loggers;

    AbstractDataLogger *defaultLogger;
};

LoggingManager::LoggingManager(QObject *parent)
    : QObject(parent)
{
    this->d = new LoggingManagerPrivate;
    d->controller       = NULL;
    d->defaultLogger    = NULL;

    this->addDataFormat(new NMEADataFormatter(this));
}

LoggingManager::~LoggingManager()
{
    delete this->d;
}

QHash<QString,AbstractDataFormatter*> LoggingManager::formatters () const
{
    return d->formatters;
}

QHash<AbstractDataLogger*,AbstractDataFormatter*> LoggingManager::loggers () const
{
    return d->loggers;
}

void LoggingManager::initialize(ApplicationController *controller)
{
    d->controller = controller;

    // If we're configured to do file logging, do it!
    if(FileDataLoggerConfig().isEnabled() && FileDataLoggerConfig().isRunOnStartup())
    {
        this->setupLogging();
    }
    else
    {
        FileDataLoggerConfig().setEnabled(false);
    }
}

void LoggingManager::setupLogging()
{
    FileDataLoggerConfig config;

    if(d->defaultLogger != NULL)
    {
        qDebug() << "LoggingManager: Destroying redundant logger.";
        this->removeLogger(d->defaultLogger);
        delete d->defaultLogger;
        d->defaultLogger = NULL;
    }

    if(config.isEnabled())
    {
        qDebug() << "LoggingManager: Setting up data logging.";

        AbstractDataFormatter *formatter = this->formatters().value(config.formatId());

        const QString logdir = QDesktopServices::storageLocation(QDesktopServices::CacheLocation);
        const QString filename = QString("yyyy-MM-dd-%1").arg(config.formatId());

        if(!QDir().exists(logdir))
        {
            qDebug() << "LoggingManager: Creating log directory:" << logdir;
            QDir().mkpath(logdir);
        }

        d->defaultLogger = new FileDataLogger(logdir, filename, this);
        this->addLogger(formatter, d->defaultLogger);
    }
}

void LoggingManager::addDataFormat(AbstractDataFormatter *formatter)
{
    d->formatters.insert(formatter->id(), formatter);
}

void LoggingManager::addLogger(AbstractDataFormatter *formatter, AbstractDataLogger *logger)
{
    QObject::connect(formatter, SIGNAL(dataReady(QByteArray)), logger, SLOT(onDataReady(QByteArray)));

    // If this formatter isn't in the active loggers repository, initialize is for receiving telemetry.
    if(!d->loggers.values().contains(formatter))
    {
        qDebug() << "LoggingManager: Connecting formatter" << formatter->id() << "to telemetry sources.";
        QObject::connect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)), formatter, SLOT(onPositionUpdate(QGeoPositionInfo)));
        QObject::connect(d->controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)), formatter, SLOT(onSatellitesInUse(QList<QGeoSatelliteInfo>)));
        QObject::connect(d->controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)), formatter, SLOT(onSatellitesInView(QList<QGeoSatelliteInfo>)));
    }

    d->loggers.insert(logger, formatter);
}

void LoggingManager::removeLogger(AbstractDataLogger *logger)
{
    AbstractDataFormatter *formatter = d->loggers.value(logger);

    QObject::disconnect(formatter, SIGNAL(dataReady(QByteArray)), logger, SLOT(onDataReady(QByteArray)));

    d->loggers.remove(logger);

    // If this is the last reference, disconnect telemetry sources.
    if(!d->loggers.values().contains(formatter))
    {
        qDebug() << "LoggingManager: Disconnecting formatter" << formatter->id() << "from telemetry sources.";
        QObject::disconnect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)), formatter, SLOT(onPositionUpdate(QGeoPositionInfo)));
        QObject::disconnect(d->controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)), formatter, SLOT(onSatellitesInUse(QList<QGeoSatelliteInfo>)));
        QObject::disconnect(d->controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)), formatter, SLOT(onSatellitesInView(QList<QGeoSatelliteInfo>)));
    }
}
