#include "NetworkServerConfig.h"

#include <QSettings>

#define CONFIG_KEY_NETWORK_SERVICE_IS_ENABLED               "/plugins/network-server/enable"
#define CONFIG_KEY_NETWORK_SERVICE_IS_RUN_ON_STARTUP        "/plugins/network-server/startup"
#define CONFIG_KEY_NETWORK_SERVICE_TRANSPORT                "/plugins/network-server/transport"
#define CONFIG_KEY_NETWORK_SERVICE_FORMAT                   "/plugins/network-server/format"
#define CONFIG_KEY_NETWORK_SERVICE_UDP_CONNECTION_TYPE      "/plugins/network-server/udp/type"
#define CONFIG_KEY_NETWORK_SERVICE_UDP_UNICAST_ADDRESS      "/plugins/network-server/udp/unicast/address"
#define CONFIG_KEY_NETWORK_SERVICE_UDP_UNICAST_PORT         "/plugins/network-server/udp/unicast/port"
#define CONFIG_KEY_NETWORK_SERVICE_UDP_MULTICAST_ADDRESS    "/plugins/network-server/udp/multicast/address"
#define CONFIG_KEY_NETWORK_SERVICE_UDP_MULTICAST_PORT       "/plugins/network-server/udp/multicast/port"
#define CONFIG_KEY_NETWORK_SERVICE_TCP_CONNECTION_TYPE      "/plugins/network-server/tcp/type"
#define CONFIG_KEY_NETWORK_SERVICE_TCP_CLIENT_ADDRESS       "/plugins/network-server/tcp/client/address"
#define CONFIG_KEY_NETWORK_SERVICE_TCP_CLIENT_PORT          "/plugins/network-server/tcp/client/port"
#define CONFIG_KEY_NETWORK_SERVICE_TCP_SERVER_ADDRESS       "/plugins/network-server/tcp/server/address"
#define CONFIG_KEY_NETWORK_SERVICE_TCP_SERVER_PORT          "/plugins/network-server/tcp/server/port"

class NetworkServerConfigPrivate
{
public:
    bool isEnabled;
    bool isRunOnStartup;
    NetworkServerConfig::Transport transport;

    NetworkServerConfig::UdpConnectionType udpConnectionType;
    QString udpUnicastAddress;
    quint16 udpUnicastPort;
    QString udpMulticastAddress;
    quint16 udpMulticastPort;

    NetworkServerConfig::TcpConnectionType tcpConnectionType;
    QString tcpClientAddress;
    quint16 tcpClientPort;
    QString tcpServerAddress;
    quint16 tcpServerPort;

    QString formatId;
};

NetworkServerConfig::NetworkServerConfig(QObject *parent)
    : QObject(parent)
{
    this->d = new NetworkServerConfigPrivate;
}

NetworkServerConfig::~NetworkServerConfig()
{
    delete this->d;
}

bool NetworkServerConfig::operator == (const NetworkServerConfig &other)
{
    if(this->isEnabled() != other.isEnabled() ||
       this->transport() != other.transport() ||
       this->formatId() != other.formatId())
    {
        return false;
    }

    if(this->transport() == TransportUdp)
    {
        if(this->udpConnectionType() != other.udpConnectionType()) return false;

        switch(this->udpConnectionType())
        {
        case UdpUnicast:
            if(this->udpUnicastAddress() != other.udpUnicastAddress() ||
               this->udpUnicastPort() != other.udpUnicastPort())
            {
                return false;
            }
            break;

        case UdpMulticast:
            if(this->udpMulticastAddress() != other.udpMulticastAddress() ||
               this->udpMulticastPort() != other.udpMulticastPort())
            {
                return false;
            }
            break;
        }
    }
    else if(this->transport() == TransportTcp)
    {
        if(this->tcpConnectionType() != other.tcpConnectionType()) return false;

        switch(this->tcpConnectionType())
        {
        case TcpClient:
            if(this->tcpClientAddress() != other.tcpClientAddress() ||
               this->tcpClientPort() != other.tcpClientPort())
            {
                return false;
            }
            break;

        case TcpServer:
            if(this->tcpServerAddress() != other.tcpServerAddress() ||
               this->tcpServerPort() != other.tcpServerPort())
            {
                return false;
            }
            break;
        }
    }

    return true;
}

bool NetworkServerConfig::operator != (const NetworkServerConfig &other)
{
    return !(*this == other);
}

void NetworkServerConfig::load()
{
    d->isEnabled = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_IS_ENABLED, false).toBool();
    d->isRunOnStartup = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_IS_RUN_ON_STARTUP, false).toBool();

    d->transport = (NetworkServerConfig::Transport)QSettings().value(CONFIG_KEY_NETWORK_SERVICE_TRANSPORT, (int)NetworkServerConfig::TransportTcp).toInt();

    d->udpConnectionType = (NetworkServerConfig::UdpConnectionType)QSettings().value(CONFIG_KEY_NETWORK_SERVICE_UDP_CONNECTION_TYPE, (int)NetworkServerConfig::UdpUnicast).toInt();
    d->udpUnicastAddress = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_UDP_UNICAST_ADDRESS, "127.0.0.1").toString();
    d->udpUnicastPort = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_UDP_UNICAST_PORT, 9300).toInt();
    d->udpMulticastAddress = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_UDP_MULTICAST_ADDRESS, "127.0.0.1").toString();
    d->udpMulticastPort = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_UDP_MULTICAST_PORT, 9300).toInt();

    d->tcpConnectionType = (NetworkServerConfig::TcpConnectionType)QSettings().value(CONFIG_KEY_NETWORK_SERVICE_TCP_CONNECTION_TYPE, (int)NetworkServerConfig::TcpClient).toInt();
    d->tcpClientAddress = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_TCP_CLIENT_ADDRESS, "127.0.0.1").toString();
    d->tcpClientPort = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_TCP_CLIENT_PORT, 9300).toInt();
    d->tcpServerAddress = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_TCP_SERVER_ADDRESS, "127.0.0.1").toString();
    d->tcpServerPort = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_TCP_SERVER_PORT, 9300).toInt();

    d->formatId = QSettings().value(CONFIG_KEY_NETWORK_SERVICE_FORMAT, "nmea").toString();
}

void NetworkServerConfig::save()
{
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_IS_ENABLED, d->isEnabled);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_IS_RUN_ON_STARTUP, d->isRunOnStartup);

    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_TRANSPORT, d->transport);

    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_UDP_CONNECTION_TYPE, d->udpConnectionType);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_UDP_UNICAST_ADDRESS, d->udpUnicastAddress);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_UDP_UNICAST_PORT, d->udpUnicastPort);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_UDP_MULTICAST_ADDRESS, d->udpMulticastAddress);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_UDP_MULTICAST_PORT, d->udpMulticastPort);

    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_TCP_CONNECTION_TYPE, d->tcpConnectionType);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_TCP_CLIENT_ADDRESS, d->tcpClientAddress);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_TCP_CLIENT_PORT, d->tcpClientPort);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_TCP_SERVER_ADDRESS, d->tcpServerAddress);
    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_TCP_SERVER_PORT, d->tcpServerPort);

    QSettings().setValue(CONFIG_KEY_NETWORK_SERVICE_FORMAT, d->formatId);

}

bool NetworkServerConfig::isEnabled() const
{
    return d->isEnabled;
}

bool NetworkServerConfig::isRunOnStartup() const
{
    return d->isRunOnStartup;
}

NetworkServerConfig::Transport NetworkServerConfig::transport() const
{
    return d->transport;
}

NetworkServerConfig::UdpConnectionType NetworkServerConfig::udpConnectionType() const
{
    return d->udpConnectionType;
}

QString NetworkServerConfig::udpUnicastAddress() const
{
    return d->udpUnicastAddress;
}

quint16 NetworkServerConfig::udpUnicastPort() const
{
    return d->udpUnicastPort;
}

QString NetworkServerConfig::udpMulticastAddress() const
{
    return d->udpMulticastAddress;
}

quint16 NetworkServerConfig::udpMulticastPort() const
{
    return d->udpMulticastPort;
}

NetworkServerConfig::TcpConnectionType NetworkServerConfig::tcpConnectionType() const
{
    return d->tcpConnectionType;
}

QString NetworkServerConfig::tcpClientAddress() const
{
    return d->tcpClientAddress;
}

quint16 NetworkServerConfig::tcpClientPort() const
{
    return d->tcpClientPort;
}

QString NetworkServerConfig::tcpServerAddress() const
{
    return d->tcpServerAddress;
}

quint16 NetworkServerConfig::tcpServerPort() const
{
    return d->tcpServerPort;
}

QString NetworkServerConfig::formatId() const
{
    return d->formatId;
}

void NetworkServerConfig::setEnabled(bool isEnabled)
{
    d->isEnabled = isEnabled;
}

void NetworkServerConfig::setRunOnStartup(bool isRunOnStartup)
{
    d->isRunOnStartup = isRunOnStartup;
}

void NetworkServerConfig::setTransport(Transport transport)
{
    d->transport = transport;
}

void NetworkServerConfig::setUdpConnectionType(UdpConnectionType connectionType)
{
    d->udpConnectionType = connectionType;
}

void NetworkServerConfig::setUdpUnicastAddress(const QString &address)
{
    d->udpUnicastAddress = address;
}

void NetworkServerConfig::setUdpUnicastPort(quint16 port)
{
    d->udpUnicastPort = port;
}

void NetworkServerConfig::setUdpMulticastAddress(const QString &address)
{
    d->udpMulticastAddress = address;
}

void NetworkServerConfig::setUdpMulticastPort(quint16 port)
{
    d->udpMulticastPort = port;
}

void NetworkServerConfig::setTcpConnectionType(TcpConnectionType connectionType)
{
    d->tcpConnectionType = connectionType;
}

void NetworkServerConfig::setTcpClientAddress(const QString &address)
{
    d->tcpClientAddress = address;
}

void NetworkServerConfig::setTcpClientPort(quint16 port)
{
    d->tcpClientPort = port;
}

void NetworkServerConfig::setTcpServerAddress(const QString &address)
{
    d->tcpServerAddress = address;
}

void NetworkServerConfig::setTcpServerPort(quint16 port)
{
    d->tcpServerPort = port;
}

void NetworkServerConfig::setFormatId(const QString &formatId)
{
    d->formatId = formatId;
}
