#include "BluetoothSocket.h"
#include "BluetoothAcceptor.h"

#include <QDebug>
#include <QSocketNotifier>

#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

class ServiceAcceptorPrivate
{
public:
    quint16 channel;

    sdp_record_t  *record;
    sdp_session_t *sdp;

    BluetoothSocket *server;
    BluetoothSocket *connection;

    ServicePerformer *performer;
    ServicePerformerFactory *factory;
};

BluetoothAcceptor::BluetoothAcceptor(quint16 channel, ServicePerformerFactory *factory, QObject *parent)
    : BluetoothServiceProvider(parent)
{
    qDebug() << "BluetoothAcceptor: ctor()";

    this->d = new ServiceAcceptorPrivate;
    d->factory      = factory;
    d->channel      = channel;
    d->server       = NULL;
    d->record       = NULL;
    d->sdp          = NULL;
    d->connection   = NULL;
    d->performer    = NULL;
}

BluetoothAcceptor::~BluetoothAcceptor()
{
    qDebug() << "BluetoothAcceptor: dtor()";

    if(d->connection != NULL)
    {
        qDebug() << "BluetoothAcceptor: Disconnecting from client.";
        d->connection->close();
    }

    if(d->sdp != NULL)
    {
        qDebug() << "BluetoothAcceptor: Closing connection to SDP daemon.";
        if(d->record != NULL)
        {
            sdp_record_unregister(d->sdp, d->record);
        }
        sdp_close(d->sdp);
        d->sdp = NULL;
    }

    if(d->record != NULL)
    {
        qDebug() << "BluetoothAcceptor: Unregistering SDP service record.";
        sdp_record_free(d->record);
        d->record = NULL;
    }

    if(d->server != NULL)
    {
        qDebug() << "BluetoothAcceptor: Closing server socket.";
        delete d->server;
        d->server = NULL;
    }

    delete this->d;
}

bool BluetoothAcceptor::initialize()
{
    d->server = new BluetoothSocket(this);

    QObject::connect(d->server, SIGNAL(newConnection()), this, SLOT(onConnect()));

    if(!d->server->listen("00:00:00:00:00:00", d->channel))
    {
        qDebug() << "BluetoothAcceptor: Failed to create server socket!";
        emit this->statusChanged("Failed to start");
        return false;
    }

    d->channel = d->server->channel();

    qDebug() << "BluetoothAcceptor: Socket successfully bound to channel:" << d->channel;

    uint8_t service_uuid_int[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ,0 ,0 , 0, 0, 0x11, 0x01};

    const char *svc_name = "Columbus";
    const char *svc_desc = "Columbus Navigation Telemetry.";
    const char *svc_prov = "rubyx.co.uk";

    uuid_t root_uuid, l2cap_uuid, rfcomm_uuid, svc_uuid, svc_class_uuid, profile_uuid;

    sdp_data_t *channel;
    sdp_profile_desc_t profile;
    sdp_list_t *l2cap_list, *rfcomm_list, *root_list, *proto_list, *access_list, *svc_class_list, *profile_list;

    qDebug() << "BluetoothAcceptor: Creating SDP service record.";
    d->record = sdp_record_alloc();

    if(d->record == NULL)
    {
        qWarning() << "Failed to allocate SDP service record!";
        emit this->statusChanged("Failed to start");
        return false;
    }

    // Set service UUID.
    qDebug() << "BluetoothAcceptor: Setting service UUID";
    sdp_uuid128_create(&svc_uuid, &service_uuid_int);
    sdp_set_service_id(d->record, svc_uuid);

    // Set service class.
    qDebug() << "BluetoothAcceptor: Setting service class";
    sdp_uuid16_create(&svc_class_uuid, SERIAL_PORT_SVCLASS_ID);
    svc_class_list = sdp_list_append(0, &svc_class_uuid);
    sdp_set_service_classes(d->record, svc_class_list);

    // Set service bluetooth profile.
    qDebug() << "BluetoothAcceptor: Setting service profile";
    sdp_uuid16_create(&profile_uuid, SERIAL_PORT_PROFILE_ID);
    profile.uuid = profile_uuid;
    profile.version = 0x0100;
    profile_list = sdp_list_append(0, &profile);
    sdp_set_profile_descs(d->record, profile_list);

    // Set root service accessibility.
    qDebug() << "BluetoothAcceptor: Setting service root accessibility group.";
    sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
    root_list = sdp_list_append(0, &root_uuid);
    sdp_set_browse_groups(d->record, root_list);

    // Set L2CAP information.
    qDebug() << "BluetoothAcceptor: Setting L2CAP information.";
    sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
    l2cap_list = sdp_list_append(0, &l2cap_uuid);
    proto_list = sdp_list_append(0, l2cap_list);

    // Register RFCOMM channel.
    qDebug() << "BluetoothAcceptor: Setting RFCOMM channel information.";
    sdp_uuid16_create(&rfcomm_uuid, RFCOMM_UUID);
    channel = sdp_data_alloc(SDP_UINT8, &d->channel);
    rfcomm_list = sdp_list_append(0, &rfcomm_uuid);
    sdp_list_append(rfcomm_list, channel);
    sdp_list_append(proto_list, rfcomm_list);

    // Register SDP service provider information.
    qDebug() << "BluetoothAcceptor: Setting service provider information.";
    access_list = sdp_list_append(0, proto_list);
    sdp_set_access_protos(d->record, access_list);
    sdp_set_info_attr(d->record, svc_name, svc_prov, svc_desc);

    // Register our service!
    qDebug() << "BluetoothAcceptor: Connecting to local SDP service.";
    d->sdp = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, SDP_RETRY_IF_BUSY);

    if(d->sdp == NULL)
    {
        qDebug() << "BluetoothAcceptor: Failed to connect to local SDP service.";
        emit this->statusChanged(tr("Failed to start"));
    }
    else
    {
        qDebug() << "BluetoothAcceptor: Registering service record with local SDP service.";
        sdp_record_register(d->sdp, d->record, 0);

        qDebug() << "BluetoothAcceptor: Notifying upper layer.";
        emit this->statusChanged(tr("Listening on channel %1").arg(d->channel));
    }

    // Clean up.
    qDebug() << "BluetoothAcceptor: Cleaning up temporary initialization data.";
    sdp_data_free(channel);
    sdp_list_free(l2cap_list, 0);
    sdp_list_free(rfcomm_list, 0);
    sdp_list_free(root_list, 0);
    sdp_list_free(proto_list, 0);
    sdp_list_free(access_list, 0);
    sdp_list_free(svc_class_list, 0);
    sdp_list_free(profile_list, 0);

    return true;
}

void BluetoothAcceptor::onConnect()
{
    d->connection = d->server->nextPendingConnection();

    qDebug() << "BluetoothAcceptor: Remote endpoint" << d->connection->peerAddress() << "connected, creating transport handle.";
    QObject::connect(d->connection, SIGNAL(disconnected()), this, SLOT(onDisconnect()));

    qDebug() << "BluetoothAcceptor: Creating service performer.";
    d->performer = d->factory->createInstance(d->connection, this);

    qDebug() << "BluetoothAcceptor: Notifying upper layer.";
    emit this->statusChanged(tr("Connected to %1 (Channel %2)").arg(d->connection->peerAddress(), QString::number(d->channel)));
}

void BluetoothAcceptor::onDisconnect()
{
    qDebug() << "BluetoothAcceptor: Shutting down transport device.";
    QObject::disconnect(d->connection, SIGNAL(disconnected()), this, SLOT(onDisconnect()));

    if(d->performer != NULL)
    {
        qDebug() << "BluetoothAcceptor: Remote endpoint disconnected, destroying performer.";
        d->factory->destroyInstance(d->performer);
        d->performer = NULL;
    }

    if(d->connection != NULL)
    {
        qDebug() << "BluetoothAcceptor: Destroying transport.";
        delete d->connection;
        d->connection = NULL;
    }

    qDebug() << "BluetoothAcceptor: Notifying upper layer.";
    emit this->statusChanged(tr("Listening on channel %1").arg(d->channel));
}
