#include "BluetoothSocket.h"

#include <QDebug>
#include <QAbstractSocket>
#include <QSocketNotifier>

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>

#include <sys/socket.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>

class BluetoothSocketPrivate
{
public:
    int fd;

    quint16 channel; // Channel we're currently listening on.

    QSocketNotifier *rNotifier;
    QSocketNotifier *wNotifier;

    QAbstractSocket::SocketState state;
};

BluetoothSocket::BluetoothSocket(QObject *parent)
    : QIODevice(parent)
{
    this->d = new BluetoothSocketPrivate;
    d->fd = -1;
    d->rNotifier = NULL;
    d->wNotifier = NULL;
    d->state = QAbstractSocket::UnconnectedState;

    this->setOpenMode(QIODevice::WriteOnly);
}

BluetoothSocket::BluetoothSocket(int fd, QObject *parent)
    : QIODevice(parent)
{
    this->d = new BluetoothSocketPrivate;
    d->fd = fd;
    d->wNotifier = NULL;

    int flags = ::fcntl(d->fd, F_GETFL, NULL);
    ::fcntl(d->fd, F_SETFL, flags | O_NONBLOCK);

    d->rNotifier = new QSocketNotifier(fd, QSocketNotifier::Read, this);
    QObject::connect(d->rNotifier, SIGNAL(activated(int)), this, SLOT(onReadNotify()));

    d->state = QAbstractSocket::ConnectedState;
    this->setOpenMode(QIODevice::WriteOnly);
}

BluetoothSocket::~BluetoothSocket()
{
    this->close();
    delete this->d;
}

quint16 BluetoothSocket::channel() const
{
    return d->channel;
}

QString BluetoothSocket::peerAddress() const
{
    struct sockaddr_rc addr;
    socklen_t addrlen = sizeof(addr);
    char buffer[18];

    if(::getpeername(d->fd, (struct sockaddr*)&addr, &addrlen) != 0)
    {
        return QString();
    }

    ::ba2str(&addr.rc_bdaddr, buffer);
    return QString(buffer);
}

quint16 BluetoothSocket::peerChannel() const
{
    struct sockaddr_rc addr;
    socklen_t addrlen = sizeof(addr);

    if(::getpeername(d->fd, (struct sockaddr*)&addr, &addrlen) != 0)
    {
        return -1;
    }

    return addr.rc_channel;
}

void BluetoothSocket::close()
{
    QIODevice::close();

    if(d->wNotifier != NULL)
    {
        delete d->wNotifier;
        d->wNotifier = NULL;
    }

    if(d->rNotifier != NULL)
    {
        delete d->rNotifier;
        d->rNotifier = NULL;
    }

    if(d->fd != -1)
    {
        ::close(d->fd);
        d->fd = -1;
    }

    d->state = QAbstractSocket::UnconnectedState;
}

bool BluetoothSocket::createSocket()
{
    if(d->fd != -1)
    {
        this->close();
    }

    d->fd = ::socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);

    if(d->fd == -1)
    {
        qWarning() << "Failed to create socket!";
        return false;
    }

    d->rNotifier = new QSocketNotifier(d->fd, QSocketNotifier::Read, this);
    QObject::connect(d->rNotifier, SIGNAL(activated(int)), this, SLOT(onReadNotify()));

    return true;
}

bool BluetoothSocket::listen(quint16 channel)
{
    return this->listen("00:00:00:00:00:00", channel);
}

bool BluetoothSocket::listen(const QString &bdaddr, quint16 channel)
{
    struct sockaddr_rc addr;
    addr.rc_family = AF_BLUETOOTH;
    ::str2ba(bdaddr.toAscii().constData(), &addr.rc_bdaddr);

    if(!this->createSocket()) return false;

    int flags = ::fcntl(d->fd, F_GETFL, NULL);
    if(::fcntl(d->fd, F_SETFL, flags | O_NONBLOCK) == -1)
    {
        qWarning() << "Failed to set flags on socket!";
        return false;
    }

    if(channel == 0)
    {
        for(quint16 i = 1; i < 31; i++)
        {
            addr.rc_channel = i;

            if(::bind(d->fd, (struct sockaddr*)&addr, sizeof(addr)) == 0)
            {
                break;
            }
        }

        if(addr.rc_channel == 31)
        {
            this->close();
            return false;
        }
    }
    else
    {
        addr.rc_channel = channel;

        if(::bind(d->fd, (struct sockaddr*)&addr, sizeof(addr)) != 0)
        {
            this->close();
            return false;
        }
    }

    d->channel = addr.rc_channel;

    if(::listen(d->fd, 1) == -1)
    {
        return false;
    }

    d->state = QAbstractSocket::ListeningState;
    return true;
}

bool BluetoothSocket::connectToHost(const QString &bdaddr, quint16 channel)
{
    struct sockaddr_rc addr;
    addr.rc_family = AF_BLUETOOTH;
    addr.rc_channel = channel;
    ::str2ba(bdaddr.toAscii().constData(), &addr.rc_bdaddr);

    if(!this->createSocket()) return false;

    qDebug() << "Attempting to connect to" << bdaddr << "channel" << channel;
    if(::connect(d->fd, (struct sockaddr*)&addr, sizeof(addr)) != 0)
    {
        perror("Failed to connect");
        this->close();
        return false;
    }

    int flags = ::fcntl(d->fd, F_GETFL, NULL);
    if(::fcntl(d->fd, F_SETFL, flags | O_NONBLOCK) == -1)
    {
        qWarning() << "Failed to set flags on socket!";
        return false;
    }

    d->wNotifier = new QSocketNotifier(d->fd, QSocketNotifier::Write, this);
    QObject::connect(d->wNotifier, SIGNAL(activated(int)), this, SLOT(onWriteNotify()));

    d->state = QAbstractSocket::ConnectingState;
    return true;
}

void BluetoothSocket::onReadNotify()
{
    char buffer[16387];
    int bRead = ::read(d->fd, buffer, 16387);

    switch(d->state)
    {
    case QAbstractSocket::ListeningState:
        emit this->newConnection();
        break;

    case QAbstractSocket::ConnectedState:
        if(bRead <= 0) this->close();
        emit this->disconnected();
        break;

    default:
        break;
    }
}

void BluetoothSocket::onWriteNotify()
{
    delete d->wNotifier;
    d->wNotifier = NULL;

    switch(d->state)
    {
    case QAbstractSocket::ConnectingState:
        d->state = QAbstractSocket::ConnectedState;
        emit this->connected();
        break;

    default:
        break;
    }
}

BluetoothSocket* BluetoothSocket::nextPendingConnection()
{
    struct sockaddr_rc addr;
    socklen_t addrlen = sizeof(addr);
    return new BluetoothSocket(::accept(d->fd, (struct sockaddr*)&addr, &addrlen), this);
}

qint64 BluetoothSocket::readData(char*, qint64) {return 0;}

qint64 BluetoothSocket::writeData(const char *data, qint64 maxlen)
{
    return ::write(d->fd, data, maxlen);
}
