#include "BluetoothOptionsDialog.h"

#include <QtGui>
#include <QtMaemo5>

class BluetoothOptionsDialogPrivate
{
public:
    LoggingManager *logger;
    BluetoothServerConfig *config;

    QScrollArea *scrollarea;

    QCheckBox *isEnabled;
    QCheckBox *runOnStartup;

    QMaemo5ValueButton *mode;
    QMaemo5ListPickSelector *modeSelector;

    QMaemo5ValueButton *channel;
    QMaemo5ListPickSelector *channelSelector;

    QMaemo5ValueButton *format;
    QMaemo5ListPickSelector *formatSelector;

    QLineEdit *bdaddr;
};

BluetoothOptionsDialog::BluetoothOptionsDialog(BluetoothServerConfig *config, LoggingManager *manager, QWidget *parent)
    : Maemo5Dialog(parent)
{
    new QVBoxLayout(this);

    this->d = new BluetoothOptionsDialogPrivate;

    d->config = config;
    d->logger = manager;

    this->setWindowTitle(tr("Bluetooth"));

    this->setupUi();
    this->restoreConfig();
    this->updateUi();

    QObject::connect(this, SIGNAL(finished(int)), this, SLOT(updateConfig()));
}

BluetoothOptionsDialog::~BluetoothOptionsDialog()
{
    delete this->d;
}

void BluetoothOptionsDialog::setupUi()
{
    QStringList fStrings;

    foreach(AbstractLoggingFormatter *formatter, d->logger->formatters())
    {
        fStrings.append(formatter->friendlyName());
    }

    qSort(fStrings);

    QStringListModel *mModel = new QStringListModel(QStringList()
                                                           << tr("Master")
                                                           << tr("Slave"),
                                                           this);

    QStringList cStrings = QStringList() << tr("Auto");

    for(int i = 1; i <= 30; i++)
    {
        cStrings << QString::number(i);
    }

    QStringListModel *cModel = new QStringListModel(cStrings, this);

    QStringListModel *formatModel = new QStringListModel(fStrings, this);

    d->scrollarea = new QScrollArea(this);
    d->scrollarea->setWidgetResizable(true);
    d->scrollarea->setWidget(new QWidget(this));

    QVBoxLayout *vbox = new QVBoxLayout(d->scrollarea->widget());
    QHBoxLayout *hbox = new QHBoxLayout();

    d->modeSelector = new QMaemo5ListPickSelector(this);
    d->modeSelector->setModel(mModel);

    d->channelSelector = new QMaemo5ListPickSelector(this);
    d->channelSelector->setModel(cModel);

    d->formatSelector = new QMaemo5ListPickSelector(this);
    d->formatSelector->setModel(formatModel);

    d->isEnabled = new QCheckBox(tr("Enabled"), this);
    d->runOnStartup = new QCheckBox(tr("Run on Startup"), this);

    d->mode = new QMaemo5ValueButton(tr("Mode"), this);
    //d->mode->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->mode->setPickSelector(d->modeSelector);

    d->channel = new QMaemo5ValueButton(tr("Channel"), this);
    //d->channel->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->channel->setPickSelector(d->channelSelector);

    d->bdaddr = new QLineEdit(this);
    d->bdaddr->setPlaceholderText(tr("Enter Bluetooth Address"));

    d->format = new QMaemo5ValueButton(tr("Format"), this);
    //d->format->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->format->setPickSelector(d->formatSelector);

    vbox->addWidget(d->isEnabled);
    vbox->addWidget(d->runOnStartup);

    hbox->addWidget(d->mode);
    hbox->addWidget(d->channel);

    vbox->addLayout(hbox);
    vbox->addWidget(d->bdaddr);

    vbox->addWidget(d->format);

    this->layout()->addWidget(d->scrollarea);

    QObject::connect(d->isEnabled, SIGNAL(toggled(bool)), this, SLOT(updateUi()));
    QObject::connect(d->modeSelector, SIGNAL(selected(QString)), this, SLOT(updateUi()));
}

void BluetoothOptionsDialog::updateUi()
{
    qDebug() << "BluetoothOptionsDialog: Performing UI update.";

    d->runOnStartup->setEnabled(d->isEnabled->isChecked());
    d->mode->setEnabled(d->isEnabled->isChecked());
    d->channel->setEnabled(d->isEnabled->isChecked());

    if(d->mode->valueText().toLower() == "slave")
    {
        d->bdaddr->setEnabled(d->isEnabled->isChecked());
        d->bdaddr->setText(d->config->peerAddress());
    }
    else
    {
        d->bdaddr->setEnabled(false);
        d->bdaddr->setText("");
    }

    d->format->setEnabled(d->isEnabled->isChecked());
}

void BluetoothOptionsDialog::restoreConfig()
{
    qDebug() << "BluetoothOptionsDialog: Restoring configuration.";
    d->isEnabled->setChecked(d->config->isEnabled() && d->config->isRunOnStartup());
    d->runOnStartup->setChecked(d->config->isRunOnStartup());
    d->modeSelector->setCurrentIndex((int)d->config->mode());
    d->channelSelector->setCurrentIndex(d->config->channel());
    d->formatSelector->setCurrentIndex(0);
}

void BluetoothOptionsDialog::updateConfig()
{
    qDebug() << "BluetoothOptionsDialog: Updating configuration.";
    d->config->setEnabled(d->isEnabled->isChecked());
    d->config->setRunOnStartup(d->runOnStartup->isChecked());
    d->config->setMode((BluetoothServerConfig::OperatingMode)d->modeSelector->currentIndex());
    d->config->setChannel(d->channelSelector->currentIndex());

    if(d->config->mode() == BluetoothServerConfig::SlaveMode)
    {
        d->config->setPeerAddress(d->bdaddr->text());
    }

    foreach(AbstractLoggingFormatter *formatter, d->logger->formatters().values())
    {
        if(formatter->friendlyName() == d->format->valueText())
        {
            d->config->setFormatId(formatter->id());
            break;
        }
    }
}
