#include "SerialPort.h"

#include <fcntl.h>
#include <termios.h>

#include <QDebug>
#include <QSocketNotifier>

class SerialPortPrivate
{
public:
    QString deviceName;

    struct termios prevAttrs;
    struct termios portAttrs;

    int fd;
    QSocketNotifier *notifier;
};

SerialPort::SerialPort(QObject *parent)
    : QIODevice(parent)
{
    this->d = new SerialPortPrivate;
    d->deviceName = "/dev/ttyS0";
    d->fd = -1;
    d->notifier = NULL;
}

SerialPort::~SerialPort()
{
    if(d->fd >= 0)
    {
        this->close();
        d->fd = -1;
    }

    delete this->d;
}

QString SerialPort::deviceName() const
{
    return d->deviceName;
}

bool SerialPort::open(const QString &deviceName, OpenMode mode)
{
    if(d->fd != -1)
    {
        this->close();
    }

    this->setOpenMode(mode);

    d->fd = ::open(deviceName.toAscii().constData(), O_RDWR | O_NOCTTY | O_NONBLOCK);
    if(d->fd < 0)
    {
        qDebug() << "SerialPort: Failed to open serial device" << deviceName;
        ::perror("Error:");
        return false;
    }

    d->deviceName = deviceName;

    tcgetattr(d->fd, &d->prevAttrs);
    bzero(&d->portAttrs, sizeof(d->portAttrs));

    d->portAttrs.c_cflag = B9600 | CRTSCTS | CS8 | CLOCAL | CREAD;
    d->portAttrs.c_iflag = IGNPAR | ICRNL;
    d->portAttrs.c_oflag = 0;
    d->portAttrs.c_lflag = ICANON;

    tcflush(d->fd, TCIFLUSH);
    tcsetattr(d->fd, TCSANOW, &d->portAttrs);

    return true;
}

void SerialPort::close()
{
    if(d->notifier != NULL)
    {
        delete d->notifier;
        d->notifier = NULL;
    }

    if(d->fd >= 0)
    {
        tcsetattr(d->fd, TCSANOW, &d->prevAttrs);
        ::close(d->fd);
        d->fd = -1;
    }
}

qint64 SerialPort::readData(char *data, qint64 maxlen)
{
    return ::read(d->fd, data, maxlen);
}

qint64 SerialPort::writeData(const char *data, qint64 len)
{
    return ::write(d->fd, data, len);
}
