#include "TCPNetworkAcceptor.h"

#include <QDebug>
#include <QHash>
#include <QTcpSocket>
#include <QTcpServer>

class TCPNetworkAcceptorPrivate
{
public:
    QString address;
    quint16 port;

    QTcpServer *server;
    LoggingServicePerformerFactory *factory;

    QHash<QTcpSocket*,LoggingServicePerformer*> connections;
};

TCPNetworkAcceptor::TCPNetworkAcceptor(const QString &address, quint16 port, LoggingServicePerformerFactory *factory, QObject *parent)
    : NetworkServiceProvider(parent)
{
    this->d = new TCPNetworkAcceptorPrivate;

    d->address = address;
    d->port = port;
    d->factory = factory;
}

TCPNetworkAcceptor::~TCPNetworkAcceptor()
{
    if(d->server->isListening()) d->server->close();

    foreach(QTcpSocket *socket, d->connections.keys())
    {
        socket->close();
    }

    delete d->server;
    delete this->d;
}

bool TCPNetworkAcceptor::initialize()
{
    qDebug() << "TCPNetworkAcceptor - Instantiating TCP service acceptor transport device.";
    d->server = new QTcpServer(this);

    qDebug() << "TCPNetworkAcceptor - Registering event handlers.";
    QObject::connect(d->server, SIGNAL(newConnection()), this, SLOT(onNewConnection()));

    qDebug() << "TCPNetworkAcceptor - Waiting for incoming connections.";
    if(!d->server->listen(QHostAddress(d->address), d->port))
    {
        emit this->statusChanged(QString("Failed to bind %1:%2").arg(d->address, QString::number(d->port)));
        return false;
    }

    emit this->statusChanged("Waiting for connections.");
    return true;
}

void TCPNetworkAcceptor::onNewConnection()
{
    QTcpSocket *socket = d->server->nextPendingConnection();
    qDebug() << "TCPNetworkAcceptor - Accepting incoming connection from"
             << QString("%1:%2").arg(socket->peerAddress().toString(), QString::number(socket->peerPort()));

    qDebug() << "TCPNetworkAcceptor - Registering disconnect event handler.";
    QObject::connect(socket, SIGNAL(disconnected()), this, SLOT(onClientDisconnect()));

    qDebug() << "TCPNetworkAcceptor - Creating service performer.";
    LoggingServicePerformer *performer = d->factory->createInstance(socket, this);
    d->connections.insert(socket, performer);

    emit this->statusChanged(QString("Connected to %1 client/s").arg(d->connections.count()));
}

void TCPNetworkAcceptor::onClientDisconnect()
{
    QTcpSocket *socket = qobject_cast<QTcpSocket*>(QObject::sender());
    LoggingServicePerformer *performer = d->connections.value(socket);
    qDebug() << "TCPNetworkAcceptor - Handling client disconnect from"
             << QString("%1:%2").arg(socket->peerAddress().toString(), QString::number(socket->peerPort()));
    d->factory->destroyInstance(performer);
    d->connections.remove(socket);
    emit this->statusChanged(QString("Connected to %1 client/s").arg(d->connections.count()));
}
