/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#include "SearchBar.h"
#include "../main.h"
#include "../display/texture.h"
#include "../display/glcanvas.h"
#include <QtCore/QDebug>
#include "../resources/TextSurfaceManager.h"

static GLfloat textCoords[8 * 2] = {
		0, 0,
		0, 1,
		31/64.0, 0,
		31/64.0, 1,
		32/64.0, 0,
		32/64.0, 1,
		1, 0,
		1, 1
};

static GLfloat verticesLandscape[8 * 3] = { 0, 0, 0, 0, 64, 0, 32, 0, 0, 32, 64, 0, SCREEN_WIDTH - 32, 0, 0, SCREEN_WIDTH - 32, 64, 0, SCREEN_WIDTH, 0, 0, SCREEN_WIDTH, 64, 0 };

static GLfloat verticesPortrait[8 * 3] = {
		0, 0, 0,
		64, 0, 0,
		0, 32, 0,
		64, 32, 0,
		0, SCREEN_HEIGHT - 32, 0,
		64, SCREEN_HEIGHT - 32, 0,
		0, SCREEN_HEIGHT, 0,
		64, SCREEN_HEIGHT, 0
};

SearchBar::SearchBar() {
	SDL_Surface* maskSurface = NULL;
	SDL_Surface* surface = loadTextureFromFile("/opt/cloudgps/res/text_input.png", NULL, &maskSurface);
	texture = createTexture((GLushort*) surface -> pixels, surface -> w, surface -> h, FALSE);
	mask = createTexture((GLushort*) maskSurface -> pixels, maskSurface -> w, maskSurface -> h, FALSE);
	cursorInput = 0;
	cursorScreen = 0;
	inputValue = "";
	SDL_FreeSurface(surface);
	SDL_FreeSurface(maskSurface);
	recomputeScreenCursor();
}

void SearchBar::cursorLeft() {
	if (cursorInput > 0) {
		cursorInput--;
	}
	recomputeScreenCursor();
}

void SearchBar::cursorRight() {
	if (cursorInput <= inputValue.size()) {
		cursorInput++;
	}
	recomputeScreenCursor();
}

int SearchBar::handleMouse() {
	TextSurfaceManager* txtMan = TextSurfaceManager::instance();
	if (mouse[0].button != 0 && mouse[0].x > SCREEN_WIDTH - (int) canvas.searchBarY && mouse[0].x < SCREEN_WIDTH - (int) canvas.searchBarY + 64) {
		int textPixels = SCREEN_HEIGHT - 16 - mouse[0].y;
		int i;
		for (i = 0; i <= inputValue.size(); i++) {
			QString textBeforeCursor = inputValue.left(i);
			int sizeOfTextBeforeCursor = txtMan -> getTextWidth(32, textBeforeCursor);
			if (sizeOfTextBeforeCursor > textPixels) {
				cursorInput = i;
				recomputeScreenCursor();
				break;
			}
		}
		if (i == inputValue.size() + 1) {
			cursorInput = i - 1;
			recomputeScreenCursor();
		}
		return TRUE;
	}
	return FALSE;
}


void SearchBar::recomputeScreenCursor() {
	TextSurfaceManager* txtMan = TextSurfaceManager::instance();
	QString textBeforeCursor = inputValue.left(cursorInput);
	int sizeOfTextBeforeCursor = txtMan -> getTextWidth(32, textBeforeCursor);
	cursorScreen = SCREEN_HEIGHT - 16 - sizeOfTextBeforeCursor;
}

void SearchBar::addChars(QString chars) {
	inputValue = inputValue.left(cursorInput) + chars + inputValue.mid(cursorInput);
	cursorInput += chars.length();
	lastChange = nowMillis;
	recomputeScreenCursor();
}

QString SearchBar::getValue() {
	return inputValue;
}

void SearchBar::removeChar() {
	if (!inputValue.isEmpty() && cursorInput > 0) {
		inputValue = inputValue.left(cursorInput - 1) + inputValue.mid(cursorInput);
		cursorInput--;
	}
	recomputeScreenCursor();
}

bool SearchBar::isActive() {
	return canvas.searchBarY != 0.0;
}

void SearchBar::render() {
	if (isActive()) {
		if (canvas.orientationTransitionLinear > 80) {
			glTranslatef(SCREEN_WIDTH - (int) canvas.searchBarY, 0, 0);
			glVertexPointer(3, GL_FLOAT, 0, verticesPortrait);
		} else {
			glTranslatef(0, SCREEN_HEIGHT - (int) canvas.searchBarY, 0);
			glVertexPointer(3, GL_FLOAT, 0, verticesLandscape);
		}

		glTexCoordPointer(2, GL_FLOAT, 0, textCoords);
		glColor4f(1, 1, 1, 1);
		glBlendFunc(GL_DST_COLOR, GL_ZERO);
		glBindTexture(GL_TEXTURE_2D, mask);
		glDrawArrays(GL_TRIANGLE_STRIP, 0, 8);

		glBlendFunc(GL_ZERO, GL_ONE_MINUS_SRC_COLOR);
		glBindTexture(GL_TEXTURE_2D, texture);
		glDrawArrays(GL_TRIANGLE_STRIP, 0, 8);

		if (!inputValue.isEmpty()) {
			TextSurfaceManager* txtMan = TextSurfaceManager::instance();
			TextEntry* entry = txtMan -> getTextEntry(32, inputValue);
			resetQuadStripVertices();
			GLfloat tx, ty;
			if (canvas.orientationTransitionLinear > 80) {
				setQuadStripSize(entry -> height, entry -> width, quadStripVertices);
				tx = 32 - entry -> height / 2;
				ty = SCREEN_HEIGHT - entry -> width - 16;
				glTexCoordPointer(2, GL_FLOAT, 0, txtMan -> getTexCoords(entry));
			} else {
				setQuadStripSize(entry -> width, entry -> height, quadStripVertices);
				ty = 32 - entry -> height / 2;
				tx = 16;
				glTexCoordPointer(2, GL_FLOAT, 0, txtMan -> getTexCoordsLandscape(entry));
			}
			glTranslatef(tx, ty, 0);
			glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_COLOR);
			glVertexPointer(3, GL_FLOAT, 0, quadStripVertices);
			glBindTexture(GL_TEXTURE_2D, entry -> texture);
			glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
			glTranslatef(-tx, -ty, 0);

		}
		if (((nowMillis - lastChange) / 200) % 2 == 0) {
			GLfloat cursor[] = { 16, cursorScreen, 0, 48, cursorScreen, 0,
								 16, cursorScreen + 1, 0, 48, cursorScreen + 1, 0 };
			glColor4f(1, 1, 1, 1);
			glDisable(GL_BLEND);
			glBindTexture(GL_TEXTURE_2D, 0);
			glVertexPointer(3, GL_FLOAT, 0, cursor);
			glDrawArrays(GL_LINES, 0, 4);
			glEnable(GL_BLEND);
		}
		if (canvas.orientationTransitionLinear > 80) {
			glTranslatef(-SCREEN_WIDTH + (int) canvas.searchBarY, 0, 0);
		} else {
			glTranslatef(0, -SCREEN_HEIGHT + (int) canvas.searchBarY, 0);
		}
	}
}

SearchBar::~SearchBar() {
	// TODO Auto-generated destructor stub
}
