/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#ifndef MAIN_H_
#define MAIN_H_

#include <glib-2.0/glib.h>
#include <math.h>

#ifdef N900
#include <GLES/gl.h>
#include <GLES/glext.h>
#else
#include <GL/gl.h>
#endif
#define M_PI 3.14159265358979323846

#define CLOUDGPS_VERION "0.4.4"

/** Equatorial earth radius in meters. */
#define EARTH_RADIUS 6378137

#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define TILE_SIZE     256
#define PORTRAIT_ROTATION_Y 14
#define LANDSCAPE_ROTATION_X 14

// how many satellite bars are drawn in status bar
#define MAX_SATTELITES 12

// how many tiles horizontally
#define TILES_X 8

// how many tiles vertically
#define TILES_Y 8

#define MAX_TEXTURES_CREATED_PER_FRAME 1

typedef enum {
    LANDSCAPE = 0, PORTRAIT = 1
} Orientation;

struct {
    int mipmapToggle;
    int showGrid, showCoordinates, accelerometerEnabled;
    int showDebugTiles;
    int locallyLoadedFadeTime, downloadedFadeTime, zoomChangeFadeTime;

    Orientation orientation;
    char *userAgent;
    char *referer;
} options;

//TTF_Font *font = NULL;


typedef struct {
    char* name;
    int minZoom, maxZoom;
    char* filenameFormat;
    char* urlFormat;
    char* urlFormatType;
    char quadKeyFirstChar;
} TileProvider;

typedef enum {
    VIEW_2D, VIEW_3D
} ViewMode;

typedef struct {
    double latitude, longitude;
} WorldCoordinate;

typedef struct {
    int tilex, tiley, zoom;
    double x, y;
} TileCoordinate;

typedef struct {
	GLfloat x, y;
} Point;

typedef struct {
    GLfloat r, g, b, a;
} Color;

typedef enum {
    UI_HIDDEN = 0, UI_COMING = 1, UI_SHOWN = 2, UI_GOING = 3
} uiElemStatus;

typedef enum {
	NONE = 0, LINEAR = 1, CUBIC = 2
} AttractionMode;

typedef struct {
    GLuint name;
    int size;
} Texture;

typedef struct {
    GLfloat portraitx, portraity;
    GLfloat landscapex, landscapey;
    GLfloat x, y;
    GLfloat destx, desty;
    long stateChangeMilis;
    uiElemStatus status;
    Texture texture;
    Texture mask;
    Color color;
    void (*handlePressed)();
    void (*handleDragged)();
    GLfloat * texCoords;
} UiElement;

typedef struct {
    GLfloat x, y, dx, dy, scale, oldScale, destScale, rotx, roty, rotz, destRotx, destRoty, destRotz, drotz, friction;
    int attractedToRotZ;
    int tilex, tiley;
    int zoom, oldZoom, attractionIntermediateZoom, attractionDestZoom;
    int attractionZooming; // 0 - not zooming, 1 - zooming in, -1 - zooming out
    WorldCoordinate center;
    WorldCoordinate previousCenter; // before following mode turned on
    WorldCoordinate attraction; // position to which canvas is attracted (ie current position in following mode)
    WorldCoordinate currentPos;
    TileProvider* provider;
    ViewMode viewMode;
    GLfloat orientationTransition;
    int orientationTransitionLinear;
    int followingMypos;
    int attractionToPoint;

    int rotation2d;
    int searchBarActive;
    int zoomBarActive;
    long zoomBarVisibleMilis;
    long zoomBarLastDragged;
    GLfloat zoomBarAlpha;
    GLfloat zoomKnotPosition;
    GLfloat zoomingSpeed;
    GLfloat searchBarY;
    GArray* buildings;
    GList* markers;
    int buildingsVerticesPrecomputed;
    GLfloat buildingsHeight;
    GLfloat buildingsTileX;
    GLfloat buildingsTileY;
    TileCoordinate boundingTrapezoid[4]; // points in clockwise order

    int arrowPos; // 0 -none, 1 - left, 2 - right, 3 - up, 4 - down
    GLfloat arrowPosX, arrowPosY;

    volatile int lastSearchId;
} Canvas;
extern Canvas canvas;

typedef struct {
    char input[100];
    int cursorScreen; // Cursor position on screen (in pixels)
    int cursorInput; // Cursor position in input string (in bytes)
    int selectionStart;
    int selectionCount;
    int historyIndex;
    char** history;
    long lastChange;
} SearchBar;
extern SearchBar searchBar;

typedef struct {
    int x, y, oldx, oldy, button, oldButton, xdelta, ydelta;
    long pressed, moved; // milliseconds of last pressed and moved events, -1 if mousebutton is not down
} Mouse;
extern Mouse mouse;


typedef struct {
   const char* name;
   const char* description;
   const char* wikipedia;
   WorldCoordinate centroid;
   WorldCoordinate bounds[2];
   int searchId;
} GeoCodeResult;

typedef enum {
	SEARCH_RESULT, FAVOURITE
} MarkerType;

typedef struct {
	MarkerType type;
	GeoCodeResult* geoCodeResult;
	TileCoordinate tileCoordinate;
	Point screenCoordinate;
	int visible, raised, recalculateCoords, poi;
	GLfloat rotz;
	int raiseLevel;
} ScreenMarker;

typedef struct {
    int x, y, z;
    int calibrateX, calibrateY, calibrateZ;
    int performCalibration; // if set to TRUE calibration will be performed on next accelerometer read.
} Accelerometer;
extern Accelerometer accelerometer;

extern int quit;
extern int forceGarbageCollection;
extern int currentTilesIdx;
extern volatile int syncLoadedTilesAppearance;
extern int texturesCreatedThisFrame;
extern long nowMillis;
extern GQueue* downloadQueue;
extern int tileCoordinateVisibility[TILES_X + 1][TILES_Y + 1];

#ifdef N900
#include <location/location-gps-device.h>
extern LocationGPSDevice *device;
#endif

#endif /* MAIN_H_ */
