/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
#include <sys/stat.h>
#include <unistd.h>
#include <libgen.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>

typedef enum {
    STATE_EMPTY = 0,
    STATE_LOADING = 1,
    STATE_ERROR = 2,
    STATE_LOADED = 3,
    STATE_SCALED_LOADING = 4,
    STATE_TEXTURE_BIND_WAIT = 5,
    STATE_LOADED_SYNC_WAIT = 6
} TileState;

typedef struct {
    int zoom;
    int lon, lat; // TODO change to tilex and tiley
    GLuint texture;
    GLuint oldTexture[4];
    GLfloat oldTextureCoords[4][8];
    GLfloat oldTextureSize;
    TileState state;
    long stateChangeTime;
    int transitionTime;

    // TODO
    SDL_Surface * tmpSurface; // surface used to create pixels
    GLushort * pixels4444; // pixels used as texture, to be freed when texture is deleted

    SDL_Surface * oldTmpSurface[4]; // the same pair as above but for zoom changing transitions
    GLushort * oldPixels4444[4];
} t_tile;

char* getTileFileName(TileProvider provider, t_tile *tile);
char* getTileUrl(TileProvider provider, t_tile *tile);
int downloadUrl(char *url, char *filename);
GLushort * convert8888to4444(SDL_Surface* surfaceRGBA);

void deallocateTile(t_tile *tile) {
    //    fprintf(stderr, "deallocate tile not implemented yet\n");
    if (tile->texture != 0) {
        glDeleteTextures(1, &tile->texture);
        tile->texture = 0;
    }
    free(tile -> pixels4444);
    tile -> pixels4444 = NULL;

    if (tile -> tmpSurface) {
        SDL_FreeSurface(tile -> tmpSurface);
        tile -> tmpSurface = NULL;
    }
    //free(tile);
}

// todo: move to file.c
void ensureDirExists(char *filename) {
    g_mkdir_with_parents(dirname(strdup(filename)), 0755);
}

SDL_Surface* loadTextureFromFile(char *filename);

int loadTileTexture(char* filename, t_tile* tile) {
    SDL_Surface* texture = loadTextureFromFile(filename);

    if (texture) {
        SDL_Event event;
        event.type = SDL_USEREVENT;
        event.user.code = 0;
        event.user.data1 = convert8888to4444(texture);
        tile -> pixels4444 = event.user.data1;
        event.user.data2 = tile;
        tile -> tmpSurface = texture;
        // int result =
        SDL_PushEvent(&event);
        // fprintf(stderr, "loading from file '%s' succeded in %ld ms. push event result = %d\n", filename, (SDL_GetTicks() - startTime), result);
        // free(filename);
        return TRUE;
    }
    return FALSE;
}

int shouldAbortLoading(t_tile * tile) {
    if ((tile -> lat < canvas.tilex) || (tile -> lon < canvas.tiley) || (tile -> lat >= (canvas.tilex + TILES_X)) || (tile -> lon >= (canvas.tiley
            + TILES_Y)) || (tile -> zoom != canvas.zoom)) {
        return TRUE;
    }
    return FALSE;
}

int downloadThread(void* queue) {
    t_tile * tile = NULL;

    setpriority(PRIO_PROCESS, 0, 19);

    while (!quit) {
        tile = (t_tile*) dequeue(queue);
        while (tile == NULL) {
            //            fprintf(stderr, "queue empty, waiting, thread id = %d\n", SDL_ThreadID());
            SDL_Delay(0);
            if (quit) {
                return 0;
            }
            tile = (t_tile*) dequeue(queue);
        }

        int mapSize = 1 << tile -> zoom;

        if (tile -> lat < 0 || tile -> lon < 0 || tile -> lat >= mapSize || tile -> lon >= mapSize) {
            tile -> state = STATE_EMPTY;
            continue;
        }

        if (shouldAbortLoading(tile)) {
            continue;
        }

        if (tile -> state != STATE_SCALED_LOADING) {
            tile -> state = STATE_LOADING;
        }

        //        fprintf(stderr, "load tile start, thread id = %d\n", SDL_ThreadID());
        char *filename;
        if (!loadTileTexture(getTileFileName(canvas.provider, tile), tile)) {

            if (shouldAbortLoading(tile)) {
                continue;
            }

            //            fprintf(stderr, "load tile from local file failed, trying to download tile\n");
            filename = getTileFileName(canvas.provider, tile);
            ensureDirExists(filename);
            if (downloadUrl(getTileUrl(canvas.provider, tile), filename)) {

                if (shouldAbortLoading(tile)) {
                    continue;
                }

                if (loadTileTexture(getTileFileName(canvas.provider, tile), tile)) {
                    tile -> state = STATE_TEXTURE_BIND_WAIT;
                    if (tile ->transitionTime == 0) {
                        tile ->transitionTime = 2000;
                    }
                    //                    fprintf(stderr, "download successfull\n");
                } else {
                    tile -> state = STATE_ERROR;
                    //                    fprintf(stderr, "unable to load downloaded tile\n");
                }
            } else {
                tile -> state = STATE_ERROR;
                //                fprintf(stderr, "download error\n");
            }
        } else {
            if (tile ->transitionTime == 0) {
                tile -> transitionTime = 200;
            }
        }
    }
    fprintf(stderr, "finished thread, id = %d\n", SDL_ThreadID());
    return 0;
}
