/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

void loadTexture(Texture* texture, char* filename) {
    SDL_Surface* textureRGB = NULL;
    //    fprintf(stderr, "loading texture from %s\n", filename);
    textureRGB = loadTextureFromFile(filename);
    //    fprintf(stderr, "loadTextureFromFile returned: %p\n", textureRGB);
    if (textureRGB != NULL) {
        texture -> size = textureRGB -> w;
        texture -> name = createTexture(convert8888to4444(textureRGB), texture->size, texture-> size);
    } else {
        fprintf(stderr, "loadTexture %s failed\n", filename);
    }
}

UiElement* createUiElement(char* texture, char* mask, int x, int y, Anchor anchory, Anchor anchorx, void(*handlePressed)(), void(*handleDragged)(),
        int addToQueue) {
    UiElement* element = calloc(sizeof(UiElement), 1);
    if (element == NULL) {
        fprintf(stderr, "memory allocation failed in createUiElement\n");
        return NULL;
    }

    loadTexture(&(element -> mask), mask);
    loadTexture(&(element -> texture), texture);

    switch (anchorx) {
        case ANCHOR_LEFT:
            break;
        case ANCHOR_RIGHT:
            x -= (element -> texture.size);
            break;
        case ANCHOR_CENTER:
        default:
            x -= (element -> texture.size) / 2;
            break;
    }
    switch (anchory) {
        case ANCHOR_UP:
            break;
        case ANCHOR_DOWN:
            y -= (element -> texture.size);
            break;
        case ANCHOR_CENTER:
        default:
            y -= (element -> texture.size) / 2;
            break;
    }

    element -> x = x;
    element -> y = y;
    element -> status = UI_SHOWN;
    element -> color.r = 1.0;
    element -> color.g = 1.0;
    element -> color.b = 1.0;
    element -> color.a = 1.0;
    element -> handlePressed = handlePressed;
    element -> handleDragged = handleDragged;

    //    fprintf(stderr, "addtoqueue = %d\n", addToQueue);
    if (addToQueue) {
        enqueue(uiElems, element);
    }

    return element;
}

void destroyUiElement(UiElement* elem) {
    fprintf(stderr, "STUB - destroyUiElement(UiElement* elem)\n");
}

void drawUiElement(UiElement* elem) {

    glTranslatef(elem -> x, elem -> y, 0);
    glEnable(GL_BLEND);
    // mask
    glBlendFunc(GL_DST_COLOR, GL_ZERO);
    glColor4f(1.0, 1.0, 1.0, 1.0);
    glBindTexture(GL_TEXTURE_2D, elem->mask.name);
    setQuadSize(elem->mask.size, elem->mask.size, quadVertices);
    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
    glTexCoordPointer(2, GL_FLOAT, 0, texCoords[0]);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    // texture
    glColor4f(elem->color.r, elem->color.g, elem->color.b, elem->color.a);
    glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
    glBindTexture(GL_TEXTURE_2D, elem-> texture.name);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    glTranslatef(-elem -> x, -elem -> y, 0);
}

void loadUI() {
    loadTexture(&fontMedium, "/opt/cloudgps/res/font-medium.png");
    loadTexture(&fontMediumMask, "/opt/cloudgps/res/font-medium-mask.png");
    loadTexture(&barbgmask, "/opt/cloudgps/res/bar-bg-mask.png");
    //    loadTexture(&barbg, "/opt/cloudgps/res/bar-bg.png");
    //    loadTexture(&barbg, "/opt/cloudgps/res/zoom-minus-mask.png");
    uiElems = createQueue();

    zoomOut = createUiElement("/opt/cloudgps/res/zoom-minus.png", "/opt/cloudgps/res/zoom-minus-mask.png", 50, SCREEN_HEIGHT - 50, ANCHOR_RIGHT,
            ANCHOR_DOWN, &tileEngineZoomOut, &tileEngineZoomOut, TRUE);

    zoomIn = createUiElement("/opt/cloudgps/res/zoom-plus.png", "/opt/cloudgps/res/zoom-plus-mask.png", SCREEN_WIDTH - 50, SCREEN_HEIGHT - 50,
            ANCHOR_RIGHT, ANCHOR_UP, &tileEngineZoomIn, &tileEngineZoomIn, TRUE);
    gotomypos = createUiElement("/opt/cloudgps/res/gotomypos.png", "/opt/cloudgps/res/gotomypos-mask.png", SCREEN_WIDTH - 150, SCREEN_HEIGHT - 50,
            ANCHOR_RIGHT, ANCHOR_UP, &tileEngineGotomypos, &tileEngineGotomypos, TRUE);
    position = createUiElement("/opt/cloudgps/res/position.png", "/opt/cloudgps/res/position-mask.png", SCREEN_WIDTH - 50, 50, ANCHOR_RIGHT,
            ANCHOR_UP, NULL, NULL, TRUE);
    view2d = createUiElement("/opt/cloudgps/res/2d.png", "/opt/cloudgps/res/2d-mask.png", 150, SCREEN_HEIGHT - 50,
            ANCHOR_LEFT, ANCHOR_DOWN, &tileEngineViewMode2D, NULL, TRUE);
    view2d -> status = UI_HIDDEN;
    view3d = createUiElement("/opt/cloudgps/res/3d.png", "/opt/cloudgps/res/3d-mask.png", 150, SCREEN_HEIGHT - 50,
            ANCHOR_LEFT, ANCHOR_DOWN, &tileEngineViewMode3D, NULL, TRUE);
    crosshair = createUiElement("/opt/cloudgps/res/crosshair.png", "/opt/cloudgps/res/crosshair-mask.png", SCREEN_WIDTH / 2, SCREEN_HEIGHT / 2,
            ANCHOR_CENTER, ANCHOR_CENTER, NULL, NULL, TRUE);

}

void drawUI() {
    element* elem = uiElems -> first;

    while (elem != NULL) {
        UiElement* uiElem = elem -> data;
        if (uiElem -> status != UI_HIDDEN && uiElem != position) {
            drawUiElement(elem -> data);
        }
        elem = elem -> next;
    }

    glEnable(GL_BLEND);
    glBlendFunc(GL_DST_COLOR, GL_ZERO);
    glColor4f(.4, .4, .4, 0.5);
    setQuadSize(SCREEN_WIDTH, 50, quadVertices);
    glBindTexture(GL_TEXTURE_2D, 0);
    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    glTranslatef(10, 5, 0);
    setQuadSize(64, 64, quadVertices);
    glColor4f(1.0, 1.0, 1.0, 1.0);
    glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
    glBindTexture(GL_TEXTURE_2D, zoomOut->texture.name);
    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
    glTexCoordPointer(2, GL_FLOAT, 0, texCoords[0]);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    sprintf(buf, "Downloaded: %.2f kb", downloaded / 1024.0);
    drawString(buf, 1.0, 1.0, 1.0);
    glTranslatef(0, 20, 0);
    switch (device -> fix -> mode) {
        case LOCATION_GPS_DEVICE_MODE_2D:
            sprintf(buf, "GPS fix: 2D");
            break;
        case LOCATION_GPS_DEVICE_MODE_3D:
            sprintf(buf, "GPS fix: 3D");
            break;
        case LOCATION_GPS_DEVICE_MODE_NOT_SEEN:
            sprintf(buf, "GPS fix: not seen");
            break;
        case LOCATION_GPS_DEVICE_MODE_NO_FIX:
            sprintf(buf, "GPS fix: no fix");
            break;
    }
    drawString(buf, 1.0, 1.0, 1.0);
    glTranslatef(180, -20, 0);

    if (device -> fix -> fields | LOCATION_GPS_DEVICE_ALTITUDE_SET && !isnan(device -> fix -> altitude)) {
        sprintf(buf, "Altitude: %.2f m", device -> fix -> altitude);
    } else {
        sprintf(buf, "Altitude: ---");
    }
    drawString(buf, 1.0, 1.0, 1.0);
    glTranslatef(0, 20, 0);

    if (device -> fix -> fields | LOCATION_GPS_DEVICE_SPEED_SET && !isnan(device -> fix -> speed)) {
        sprintf(buf, "Speed: %.2f km/h", device -> fix -> speed);
    } else {
        sprintf(buf, "Speed: ---");
    }
    drawString(buf, 1.0, 1.0, 1.0);

    glTranslatef(SCREEN_WIDTH - 280, 0, 0);
    sprintf(buf, "Battery: %d%%", batteryPercentage);
    drawString(buf, 1.0, 1.0, 1.0);

    glTranslatef(-SCREEN_WIDTH + 110, -25, 0);

    int i;
    setBoxSize(12, 32, boxVertices);
    glTranslatef(SCREEN_WIDTH - 12 * 18 - 130, 3, 0);
    for (i = 0; i < 12; i++) {
        glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
        glBindTexture(GL_TEXTURE_2D, 0);
        glColor4f(.7, .7, .7, 1.0);
        glVertexPointer(3, GL_FLOAT, 0, boxVertices);
        glDrawArrays(GL_LINE_LOOP, 0, 4);

        if (device && device -> satellites) {
            if (i < device -> satellites -> len) {
                LocationGPSDeviceSatellite* sat = device -> satellites -> pdata[i];
                int barHeight = (sat -> signal_strength) / 2;
                glTranslatef(0, 32 - barHeight, 0);
                // fprintf(stderr, "prn = %d, signal_strenght = %d, azimuth = %d, elevation = %d, in_use = %d\n", sat -> prn, sat-> signal_strength, sat -> azimuth, sat -> elevation, sat -> in_use);
                if (sat -> in_use) {
                    glColor4f(1.0, 1.0, 1.0, 0.5);
                } else {
                    glColor4f(.5, .5, .5, 0.5);
                }
                setQuadSize(12, barHeight, quadVertices);
                glVertexPointer(3, GL_FLOAT, 0, quadVertices);
                glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
                sprintf(buf, "%d", sat -> prn);
                glTranslatef(5 - stringWidth(buf) / 2, barHeight, 0);
                drawString(buf, .7 + .3 * (i % 2), .7 + .3 * (i % 2), .7 + .3 * (i % 2));
                glTranslatef(-5 + stringWidth(buf) / 2, -32, 0);
            }
        }
        glTranslatef(18, 0, 0);
    }
}

int processUiMouse() {
    element* elem = uiElems -> first;

    if (mouse.button == 1) {
        if (mouse.oldButton == 0) {
            while (elem != NULL) {
                UiElement* e = (UiElement*) elem->data;
                if (e -> status == UI_SHOWN && mouse.x >= e->x && mouse.x <= e->x + e->texture.size && mouse.y >= e->y && mouse.y <= e->y + e->texture.size) {
                    pressedUiElem = e;
                    if (pressedUiElem -> handlePressed != NULL) {
                        pressedUiElem -> handlePressed();
                        return TRUE;
                    }
                    return FALSE;
                }

                elem = elem -> next;
            }
            pressedUiElem = NULL;
        } else {
            if (pressedUiElem != NULL && pressedUiElem -> handleDragged != NULL) {
                pressedUiElem -> handleDragged();
                return TRUE;
            }
        }
    }
    return FALSE;
}

void updateUi() {
    if (device && device -> fix) {

        if (device -> fix -> fields | LOCATION_GPS_DEVICE_LATLONG_SET) {
            position -> status = UI_SHOWN;
            WorldCoordinate wc;
            wc.latitude = device -> fix -> latitude;
            wc.longitude = device -> fix -> longitude;
            TileCoordinate tc;
            toTileCoordinate(&wc, &tc, canvas.zoom);
            toScreenCoordinate(&tc, &position ->x, &position -> y);
            position -> x -= position -> texture.size / 2;
            position -> y -= position -> texture.size / 2;

            position -> color.a = 0.75 + 0.25 * sin(nowMillis / 500.0);
            position -> color.r = position -> color.a;
            position -> color.g = position -> color.a;

            if (device -> satellites_in_use > 0) {
                position -> color.b = 1.0 - position -> color.a;
            } else {
                position -> color.b = position -> color.a;
            }

            //            fprintf(stderr, "position on screeen: x = %f, y = %f, color = %f\n", position -> x, position -> y, position -> color.a);
        } else {
            position -> status = UI_HIDDEN;
        }
    }

    gotomypos -> color.r = followingMypos * .3 + .4;
    gotomypos -> color.g = followingMypos * .3 + .4;
    gotomypos -> color.b = followingMypos * .3 + .4;
}

