/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */

#define DOWNLOADING_THREADS 4

SDL_Thread* downloadThreads[DOWNLOADING_THREADS];
queue * downloadQueue;

//SDL_Thread* fileIOthread;
//queue * fileIOqueue;

int followingMypos;

int downloadThread(void* queue);

void toScreenCoordinate(TileCoordinate *tc, GLfloat *x, GLfloat *y) {
    *x = (tc->tilex - canvas.tilex) * TILE_SIZE + tc->x;
    *y = (tc->tiley - canvas.tiley) * TILE_SIZE + tc->y;
}

void toTileCoordinate(WorldCoordinate* wc, TileCoordinate* result, int zoom) {
    result -> zoom = zoom;

    double p = pow(2.0, zoom);
    double lat = (wc->longitude + 180.) / 360.0 * p;
    result -> tilex = (int) lat;
    result ->x = (lat - result->tilex) * TILE_SIZE;

    double lon = (1.0 - log(tan(wc->latitude * M_PI / 180.0) + 1.0 / cos(wc->latitude * M_PI / 180.0)) / M_PI) / 2.0 * p;
    result->tiley = (int) lon;
    result ->y = (lon - result->tiley) * TILE_SIZE;
}

void toWorldCoordinate(TileCoordinate* tc, WorldCoordinate* result) {
    double p = pow(2.0, tc->zoom);
    double xx = (double) tc->tilex + (tc->x / (double) TILE_SIZE);
    result->longitude = ((xx * 360.0) / p) - 180.0;

    double yy = tc->tiley + tc->y / (double) TILE_SIZE;
    double n = M_PI - 2.0 * M_PI * yy / p;
    double m = 0.5 * (exp(n) - exp(-n));
    double a = atan(m);
    result->latitude = (180.0 / M_PI * a);
}

void canvasCenterToTileCoordinate(TileCoordinate* tc) {
    tc -> zoom = canvas.zoom;
    tc -> tilex = canvas.tilex;
    tc -> tiley = canvas.tiley;
    tc -> x = (TILE_SIZE - canvas.x) + SCREEN_WIDTH / 2;
    tc -> y = (TILE_SIZE - canvas.y) + SCREEN_HEIGHT / 2;
}

void initTileEngine() {
    int i, j;

    downloadQueue = createQueue();
    canvas.scale = 1.0;
    canvas.viewMode = VIEW_2D;

    for (i = 0; i < DOWNLOADING_THREADS; i++) {
        //        fprintf(stderr, "before SDL_CreateThread\n");
        downloadThreads[i] = SDL_CreateThread(downloadThread, (void*) downloadQueue);
        //        fprintf(stderr, "after SDL_CreateThread\n");
        SDL_Delay(15);
    }

    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            tiles[i][j][currentTilesIdx] = (t_tile*) calloc(sizeof(t_tile), 1);

            tiles[i][j][currentTilesIdx] ->lat = canvas.tilex + i;
            tiles[i][j][currentTilesIdx]->lon = canvas.tiley + j;
            tiles[i][j][currentTilesIdx]->zoom = canvas.zoom;

            enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
        }
    }
}

int roundDown(double x) {
    return x < 0 ? (int) (x - 1.0) : (int) x;
}

int adjustShifts() {
    int horizontalShift = roundDown(canvas.x / (double) TILE_SIZE);
    int verticalShift = roundDown(canvas.y / (double) TILE_SIZE);

    canvas.x -= horizontalShift * TILE_SIZE;
    canvas.y -= verticalShift * TILE_SIZE;

    canvas.tilex += horizontalShift;
    canvas.tiley += verticalShift;
    if (horizontalShift != 0 || verticalShift != 0) {
        return TRUE;
    }
    return FALSE;
}

void recreateTiles() {
    int i, j;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            deallocateTile(tiles[i][j][currentTilesIdx]);
        }
    }
    // start loading new tile textures:
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            memset(tiles[i][j][currentTilesIdx], 0, sizeof(t_tile));
            tiles[i][j][currentTilesIdx] ->lat = canvas.tilex + i;
            tiles[i][j][currentTilesIdx]->lon = canvas.tiley + j;
            tiles[i][j][currentTilesIdx]->zoom = canvas.zoom;
            enqueue(downloadQueue, tiles[i][j][currentTilesIdx]);
        }
    }
}

void handleZoomChange() {
    // compute new tilex, tiley, x and y for canvas using canvas.center
    int i, j;

    TileCoordinate tc;

    toTileCoordinate(&canvas.center, &tc, canvas.zoom);

    canvas.tilex = tc.tilex;
    canvas.tiley = tc.tiley;
    canvas.x = tc.x - SCREEN_WIDTH / 2;
    canvas.y = tc.y - SCREEN_HEIGHT / 2;

    adjustShifts();

    canvas.x = TILE_SIZE - canvas.x;
    canvas.y = TILE_SIZE - canvas.y;

    syncLoadedTilesAppearance = TRUE;

    if (canvas.zoom - canvas.oldZoom == 1) {
        //        fprintf(stderr, "zoomIn 1\n");
        int oldtilex = tiles[0][0][currentTilesIdx] -> lat;
        int oldtiley = tiles[0][0][currentTilesIdx] -> lon;

        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                tiles[i][j][1 - currentTilesIdx] = (t_tile*) calloc(sizeof(t_tile), 1);

                tiles[i][j][1 - currentTilesIdx] -> lat = canvas.tilex + i;
                tiles[i][j][1 - currentTilesIdx] -> lon = canvas.tiley + j;
                tiles[i][j][1 - currentTilesIdx] -> zoom = canvas.zoom;

                int oldi = tiles[i][j][1 - currentTilesIdx] -> lat - (oldtilex * 2);
                int oldj = tiles[i][j][1 - currentTilesIdx] -> lon - (oldtiley * 2);

                if (tiles[oldi / 2][oldj / 2][currentTilesIdx] -> state == STATE_LOADED) {
                    tiles[i][j][1 - currentTilesIdx] -> state = STATE_SCALED_LOADING;
                    tiles[i][j][1 - currentTilesIdx] -> oldTextureSize = TILE_SIZE;

                    tiles[i][j][1 - currentTilesIdx] -> oldTexture[0] = tiles[oldi / 2][oldj / 2][currentTilesIdx] -> texture;
                    tiles[i][j][1 - currentTilesIdx] -> oldPixels4444[0] = tiles[oldi / 2][oldj / 2][currentTilesIdx] ->pixels4444;
                    tiles[i][j][1 - currentTilesIdx] -> oldTmpSurface[0] = tiles[oldi / 2][oldj / 2][currentTilesIdx] ->tmpSurface;

                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][0] = (oldi % 2) * 0.5;
                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][1] = (oldj % 2) * 0.5;

                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][2] = ((oldi % 2) + 1) * 0.5;
                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][3] = (oldj % 2) * 0.5;

                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][4] = (oldi % 2) * 0.5;
                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][5] = ((oldj % 2) + 1) * 0.5;

                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][6] = ((oldi % 2) + 1) * 0.5;
                    tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[0][7] = ((oldj % 2) + 1) * 0.5;
                    tiles[i][j][1 - currentTilesIdx] -> transitionTime = 1500;
                    enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);

                } else {
                    deallocateTile(tiles[oldi / 2][oldj / 2][currentTilesIdx]);
                    tiles[i][j][1 - currentTilesIdx] -> transitionTime = 1500;
                    enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
                }
            }
        }
        currentTilesIdx = 1 - currentTilesIdx;
    } else if (canvas.zoom - canvas.oldZoom == -1) {
        int oldtilex = tiles[0][0][currentTilesIdx] -> lat;
        int oldtiley = tiles[0][0][currentTilesIdx] -> lon;
        int ii, jj;

        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                tiles[i][j][1 - currentTilesIdx] = (t_tile*) calloc(sizeof(t_tile), 1);

                tiles[i][j][1 - currentTilesIdx] -> lat = canvas.tilex + i;
                tiles[i][j][1 - currentTilesIdx] -> lon = canvas.tiley + j;
                tiles[i][j][1 - currentTilesIdx] -> zoom = canvas.zoom;

                for (ii = 0; ii < 2; ii++) {
                    for (jj = 0; jj < 2; jj++) {
                        int oldi = (tiles[i][j][1 - currentTilesIdx] -> lat - (oldtilex / 2)) * 2 + ii - (oldtilex % 2);
                        int oldj = (tiles[i][j][1 - currentTilesIdx] -> lon - (oldtiley / 2)) * 2 + jj - (oldtiley % 2);

                        if (oldi >= 0 && oldi < TILES_X && oldj >= 0 && oldj < TILES_Y) {
                            if (tiles[oldi][oldj][currentTilesIdx] -> state == STATE_LOADED) {
                                tiles[i][j][1 - currentTilesIdx] -> oldTextureSize = TILE_SIZE / 2;
                                tiles[i][j][1 - currentTilesIdx] -> state = STATE_SCALED_LOADING;

                                tiles[i][j][1 - currentTilesIdx] -> oldTexture[ii * 2 + jj] = tiles[oldi][oldj][currentTilesIdx] -> texture;
                                tiles[i][j][1 - currentTilesIdx] -> oldPixels4444[ii * 2 + jj] = tiles[oldi][oldj][currentTilesIdx] -> pixels4444;
                                tiles[i][j][1 - currentTilesIdx] -> oldTmpSurface[ii * 2 + jj] = tiles[oldi][oldj][currentTilesIdx] -> tmpSurface;

                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][0] = 0.0;
                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][1] = 0.0;

                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][2] = 1.0;
                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][3] = 0.0;

                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][4] = 0.0;
                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][5] = 1.0;

                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][6] = 1.0;
                                tiles[i][j][1 - currentTilesIdx] -> oldTextureCoords[ii * 2 + jj][7] = 1.0;
                            }
                        }
                    }
                }
                tiles[i][j][1 - currentTilesIdx] -> transitionTime = 1500;
                enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
            }
        }
        currentTilesIdx = 1 - currentTilesIdx;
    } else {
        recreateTiles();
    }
}

int getLowerDeallocateBound(int size, int shift) {
    return shift < 0 ? 0 : (size - shift) < 0 ? 0 : (size - shift);
}

int getUpperDeallocateBound(int size, int shift) {
    return shift < 0 ? (-shift < size ? -shift : size) : size;
}

void deallocateTiles(int horizontalShift, int verticalShift) {

    int i, j;
    for (i = getLowerDeallocateBound(TILES_X, horizontalShift); i < getUpperDeallocateBound(TILES_X, horizontalShift); i++) {
        for (j = 0; j < TILES_Y; j++) {
            deallocateTile(tiles[i][j][currentTilesIdx]);
        }
    }
    for (i = 0; i < TILES_X; i++) {
        for (j = getLowerDeallocateBound(TILES_Y, verticalShift); j < getUpperDeallocateBound(TILES_Y, verticalShift); j++) {
            deallocateTile(tiles[i][j][currentTilesIdx]);
        }
    }
}

void updateTiles(int horizontalShift, int verticalShift) {
    deallocateTiles(horizontalShift, verticalShift);
    int i, j;
    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            int newPosX = i - horizontalShift;
            int newPosY = j - verticalShift;
            if (newPosX >= 0 && newPosX < TILES_X && newPosY >= 0 && newPosY < TILES_Y) {
                tiles[i][j][1 - currentTilesIdx] = tiles[newPosX][newPosY][currentTilesIdx];
            } else {
                tiles[i][j][1 - currentTilesIdx] = (t_tile*) calloc(sizeof(t_tile), 1);
                tiles[i][j][1 - currentTilesIdx]->lat = tiles[i][j][currentTilesIdx]->lat - horizontalShift;
                tiles[i][j][1 - currentTilesIdx]->lon = tiles[i][j][currentTilesIdx]->lon - verticalShift;
                tiles[i][j][1 - currentTilesIdx]->zoom = canvas.zoom;

                enqueue(downloadQueue, tiles[i][j][1 - currentTilesIdx]);
            }
        }
    }
    currentTilesIdx = 1 - currentTilesIdx;
}

void tileEngineProcessMouse(int uiElemPressed) {
    canvas.dx *= 0.97;
    canvas.dy *= 0.97;

    if (uiElemPressed == FALSE) {
        if (mouse.button) {
            followingMypos = FALSE;
        }
        canvas.dx += mouse.xdelta / 10.0;
        canvas.dy += mouse.ydelta / 10.0;

        if (mouse.pressed > 0 && nowMillis - mouse.pressed > 30) {
            canvas.dx *= 0.97;
            canvas.dy *= 0.97;
        }

        if (mouse.pressed > 0 && nowMillis - mouse.moved > 50) {
            canvas.dx *= 0.2;
            canvas.dy *= 0.2;
        }
    }

    canvas.x += canvas.dx;
    canvas.y += canvas.dy;
}

void tileEngineUpdateCoordinates() {
    //int zoomChange = (int) round(log2((double) canvas.scale));
    int zoomChange = 0;

    if (followingMypos) {
        if (device -> fix -> fields | LOCATION_GPS_DEVICE_LATLONG_SET) {
            GLfloat x, y;
            WorldCoordinate wc;
            wc.latitude = device -> fix -> latitude;
            wc.longitude = device -> fix -> longitude;
            TileCoordinate tc;
            toTileCoordinate(&wc, &tc, canvas.zoom);
            toScreenCoordinate(&tc, &x, &y);
            //            fprintf(stderr, "x = %f, y = %f\t\ttilex = %d, tiley = %d, tc.x = %f, tc.y = %f \n", x, y, tc.tilex, tc.tiley, tc.x, tc.y);

            canvas.dx = (SCREEN_WIDTH / 2.0 - (canvas.x - TILE_SIZE + x)) / 20.0;
            canvas.dy = (SCREEN_HEIGHT / 2.0 - (canvas.y - TILE_SIZE + y)) / 20.0;
            if (canvas.viewMode == VIEW_3D) {
                if ((device -> fix -> fields | LOCATION_GPS_DEVICE_TRACK_SET) && device -> satellites_in_use > 0 && device -> fix -> speed > 7) {
                    canvas.rotz += (device -> fix -> track - canvas.rotz) * 0.03;
                }
            }
        }
    }

    if (canvas.destScale != 0.0) {
        GLfloat tmp = canvas.destScale - canvas.scale;
        canvas.oldScale = canvas.scale;
        canvas.scale += tmp / 5;

        if (fabs(tmp) < 0.001) {
            canvas.scale = canvas.destScale;
            canvas.destScale = 0.0;
        }
    } else {
        if (canvas.scale < 1.0) {
            zoomChange = -1;
        }
        if (canvas.scale >= 2.0) {
            zoomChange = 1;
        }
    }

    if ((canvas.zoom + zoomChange) < 0 || (canvas.zoom + zoomChange) > canvas.maxZoom) {
        zoomChange = 0;
    }

    if (zoomChange == 0) {
        int horizontalShift = roundDown(canvas.x / TILE_SIZE);
        int verticalShift = roundDown(canvas.y / TILE_SIZE);

        if (horizontalShift != 0 || verticalShift != 0) {
            canvas.x -= horizontalShift * TILE_SIZE;
            canvas.y -= verticalShift * TILE_SIZE;

            canvas.tilex -= horizontalShift;
            canvas.tiley -= verticalShift;
            updateTiles(horizontalShift, verticalShift);
        }
    }

    TileCoordinate tc;
    canvasCenterToTileCoordinate(&tc);

    int horizontalShift = roundDown(tc.x / (double) TILE_SIZE);
    int verticalShift = roundDown(tc.y / (double) TILE_SIZE);

    tc.x -= horizontalShift * TILE_SIZE;
    tc.y -= verticalShift * TILE_SIZE;

    tc.tilex += horizontalShift;
    tc.tiley += verticalShift;

    toWorldCoordinate(&tc, &canvas.center);

    if (zoomChange != 0 /* && nowMillis - lastZoomChangeMilis > 500 */) {
        canvas.oldZoom = canvas.zoom;
        canvas.zoom += zoomChange;
        handleZoomChange();
        canvas.scale /= pow(2.0, zoomChange);
    }

    canvas.rotx += (canvas.destRotx - canvas.rotx) * 0.03;
    if (canvas.viewMode == VIEW_2D) {
        canvas.rotz /= 1.03;
    } else {
        canvas.rotz += canvas.drotz;
        canvas.drotz *= 0.9;
        if (canvas.rotz > 360) {
            canvas.rotz -= 360;
        }
        if (canvas.rotz < 0) {
            canvas.rotz += 360;
        }
    }

}

void tileEngineZoomIn() {
    if (SDL_GetModState() & KMOD_SHIFT) {
        canvas.oldScale = canvas.scale;
        canvas.scale *= 1.005;
    } else {
        if (canvas.destScale == 0.0) {
            if (canvas.scale <= 0.5) {
                canvas.destScale = 1.0;
            } else {
                canvas.destScale = 2.0;
            }
        }
    }
}

void tileEngineZoomOut() {
    if (SDL_GetModState() & KMOD_SHIFT) {
        canvas.oldScale = canvas.scale;
        canvas.scale /= 1.005;
    } else {
        if (canvas.destScale == 0.0) {
            if (canvas.scale >= 2.0) {
                canvas.destScale = 1.0;
            } else {
                canvas.destScale = 0.5;
            }
        }
    }
}

void tileEngineGotomypos() {
    if (mouse.oldButton == 0) {
        followingMypos = 1 - followingMypos;
    }
}

void tileEngineViewMode2D() {
    view2d -> status = UI_HIDDEN;
    view3d -> status = UI_SHOWN;
    canvas.destRotx = 0.0;
    canvas.viewMode = VIEW_2D;
}

void tileEngineViewMode3D() {
    view2d -> status = UI_SHOWN;
    view3d -> status = UI_HIDDEN;
    canvas.destRotx = 23.5;
    canvas.viewMode = VIEW_3D;
}

void shutdownTileEngine() {
    int i;
    for (i = 0; i < DOWNLOADING_THREADS; i++) {
        //        fprintf(stderr, "waiting for download thread %d to finish...\n", SDL_GetThreadID(downloadThreads[i]));
        SDL_WaitThread(downloadThreads[i], NULL);
    }
}

void runTests() {
    int i;
    fprintf(stderr, "tileEngine.runTests()\n");

    TileCoordinate tc;
    WorldCoordinate wc;

    wc.latitude = 50;
    wc.longitude = 20;
    for (i = 0; i < 5; i++) {
        toTileCoordinate(&wc, &tc, 12 + i);
        toWorldCoordinate(&tc, &wc);

        fprintf(stderr, "tilex = %d, tiley = %d, x = %.4f, y= %.4f, zoom=%d\n", tc.tilex, tc.tiley, tc.x, tc.y, tc.zoom);
        fprintf(stderr, "lat = %f, lon = %f\n", wc.latitude, wc.longitude);
    }
}
