/*!
** @file	FocusButton.cpp
**
** @brief	Custom button widget for focus mode
**
*/
/*---------------------------------------------------------------------------
** Includes
*/
#include "focusbutton.h"
#include "ClassicPrintSettings.h"

/*---------------------------------------------------------------------------
** Defines and Macros
*/

/*---------------------------------------------------------------------------
** Typedefs
*/
//!
//! Structure that contains information about a focus mode
//!
struct FOCUSBUTTON_MODE {
	CAMERA_FOCUS_MODE	mode;
	const char*			icon_path;
};

/*---------------------------------------------------------------------------
** Local function prototypes
*/

/*---------------------------------------------------------------------------
** Data
*/
static FOCUSBUTTON_MODE	focusbutton_modes[] = {
	{ CAMERA_FOCUS_MODE_AUTO, "camera_auto_mode.png" },
	{ CAMERA_FOCUS_MODE_FIXED, "camera_scene_mode_landscape.png" }
};

static const int focusbutton_num_modes = sizeof(focusbutton_modes) / sizeof(FOCUSBUTTON_MODE);

//--------------------------------------------------------------------------
/*!
** @brief   	Constructor
*/
FocusButton::FocusButton(QWidget* parent) : QToolButton(parent) {
	m_focus_mode = 0;

	// Set the icon for the current mode
	QString icon(ClassicPrintSettings::config_dir() + QString("/icons/") +
				 focusbutton_modes[m_focus_mode].icon_path);
	setIcon(QIcon(icon));

	// Handle button presses
	connect(this, SIGNAL(clicked(bool)), this, SLOT(clicked(bool)));
}

//--------------------------------------------------------------------------
/*!
** @brief   	Destructor
*/
FocusButton::~FocusButton() {
}
	
//--------------------------------------------------------------------------
/*!
** @brief		Slot that is called when the user selects the button
**
*/
void FocusButton::clicked(bool checked) {
	Q_UNUSED(checked)

	// Loop around the focus modes
	if (++m_focus_mode == focusbutton_num_modes) {
		m_focus_mode = 0;
	}
	// Update the icon
	QString icon(ClassicPrintSettings::config_dir() + QString("/icons/") +
				 focusbutton_modes[m_focus_mode].icon_path);
	setIcon(QIcon(icon));

	// Tell any interested parties about the change
	emit focus_mode_changed(focusbutton_modes[m_focus_mode].mode);
}

//--------------------------------------------------------------------------
/*!
** @brief		Set the button to display the current focus mode
**
** @param[In]	New focus mode
**
*/
void FocusButton::set_focus_mode(CAMERA_FOCUS_MODE new_focus_mode) {
	m_focus_mode = new_focus_mode;
	// Update the icon
	QString icon(ClassicPrintSettings::config_dir() + QString("/icons/") +
				 focusbutton_modes[m_focus_mode].icon_path);
	setIcon(QIcon(icon));
}

