/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class MovieWindow : StackableWindow {
	private MovieMenu menu;
	private Gdk.Pixbuf no_poster;
	private MoviePoster.Factory poster_factory;
	private HBox landscape;
	private VBox portrait;
	private Image image;
	private Label label;
	private bool portrait_mode;

	public MovieWindow.with_movie (Movie movie, MovieListStore store) {
		set_title (movie.title);

		// View menu
		menu = new MovieMenu (movie, store, this);

		set_main_menu (menu);

		// Poster
		image = new Image ();

		if (movie.poster != null && movie.poster.pixbuf != null) {
			image.pixbuf = movie.poster.pixbuf;
		} else {
			movie.notify.connect (this.on_movie_changed);
			if (movie.poster != null && movie.poster.thumbnail != null) {
				// FIXME
				image.pixbuf = movie.poster.thumbnail.scale_simple (244, 400, Gdk.InterpType.BILINEAR);
			} else {
				// FIXME
				if (no_poster == null) try {
					no_poster = new Gdk.Pixbuf.from_file ("/usr/share/icons/hicolor/124x124/hildon/general_video.png");
				} catch (Error e) {
					critical ("Missing general_video icon: %s\n", e.message);
				}
				image.pixbuf = no_poster;
			}
			try {
				poster_factory = MoviePoster.Factory.get_instance ();
				poster_factory.queue (movie, receive_poster);
			} catch (Error e) {
				warning ("Failed to queue poster request: %s\n", e.message);
			}
		}

		// Text area
		string genres = movie.genres.to_string ();
		string year = movie.year > 0 ? " (%d)".printf (movie.year) : "";
		string text = "<b>%s</b>%s".printf (genres, year);

		label = new Label (text);
		label.wrap = true;
		label.use_markup = true;
		label.set_alignment (0.0f, 0.0f);

		landscape = new HBox (false, 0);
		portrait = new VBox (false, 0);

		var vbox = new VBox (false, 0);
		vbox.pack_start (landscape, true, true, MARGIN_DOUBLE);
		vbox.pack_start (portrait, true, true, MARGIN_DOUBLE);

		portrait_mode = CinaestProgram.orientation.portrait;
		if (portrait_mode) {
			portrait.pack_start (image, false, false, 0);
			portrait.pack_start (label, true, true, MARGIN_DOUBLE);
		} else {
			landscape.pack_start (image, false, true, 0);
			landscape.pack_start (label, true, true, MARGIN_DOUBLE);
		}

		vbox.show_all ();
		add (vbox);

		// Connect signals
		menu.movie_deleted.connect (() => { destroy (); });
		Gdk.Screen.get_default ().size_changed.connect (on_orientation_changed);
	}

	private void receive_poster (Gdk.Pixbuf pixbuf, Movie movie) {
		var poster = new Poster();

		poster.pixbuf = pixbuf;
		if (movie.poster != null)
			poster.thumbnail = movie.poster.thumbnail;
		movie.poster = poster;
 	}

	private void on_movie_changed (GLib.Object source, GLib.ParamSpec spec) {
		var movie = (Movie) source;

		if ((spec.name == "poster") && (movie.poster != null) && (movie.poster.pixbuf != null)) {
			image.pixbuf = movie.poster.pixbuf;
		}
	}

	private void on_orientation_changed (Gdk.Screen screen) {
		if (CinaestProgram.orientation.portrait == portrait_mode)
			return;

		portrait_mode = CinaestProgram.orientation.portrait;
		if (portrait_mode) {
			landscape.remove (label);
			landscape.remove (image);
			portrait.pack_start (image, false, false, 0);
			portrait.pack_start (label, true, true, MARGIN_DOUBLE);
		} else {
			portrait.remove (label);
			portrait.remove (image);
			landscape.pack_start (image, false, true, 0);
			landscape.pack_start (label, true, true, MARGIN_DOUBLE);
		}
	}
}

