abstract class LineParser {
        internal unowned IMDbSqlite sqlite;

	public LineParser (IMDbSqlite _sqlite) {
		sqlite = _sqlite;
	}

	public abstract void parse_line (string line);

	internal bool skip_title (string title) {
		if (title.has_suffix ("(TV)")) {
			return true;
		}
		if (title.has_suffix ("(V)")) {
			return true;
		}
		if (title.has_suffix ("(VG)")) {
			return true;
		}
		return false;
	}
}

class MovieLineParser : LineParser {
	Regex re_movie;

	public MovieLineParser (IMDbSqlite _sqlite) {
		base (_sqlite);
		try {
			re_movie = new Regex ("^([^\t]+)\t+([0-9]+)$");
		} catch (RegexError e) {
			critical ("Failed to initialize regex: %s\n", e.message);
		}
	}

	public override void parse_line (string line) {
		MatchInfo matchinfo;

		// Skip series episodes
		if (line[0] == '"')
			return;

		if (!re_movie.match(line, 0, out matchinfo))
			return;

		string title;
		string year = matchinfo.fetch (2);
	        try {
			title = convert(matchinfo.fetch (1), -1, "utf-8", "latin1");
		} catch (ConvertError e) {
			return;
		}

		if (skip_title (title))
			return;

		sqlite.add_movie (title, year.to_int ());
	}
}

class GenreLineParser : LineParser {
	Regex re_genre;

	public GenreLineParser (IMDbSqlite _sqlite) {
		base (_sqlite);
		try {
			re_genre = new Regex ("^([^\t]+)\t+([A-Za-z-]+)$");
		} catch (RegexError e) {
			critical ("Failed to initialize regex: %s\n", e.message);
		}
	}

	public override void parse_line (string line) {
		MatchInfo matchinfo;

		// Skip series episodes
		if (line[0] == '"')
			return;

		if (!re_genre.match(line, 0, out matchinfo))
			return;

		string title;
		string genre = matchinfo.fetch (2);
	        try {
			title = convert(matchinfo.fetch (1), -1, "utf-8", "latin1");
		} catch (ConvertError e) {
			return;
		}

		sqlite.movie_add_genre (title, genre);
	}
}

class RatingLineParser : LineParser {
	Regex re_rating;

	public RatingLineParser (IMDbSqlite _sqlite) {
		base (_sqlite);
		try {
			re_rating = new Regex ("^      .+ +([0-9]+) +([0-9.]+) +(.+)$");
		} catch (RegexError e) {
			critical ("Failed to initialize regex: %s\n", e.message);
		}
	}

	public override void parse_line (string line) {
		MatchInfo matchinfo;

		// Skip series episodes
		if (line[0] == '"')
			return;

		if (!re_rating.match(line, 0, out matchinfo))
			return;

		string title;
		string votes = matchinfo.fetch (1);
		string rating = matchinfo.fetch (2);
	        try {
			title = convert(matchinfo.fetch (3), -1, "utf-8", "latin1");
		} catch (ConvertError e) {
			return;
		}

		// Skip series episodes
		if (title[0] == '"')
			return;

		if (skip_title (title))
			return;

		sqlite.movie_set_rating (title, (int) (rating.to_double () * 10), votes.to_int ());
	}
}
