/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

class GooglePlugin : Plugin {
	List<MovieSource> sources;
	List<string> locations;
	string last_location;

	public override void hello (Gtk.Window window, Osso.Context context) {
		stdout.printf ("Google Plugin Loaded.\n");

		var source = new GoogleSource ();

		sources = new List<MovieSource> ();
		sources.append (source);

		locations = null;
		try {
			var config_file = Path.build_filename (Environment.get_user_config_dir (), "cinaest", "cinaest.cfg");
			var keyfile = new KeyFile ();
			if (keyfile.load_from_file (config_file, KeyFileFlags.NONE)
			    && keyfile.has_group ("GooglePlugin")) {
				if (keyfile.has_key ("GooglePlugin", "KnownLocations")) {
					var l = keyfile.get_string_list ("GooglePlugin", "KnownLocations");
					for (int i = 0; i < l.length; i++)
						locations.append (l[i]);
				}
				if (keyfile.has_key ("GooglePlugin", "LastLocation")) {
					source.location = last_location = keyfile.get_string ("GooglePlugin", "LastLocation");
				}
			}
		} catch (Error e) {
			if (!(e is KeyFileError.NOT_FOUND))
				stdout.printf ("Error loading configuration: %s\n", e.message);
		}

		// FIXME - this forces the inclusion of config.h
		(void) Config.GETTEXT_PACKAGE;
	}

	public override unowned List<MovieSource> get_sources () {
		return sources;
	}

	public override List<MovieAction> get_actions (Movie _movie, Gtk.Window window) {
		List<MovieAction> list = null;
		var movie = _movie as GoogleMovie;

		if (movie != null) {
			list.append (new MovieAction (_("Add to calendar"), on_add_calendar_event, movie, window));
		}

		return list;
	}

	private void on_add_calendar_event (Movie _movie, Gtk.Window window) {
		var movie = (GoogleMovie) _movie;

		var dialog = new PickerDialog (window);
		dialog.set_title (_("Add showtime to calendar"));

		var selector = new TouchSelector.text ();
		var showtimes = movie.showtimes.split (", ");
		foreach (string s in showtimes) {
			selector.append_text (s);
		}
		dialog.set_selector (selector);

		var res = dialog.run ();
		if (res == ResponseType.OK) {
			string s = selector.get_current_text ();
                        int hour = s.to_int ();
                        int min = s.str (":").offset (1).to_int ();

			var showtime = time_t ();
			var timeinfo = Time.local (showtime);
			timeinfo.second = 0;
			timeinfo.minute = min;
			timeinfo.hour = hour;
			showtime = timeinfo.mktime ();

			int runtime = 3600 * movie.runtime.to_int () + 60 * movie.runtime.str ("hr ").offset (3).to_int ();
			if (runtime == 0) {
				// Default to 120min if we failed to parse the runtime
				runtime = 7200;
			}

			res = Calendar.add_event (movie.title, _("Movie"), movie.cinema, showtime, showtime + runtime);
			var banner = (Banner) Banner.show_information_with_markup (window, null, (res == 0) ?
			                                                           _("Added calendar event at %d:%02d").printf (hour, min) :
			                                                           _("Failed to add calendar event"));
			banner.set_timeout (1500);
		}
		dialog.destroy ();
	}

	public override void settings_dialog (Gtk.Window window) {
		GoogleSource source = (GoogleSource) sources.data;
		var dialog = new Gtk.Dialog ();
		dialog.set_transient_for (window);
		dialog.set_title (_("Google plugin settings"));

		var selector = new TouchSelectorEntry.text ();
		insert_location_sorted (source.location);
		foreach (string l in locations)
			selector.append_text (l);

		var button = new PickerButton (SizeType.FINGER_HEIGHT, ButtonArrangement.HORIZONTAL);
		button.set_title (_("Location"));
		button.set_selector (selector);
		button.set_value (source.location);

		var content = (VBox) dialog.get_content_area ();
		content.pack_start (button, true, true, 0);

		dialog.add_button ("Done", ResponseType.ACCEPT);

		dialog.show_all ();
		int res = dialog.run ();
		if (res == ResponseType.ACCEPT) {
			source.location = button.get_value ();
			if (insert_location_sorted (source.location) || source.location != last_location) {
				var config_dir = Path.build_filename (Environment.get_user_config_dir (), "cinaest");
				var config_file = Path.build_filename (config_dir, "cinaest.cfg");

				// Make sure the directory is available
				DirUtils.create_with_parents (config_dir, 0770);

				var keyfile = new KeyFile ();
				try {
					keyfile.load_from_file (config_file, KeyFileFlags.NONE);
				} catch (Error e) {
					if (!(e is KeyFileError.NOT_FOUND))
						stdout.printf ("Error loading configuration: %s\n", e.message);
				}
				var l = new string[locations.length ()];
				for (int i = 0; i < l.length; i++) {
					l[i] = locations.nth_data (i);
				}
				keyfile.set_string_list ("GooglePlugin", "KnownLocations", l);
				keyfile.set_string ("GooglePlugin", "LastLocation", source.location);
				last_location = source.location;

				try {
					var file = File.new_for_path (config_file + ".part");
					var stream = file.create (FileCreateFlags.REPLACE_DESTINATION, null);
					var data = keyfile.to_data ();

					stream.write (data, data.length, null);
					FileUtils.rename (config_file + ".part", config_file);
				} catch (Error e) {
					stdout.printf ("Failed to store configuration: %s\n", e.message);
				}
			}
		}
		dialog.destroy ();
	}

	private bool insert_location_sorted (string? location) {
		if (location == null)
			return false;
		if (locations != null) {
			for (unowned List<string> l = locations.first (); l != null; l = l.next) {
				if (l.data == location) {
					return false;
				}
				if (l.data > location) {
					l.insert (location, 0);
					return true;
				}
			}
		}
		locations.append (location);
		return true;
	}

	public override unowned string get_name () {
		return "Google";
	}
}

class GoogleSource : MovieSource {
	public string location;
	public string description;

	public override async void get_movies (MovieFilter filter, MovieSource.ReceiveMovieFunction callback, int limit, Cancellable? cancellable) {
		var parser = new GoogleParser ();

		yield parser.query (filter, location, callback, cancellable);
		if (location == null) {
			location = parser.location;
		}
	}

	public override void add_movie (Movie movie) {
	}

	public override void delete_movie (Movie movie) {
	}

	public override unowned string get_name () {
		return _("Google");
	}

	public override unowned string get_description () {
		if (location != null && location != "") {
			description = _("Movie Showtimes near %s").printf (location);
		} else {
			description =  _("Movie Showtimes");
		}
		return description;
	}

	public override bool get_editable () {
		return false;
	}
}

[ModuleInit]
public Type register_plugin () {
	// types are registered automatically
	return typeof (GooglePlugin);
}
