/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class MovieListWindow : StackableWindow {
	private MovieListMenu menu;
	private Hildon.Entry search_field;
	private Toolbar search_bar;
	private uint source_id;
	private MovieListView movie_list;
	private MovieFilter filter;
	private MovieListStore store;
	private Label no_movies;
	private bool search_bar_visible;

	construct {
		// View menu
		menu = new MovieListMenu (this);

		set_main_menu (menu);

		// Search bar
		search_field = new Hildon.Entry (SizeType.FINGER_HEIGHT);

		var search_field_item = new ToolItem ();
		search_field_item.set_expand (true);
		search_field_item.add (search_field);

		var close_image = new Image.from_file("/usr/share/icons/hicolor/48x48/hildon/general_close.png");
		var close_button = new ToolButton (close_image, _("Close"));

		search_bar = new Toolbar ();
		search_bar.insert (search_field_item, 0);
		search_bar.insert (close_button, 1);

		add_toolbar (search_bar);

		// Movie list - connected to menu for sorting
		movie_list = new MovieListView ();
		menu.sortable = movie_list.sorted_store;
		store = movie_list.store;

		no_movies = new Label (_("No movies"));
		Hildon.helper_set_logical_font (no_movies, "LargeSystemFont");
		Hildon.helper_set_logical_color (no_movies, RcFlags.FG, StateType.NORMAL, "SecondaryTextColor");
		no_movies.set_size_request (-1, 6 * 70);
		no_movies.set_alignment ((float) 0.5, (float) 0.42);

		var vbox = new VBox (false, 0);
		vbox.pack_start (movie_list, true, true, 0);
		vbox.pack_start (no_movies, false, false, 0);

		add (vbox);

		// Connect signals
		search_field.changed.connect (on_search_field_changed);
		close_button.clicked.connect (on_close_button_clicked);
		key_press_event.connect (on_key_press_event);
		movie_list.movie_activated.connect (on_movie_activated);

		store.notify["update-running"].connect (on_update_running_changed);

		search_field.set_flags (WidgetFlags.CAN_DEFAULT);
		search_field.grab_default ();

		show_all ();
		search_bar_visible = false;
		search_bar.hide ();
		movie_list.hide ();

		filter = new MovieFilter ();
	}

	public MovieSource source {
		get { return store.source; }
		set {
			store.source = value;
			menu.source = value;
			set_title ("Cinæst: " + value.get_description ());
			filter.title = search_field.get_text ();
			if (store.start_search (filter)) {
				movie_list.show ();
				no_movies.hide ();
			}
		}
	}

	private void on_close_button_clicked () {
		search_field.set_text ("");
		search_bar_visible = false;
		search_bar.hide ();
	}

	private bool on_key_press_event (Widget widget, Gdk.EventKey event) {
		if (event.str != "") {
			if (!search_bar_visible) {
				search_bar_visible = true;
				search_bar.show ();
			}
			search_field.grab_focus ();
		}

		return false;
	}

	private void on_search_field_changed () {
		// With every change we reset the timer to 500ms
		if (source_id != 0) {
			Source.remove (source_id);
		}
		source_id = Timeout.add (500, start_search);
	}

	private bool start_search () {
		filter.title = search_field.get_text ();
		if (store.start_search (filter)) {
			movie_list.show ();
			no_movies.hide ();
		}

		// One-shot only
		return false;
	}

	private void on_movie_activated (Movie movie) {
		var window = new MovieWindow.with_movie (movie);
		window.show_all ();

	}

	private void on_update_running_changed (GLib.Object source, ParamSpec spec) {
		TreeIter iter;

		Hildon.gtk_window_set_progress_indicator (this, (int) store.update_running);
		// Update finished, but store still empty?
		if (!store.update_running && !store.get_iter_first (out iter)) {
			movie_list.hide ();
			no_movies.show ();
		}
	}
}

