/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class MovieWindow : StackableWindow {
	private MovieMenu menu;
	private Gdk.Pixbuf no_poster;
	private MoviePoster.Factory poster_factory;
	private HBox hbox;
	private Image image;
	private VBox details;
	private PannableArea pannable;
	private Label plot;
	private bool portrait_mode;

	public MovieWindow.with_movie (Movie movie, MovieListStore store) {
		set_title (movie.title);

		// View menu
		menu = new MovieMenu (movie, store, this);

		set_main_menu (menu);

		// Poster
		image = new Image ();

		if (movie.poster != null && movie.poster.pixbuf != null) {
			image.pixbuf = movie.poster.pixbuf;
		} else {
			movie.notify.connect (this.on_movie_changed);
			if (movie.poster != null && movie.poster.thumbnail != null) {
				// FIXME
				image.pixbuf = movie.poster.thumbnail.scale_simple (268, 424, Gdk.InterpType.BILINEAR);
			} else {
				// FIXME
				if (no_poster == null) try {
					no_poster = new Gdk.Pixbuf.from_file ("/usr/share/icons/hicolor/124x124/hildon/general_video.png");
				} catch (Error e) {
					critical ("Missing general_video icon: %s\n", e.message);
				}
				image.pixbuf = no_poster;
			}
			try {
				poster_factory = MoviePoster.Factory.get_instance ();
				poster_factory.queue (movie, receive_poster);
			} catch (Error e) {
				warning ("Failed to queue poster request: %s\n", e.message);
			}
		}

		// Text area
		string year = movie.year > 0 ? " (%d)".printf (movie.year) : "";
		string text = "<big><b>%s</b></big>%s\n<small>%s</small>".printf (movie.title, year, movie.secondary);
		var label = new Label (text);
		label.wrap = true;
		label.use_markup = true;
		label.set_alignment (0.0f, 0.0f);

		var header = new HBox (false, 0);
		header.pack_start (label, true, true, 0);
		if (movie.rating > 0) {
			text = "<big><b>%d.%d</b></big>".printf (movie.rating / 10, movie.rating % 10);
			var rating = new Label (text);
			rating.use_markup = true;
			rating.set_alignment (0.5f, 0.0f);
			header.pack_start (rating, false, false, MARGIN_DOUBLE);
		}

		plot = new Label (movie.get_plot ());
		plot.wrap = true;
		plot.set_alignment (0.0f, 0.0f);

		details = new VBox (false, MARGIN_DOUBLE);
		details.pack_start (header, false, false, 0);
		details.pack_start (plot, false, false, 0);

		var pannable = new PannableArea ();
		var eventbox = new EventBox ();
		eventbox.add (details);
		eventbox.above_child = true;
		pannable.add_with_viewport (eventbox);

		hbox = new HBox (false, 0);
		hbox.pack_start (pannable, true, true, 0);

		portrait_mode = CinaestProgram.orientation.portrait;
		if (portrait_mode) {
			details.pack_start (image, false, false, 0);
			details.reorder_child (image, 0);
			plot.set_size_request (480 - 2 * MARGIN_DOUBLE, -1);
		} else {
			hbox.pack_start (image, false, false, MARGIN_DOUBLE);
			hbox.reorder_child (image, 0);
			plot.set_size_request (800 - 268 /* image */ - 3 * MARGIN_DOUBLE, -1);
			pannable.set_size_request (-1, 424);
		}

		hbox.show_all ();
		add (hbox);

		// Connect signals
		menu.movie_deleted.connect (() => { destroy (); });
		Gdk.Screen.get_default ().size_changed.connect (on_orientation_changed);
	}

	private void receive_poster (Gdk.Pixbuf pixbuf, Movie movie) {
		var poster = new Poster();

		poster.pixbuf = pixbuf;
		if (movie.poster != null)
			poster.thumbnail = movie.poster.thumbnail;
		movie.poster = poster;
 	}

	private void on_movie_changed (GLib.Object source, GLib.ParamSpec spec) {
		var movie = (Movie) source;

		if ((spec.name == "poster") && (movie.poster != null) && (movie.poster.pixbuf != null)) {
			image.pixbuf = movie.poster.pixbuf;
		}
	}

	private void on_orientation_changed (Gdk.Screen screen) {
		if (CinaestProgram.orientation.portrait == portrait_mode)
			return;

		portrait_mode = CinaestProgram.orientation.portrait;
		if (portrait_mode) {
			hbox.remove (image);
			details.pack_start (image, false, false, 0);
			details.reorder_child (image, 0);
			plot.set_size_request (480 - 2 * MARGIN_DOUBLE, -1);
			pannable.set_size_request (-1, -1);
		} else {
			details.remove (image);
			hbox.pack_start (image, false, false, MARGIN_DOUBLE);
			hbox.reorder_child (image, 0);
			pannable.set_size_request (-1, 424);
			plot.set_size_request (800 - 268 /* image */ - 3 * MARGIN_DOUBLE, -1);
		}
	}
}

