/*
 *  charmap - GTK+ based simple character map
 *
 *  Copyright (C) 2014 Peter Pichler (this application)
 *  Copyrught (C) 2010 Gabriel Schulhof (the author of scv-reader,
 *                     a Maemo package this app was partially based on)
 *  Copyright (C) 2010 The GNOME Charcter Map project (the icon)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to Free Software Foundation Inc.,
 *  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <hildon/hildon.h>
#include <gucharmap/gucharmap.h>
#include <string.h>
#include <math.h>

#include "maemo-gucharmap-chartable.h"

#define APP_NAME    "Character Map"
#define APP_VERSION "1.0.3"

typedef enum
{
  CHARMAP_DIALOG_SORT_BY_SCRIPT,
  CHARMAP_DIALOG_SORT_BY_CODE_BLOCK
} CharmapDialogSortMode;

static GtkButton *btn_chapter   = NULL;
static GtkButton *btn_font_face = NULL;
static GtkButton *btn_font_size = NULL;

static GtkLabel *lbl_character   = NULL;
static GtkLabel *lbl_unicode     = NULL;
static GtkLabel *lbl_description = NULL;

static GtkEditable *edit_result = NULL;
static int sel_start, sel_end;

static GtkButton *btn_copy  = NULL;
static GtkButton *btn_clear = NULL;

static MaemoGucharmapChartable *chart = NULL;

#define BUTTON_POSITION_TOLERANCE  10.0
#define BUTTON_POSITION_RESET      -(2 * BUTTON_POSITION_TOLERANCE)

static double chart_button_pressed_x = BUTTON_POSITION_RESET;
static double chart_button_pressed_y = BUTTON_POSITION_RESET;

static const struct font_size_list
{
  int point_size;   // font's point size
  double cell_size; // size of cells in chart
  int columns;      // number of cells this font size fits on screen
} font_size_list[] =
{
  { 16, 42, 12 },
  { 18, 47, 11 },
  { 20, 52.5, 10 },
  { 22, 60, 9 },
  { 25, 70, 8 }, // this is the default
  { 29, 84, 7 },
  { 34, 84, 6 },
  { 40, 105, 5 },
  { 50, 140, 4 },
  { 67, 210, 3 },
  { 100, 420, 2 },
  { 300, 420, 1 },
};

static PangoFontFamily **font_face_list = NULL;

#define DEFAULT_FONT_FACE  "Nokia Sans"
#define DEFAULT_FONT_SIZE_INDEX 4
#define DEFAULT_FONT_SIZE  font_size_list[DEFAULT_FONT_SIZE_INDEX].point_size
#define DEFAULT_CELL_SIZE  font_size_list[DEFAULT_FONT_SIZE_INDEX].cell_size

static GtkLabel *make_label (gint width)
{
  GtkLabel *lbl = g_object_new(GTK_TYPE_LABEL, "visible", TRUE, NULL);
  gtk_widget_set_size_request(GTK_WIDGET(lbl), width, -1);
  gtk_label_set_line_wrap(lbl, TRUE);
  gtk_label_set_line_wrap_mode(lbl, PANGO_WRAP_WORD_CHAR);
  gtk_label_set_use_markup(lbl, TRUE);
  return lbl;
}

static void sort_mode_changed (HildonPickerButton *btn, void *null)
{
  GucharmapChaptersModel *model = 
    (CHARMAP_DIALOG_SORT_BY_SCRIPT == hildon_picker_button_get_active(btn))
      ? gucharmap_script_chapters_model_new()
      : gucharmap_block_chapters_model_new();
  hildon_touch_selector_set_model(hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(btn_chapter)), 0, GTK_TREE_MODEL(model));

  HildonTouchSelector *sel = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(btn_chapter));
  GtkTreeIter itr;
  gunichar new_char = g_utf8_get_char_validated("", -1);

  if (gucharmap_chapters_model_character_to_iter(model, new_char, &itr))
  {
    hildon_touch_selector_select_iter(sel, 0, &itr, FALSE);
    gucharmap_chartable_set_codepoint_list(GUCHARMAP_CHARTABLE(chart),
        gucharmap_chapters_model_get_codepoint_list(model, &itr));
    gucharmap_chartable_set_active_character(GUCHARMAP_CHARTABLE(chart), new_char);
  }
}

static GtkButton *make_sort_picker (void)
{
  HildonTouchSelector *sel = HILDON_TOUCH_SELECTOR(hildon_touch_selector_new_text());
  hildon_touch_selector_append_text(sel, "Script");
  hildon_touch_selector_append_text(sel, "Unicode Block");

  GtkButton *btn = g_object_new(HILDON_TYPE_PICKER_BUTTON,
      "visible",        TRUE,
      "arrangement",    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "style",          HILDON_BUTTON_STYLE_PICKER,
      "title",          "Sort Mode",
      "touch-selector", sel,
      NULL);
  g_signal_connect(G_OBJECT(btn), "value-changed", (GCallback)sort_mode_changed, NULL);
  hildon_picker_button_set_active(HILDON_PICKER_BUTTON(btn), CHARMAP_DIALOG_SORT_BY_CODE_BLOCK);

  gtk_widget_show(GTK_WIDGET(btn));
  return btn;
}

static void chapter_changed (HildonPickerButton *btn, GtkWidget *wnd)
{
  HildonTouchSelector *sel = hildon_picker_button_get_selector(btn);
  GtkTreeModel *tm = hildon_touch_selector_get_model(sel, 0);
  GucharmapChaptersModel *model = GUCHARMAP_CHAPTERS_MODEL(tm);
  GtkTreeIter itr;

  if (hildon_touch_selector_get_selected(sel, 0, &itr))
    gucharmap_chartable_set_codepoint_list(GUCHARMAP_CHARTABLE(chart),
        gucharmap_chapters_model_get_codepoint_list(model, &itr));
}

static GtkButton *make_chapter_picker (GtkWidget *wnd)
{
  HildonTouchSelector *sel = g_object_new(HILDON_TYPE_TOUCH_SELECTOR, NULL);
  GtkListStore *ls = gtk_list_store_new(1, G_TYPE_STRING);
  HildonTouchSelectorColumn *col = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(sel), GTK_TREE_MODEL(ls), gtk_cell_renderer_text_new(), "text", 1, NULL);
  g_object_set(col, "text-column", 1, NULL);
  g_object_unref(ls);

  GtkButton *btn = g_object_new(HILDON_TYPE_PICKER_BUTTON,
      "visible",        TRUE,
      "arrangement",    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "style",          HILDON_BUTTON_STYLE_PICKER,
      "title",          "Chapter",
      "touch-selector", sel,
      NULL);
  g_signal_connect(G_OBJECT(btn), "value-changed", (GCallback)chapter_changed, wnd);

  gtk_widget_show(GTK_WIDGET(btn));
  return btn;
}

static void font_changed (HildonPickerButton *ignore, void *null)
{
  const int face_idx = hildon_picker_button_get_active(HILDON_PICKER_BUTTON(btn_font_face));
  const char *font_face = face_idx != -1 && font_face_list != NULL
      ? pango_font_family_get_name(font_face_list[face_idx])
      : DEFAULT_FONT_FACE;

  const int size_idx = hildon_picker_button_get_active(HILDON_PICKER_BUTTON(btn_font_size));
  const int font_size = size_idx != -1
      ? font_size_list[size_idx].point_size
      : DEFAULT_FONT_SIZE;
  const double cell_size = size_idx != -1
      ? font_size_list[size_idx].cell_size
      : DEFAULT_CELL_SIZE;

  maemo_gucharmap_chartable_set_font(chart, font_face, font_size);
  maemo_gucharmap_chartable_set_scroll_adjustments(chart, cell_size);
}

static int compare_fonts (PangoFontFamily **p_font1, PangoFontFamily **p_font2, gpointer null)
{
  const char *str1 = pango_font_family_get_name(*p_font1);
  const char *str2 = pango_font_family_get_name(*p_font2);
  return strcmp(str1, str2);
}

static GtkButton *make_font_face_picker (void)
{
  HildonTouchSelector *sel = HILDON_TOUCH_SELECTOR(hildon_touch_selector_new_text());

  PangoContext *pctx = gdk_pango_context_get();
  int i, n = 0, idx = 0;
  pango_context_list_families(pctx, &font_face_list, &n);
  g_qsort_with_data(font_face_list, n, sizeof *font_face_list, (GCompareDataFunc)compare_fonts, NULL);
  for (i = 0; i < n; ++i)
  {
    const char *str = pango_font_family_get_name(font_face_list[i]);
    hildon_touch_selector_append_text(sel, str);

    if (strcmp(str, DEFAULT_FONT_FACE) == 0)
      idx = i;
  }

  GtkButton *btn = g_object_new(HILDON_TYPE_PICKER_BUTTON,
      "visible",        TRUE,
      "arrangement",    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "style",          HILDON_BUTTON_STYLE_PICKER,
      "title",          "Font Face",
      "touch-selector", sel,
      NULL);
  g_signal_connect(G_OBJECT(btn), "value-changed", (GCallback)font_changed, NULL);
  hildon_picker_button_set_active(HILDON_PICKER_BUTTON(btn), idx);

  gtk_widget_show(GTK_WIDGET(btn));
  return btn;
}

static GtkButton *make_font_size_picker (void)
{
  HildonTouchSelector *sel = HILDON_TOUCH_SELECTOR(hildon_touch_selector_new_text());

  int i, idx = 0;
  for (i = 0; i < G_N_ELEMENTS(font_size_list); ++i)
  {
    char str[18];
    sprintf(str, "%d  (%d column%s)",
        font_size_list[i].point_size,
        font_size_list[i].columns,
        font_size_list[i].columns > 1 ? "s" : "");
    hildon_touch_selector_append_text(sel, str);

    if (font_size_list[i].point_size == DEFAULT_FONT_SIZE)
      idx = i;
  }

  GtkButton *btn = g_object_new(HILDON_TYPE_PICKER_BUTTON,
      "visible",        TRUE,
      "arrangement",    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
      "style",          HILDON_BUTTON_STYLE_PICKER,
      "title",          "Font Size",
      "touch-selector", sel,
      NULL);
  g_signal_connect(G_OBJECT(btn), "value-changed", (GCallback)font_changed, NULL);
  hildon_picker_button_set_active(HILDON_PICKER_BUTTON(btn), idx);

  gtk_widget_show(GTK_WIDGET(btn));
  return btn;
}

static void menu_about_selected (GtkWidget * widget, GtkWindow * wnd)
{
  GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file(
      "/usr/share/icons/hicolor/64x64/apps/charmap.png",
      NULL);
  GtkImage *img = g_object_new(GTK_TYPE_IMAGE, "visible", TRUE, NULL);
  gtk_image_set_from_pixbuf(img, pixbuf);
  g_object_unref(pixbuf);

  GtkLabel *lbl = make_label(-1);
  gtk_label_set_justify(lbl, GTK_JUSTIFY_CENTER);
  gtk_label_set_markup(lbl,
      "<span size=\"xx-large\">" APP_NAME " " APP_VERSION "</span>\n"
      "A simple character map for Maemo.\n\n"
      "Copyright © 2014 Peter Pichler\n"
      "(Using parts of code from the Maemo package scv-reader and the icon from the GNOME package gucharmap.)");

  GtkBox *vbox = g_object_new(GTK_TYPE_VBOX, "visible", TRUE, NULL);
  gtk_box_pack_start(vbox, GTK_WIDGET(img), FALSE, FALSE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(lbl), FALSE, FALSE, 0);

  GtkDialog *dlg = g_object_new(GTK_TYPE_DIALOG,
      "visible", TRUE,
      "modal",   TRUE,
      "title",   "About " APP_NAME,
      NULL);
  gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(wnd));
  gtk_container_add(GTK_CONTAINER(dlg->vbox), GTK_WIDGET(vbox));

  gtk_dialog_run(dlg);
  gtk_widget_destroy(GTK_WIDGET(dlg));
}

static void make_menu (GtkWidget *wnd)
{
  btn_chapter = make_chapter_picker(wnd); // must be initialzed before sort picker

  GtkButton *btn_about = g_object_new(HILDON_TYPE_BUTTON,
      "visible", TRUE,
      "title",   "About",
      NULL);
  g_signal_connect(G_OBJECT(btn_about), "clicked", G_CALLBACK(menu_about_selected), wnd);

  GtkWidget *main_menu = g_object_new(HILDON_TYPE_APP_MENU, NULL);
  hildon_app_menu_append(HILDON_APP_MENU(main_menu), make_sort_picker());
  hildon_app_menu_append(HILDON_APP_MENU(main_menu), btn_chapter);
  hildon_app_menu_append(HILDON_APP_MENU(main_menu), btn_font_face = make_font_face_picker());
  hildon_app_menu_append(HILDON_APP_MENU(main_menu), btn_font_size = make_font_size_picker());
  hildon_app_menu_append(HILDON_APP_MENU(main_menu), btn_about);
  hildon_window_set_app_menu(HILDON_WINDOW(wnd), HILDON_APP_MENU(main_menu));
}

static gboolean wnd_delete_event (GtkWidget *charmap, GdkEvent *event, gpointer null)
{
  gtk_main_quit();
  g_free(font_face_list);
  return FALSE;
}

static gboolean save_selection_info (GtkWidget *hpa, GdkEventButton *event, void *null)
{
  gtk_editable_get_selection_bounds(edit_result, &sel_start, &sel_end);
  chart_button_pressed_x = event->x;
  chart_button_pressed_y = event->y;
  return FALSE;
}

static gboolean insert_active_chracter (GtkWidget *hpa, GdkEventButton *event, void *null)
{
  if (  fabs(chart_button_pressed_x - event->x) < BUTTON_POSITION_TOLERANCE
     && fabs(chart_button_pressed_y - event->y) < BUTTON_POSITION_TOLERANCE )
  {
    const gunichar c = gucharmap_chartable_get_active_character(GUCHARMAP_CHARTABLE(chart));
    const GUnicodeType type = gucharmap_unichar_type(c);

    if (type != G_UNICODE_CONTROL && type != G_UNICODE_UNASSIGNED)
    {
      GString *str = g_string_new("");
      g_string_append_unichar(str, c);

      if (sel_start != sel_end)
        gtk_editable_delete_text(edit_result, sel_start, sel_end);
      gtk_editable_insert_text(edit_result, str->str, -1, &sel_start);
      gtk_editable_set_position(edit_result, sel_end = sel_start);

      g_string_free(str, TRUE);
    }
    chart_button_pressed_x = chart_button_pressed_y = BUTTON_POSITION_RESET;
  }
  return FALSE;
}

static void notify_active_character (GucharmapChartable *chart, GParamSpec *pspec, GtkWidget *ndw)
{
  const gunichar c = gucharmap_chartable_get_active_character(chart);
  const GUnicodeType type = gucharmap_unichar_type(c);

  GString *str = g_string_new("");
  g_string_append_unichar(str, c);
  char buffer[36];

  sprintf(buffer, "<span size=\"xx-large\">%s</span>",
     (  type == G_UNICODE_CONTROL
     || type == G_UNICODE_UNASSIGNED )
         ? " "
         : (str->str[0] == '&' && str->str[1] == 0)
             ? "&amp;"
             : str->str);
  gtk_label_set_markup(lbl_character, buffer);
  g_string_free(str, TRUE);

  sprintf(buffer, "U+%04lX", (unsigned long)c);
  gtk_label_set_text(lbl_unicode, buffer);

  const gchar *description = gucharmap_get_unicode_name(c);
  gtk_label_set_text(lbl_description, description);
}

static void btn_copy_clicked (GtkWidget *btn, void *null)
{
  gtk_editable_select_region(edit_result, 0, -1);
  gtk_editable_copy_clipboard(edit_result);
}

static void btn_clear_clicked (GtkWidget *btn, void *null)
{
  gtk_editable_delete_text(edit_result, 0, -1);
}

static void make_wnd (void)
{
  GtkWidget *wnd = g_object_new(HILDON_TYPE_WINDOW, "title", APP_NAME, NULL);
  g_signal_connect(G_OBJECT(wnd), "delete-event", (GCallback)wnd_delete_event, NULL);

  chart = g_object_new(MAEMO_GUCHARMAP_CHARTABLE_TYPE, "visible", TRUE, NULL);
  g_signal_connect(G_OBJECT(chart), "notify::active-character", (GCallback)notify_active_character, wnd);

  GtkWidget *hpa = g_object_new(HILDON_TYPE_PANNABLE_AREA,
      "visible",        TRUE,
      "child",          chart,
      "hovershoot-max", 0,
      "vovershoot-max", 0,
      NULL);
  g_signal_connect(G_OBJECT(hpa), "button-press-event", (GCallback)save_selection_info, NULL);
  g_signal_connect(G_OBJECT(hpa), "button-release-event", (GCallback)insert_active_chracter, NULL);

  const gint w = 168;
  lbl_character = make_label(w);
  lbl_unicode = make_label(w);
  lbl_description = make_label(w);
  edit_result = g_object_new(HILDON_TYPE_ENTRY, "visible", TRUE, NULL);

  btn_copy = g_object_new(HILDON_TYPE_BUTTON,
      "visible", TRUE,
      "size",    HILDON_SIZE_FINGER_HEIGHT,
      "title",   "Copy",
      NULL);
  g_signal_connect(G_OBJECT(btn_copy), "clicked", (GCallback)btn_copy_clicked, NULL);

  btn_clear = g_object_new(HILDON_TYPE_BUTTON,
      "visible", TRUE,
      "size",    HILDON_SIZE_FINGER_HEIGHT,
      "title",   "Clear",
      NULL);
  g_signal_connect(G_OBJECT(btn_clear), "clicked", (GCallback)btn_clear_clicked, NULL);

  GtkBox *vbox = g_object_new(GTK_TYPE_VBOX, "visible", TRUE, NULL);
  gtk_box_pack_start(vbox, GTK_WIDGET(lbl_character),   FALSE, TRUE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(lbl_unicode),     FALSE, TRUE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(lbl_description), FALSE, TRUE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(make_label(w)),   TRUE,  TRUE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(edit_result),     FALSE, TRUE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(btn_copy),        FALSE, TRUE, 0);
  gtk_box_pack_start(vbox, GTK_WIDGET(btn_clear),       FALSE, TRUE, 0);

  GtkBox *hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, NULL);
  gtk_box_pack_start(hbox, GTK_WIDGET(hpa),  TRUE,  TRUE, 0);
  gtk_box_pack_start(hbox, GTK_WIDGET(vbox), FALSE, TRUE, 0);
  gtk_widget_grab_focus(GTK_WIDGET(chart));

  gtk_container_add(GTK_CONTAINER(wnd), GTK_WIDGET(hbox));
  make_menu(wnd);
  gtk_widget_show(wnd);
}

int main (int argc, char** argv)
{
  gtk_init(&argc, &argv);
  make_wnd();
  gtk_main();
  return 0;
}
