#include "charmap.h"
#include "charmapbutton.h"

#include <QApplication>
#include <QMessageBox>

// Make sure this section is after the all other #includes.
// The reason is that gucharmap.h uses 'signals' as an
// identifier, upsetting Qt which uses it as a keyword.
// So the first thing is to #undef signals, but that means
// nothing Qt related can be #included after this point.
#undef signals
extern "C" {
#include <gucharmap/gucharmap.h>
}

int main(int argc, char *argv[])
{
    QApplication app(argc, argv);
    MainWindow mainWindow;
    return app.exec();
}

MainWindow::MainWindow()
    : w(0)  // reset portrait-landscape switch detector
    , labelCharacter(0)
    , labelUnicode(0)
    , labelDescription(0)
    , edit(0)
{
    ui.setupUi(this);

    QApplication::setApplicationName("Character Map");
    QApplication::setApplicationVersion("0.0.1");
    QApplication::setOrganizationName(QApplication::applicationName());
    QApplication::setOrganizationDomain("https://garage.maemo.org/projects/CharMap");

    setWindowIcon(QIcon("/usr/share/icons/hicolor/48x48/apps/vumeter.png"));

#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    setAttribute(static_cast<Qt::WidgetAttribute>(130), true);
#else
    setAttribute(Qt::WA_AutoOrientation, true);
#endif

    connect(ui.buttonCopyPortrait, SIGNAL(clicked()), this, SLOT(buttonCopy_clicked()));
    connect(ui.buttonCopyLandscape, SIGNAL(clicked()), this, SLOT(buttonCopy_clicked()));
    connect(ui.buttonClearPortrait, SIGNAL(clicked()), this, SLOT(buttonClear_clicked()));
    connect(ui.buttonClearLandscape, SIGNAL(clicked()), this, SLOT(buttonClear_clicked()));

#if defined(Q_WS_SIMULATOR)
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showMaximized();
#else
    show();
#endif
}

void MainWindow::resizeEvent(QResizeEvent *event)
{
    QMainWindow::resizeEvent(event);

    const int wNew = event->size().width();
    if (wNew == w)
        return;

    // Layout switched from portrait to landscape or vice-versa
    w = wNew;

    ui.framePortrait->setVisible(w <= 480);
    ui.frameLandscape->setVisible(w > 480);

    removeCharButtons();
    addCharButtons(w > 480 ? 8 : 6);

    const QLabel *oldLabelCharacter = labelCharacter;
    const QLabel *oldLabelUnicode = labelUnicode;
    const QLabel *oldLabelDescription = labelDescription;
    const QLineEdit *oldEdit = edit;

    if (w <= 480)
    {
        labelCharacter = ui.labelCharacterPortrait;
        labelUnicode = ui.labelUnicodePortrait;
        labelDescription = ui.labelDescriptionPortrait;
        edit = ui.editPortrait;
    }
    else
    {
        labelCharacter = ui.labelCharacterLandscape;
        labelUnicode = ui.labelUnicodeLandscape;
        labelDescription = ui.labelDescriptionLandscape;
        edit = ui.editLandscape;
    }

    // Shuffle contents between portrait and landscape
    // Not applicable the first time after initialization
    if (!oldEdit)
        return;

    labelCharacter->setText(oldLabelCharacter->text());
    labelUnicode->setText(oldLabelUnicode->text());
    labelDescription->setText(oldLabelDescription->text());
    edit->setText(oldEdit->text());
    edit->setCursorPosition(oldEdit->cursorPosition());
}

void MainWindow::addCharButtons(uint columns)
{
    uint row = 0;
    uint col = 0;

    for (uint i = 0x20; i <= 0xff; ++i)
    {
        if (!gucharmap_unichar_isdefined(i))
            continue;

        CharMapButton *button = new CharMapButton(ui.scrollAreaWidgetContents, i);
        button->setMaximumSize(72, 16777215);
        button->setText(QString("&") + QChar(i));

        ui.grid->addWidget(button, row, col, 1, 1);
        if (++col == columns)
        {
            ++row;
            col = 0;
        }

        connect(button, SIGNAL(clicked()), button, SLOT(notify()));
        connect(button, SIGNAL(clicked(uint)), this, SLOT(buttonChar_clicked(uint)));
    }
}

void MainWindow::removeCharButtons()
{
    QLayoutItem *child;
    while ((child = ui.grid->takeAt(0)) != 0)
    {
        CharMapButton *button = static_cast<CharMapButton*>(child->widget());

        disconnect(button, SIGNAL(clicked()));
        disconnect(button, SIGNAL(clicked(uint)));

        ui.grid->removeWidget(button);
        delete button;
    }
}

void MainWindow::on_actionAbout_triggered()
{
    QString title("About");
    title += " ";
    title += QApplication::applicationName();

    QString body("<h3>");
    body += QApplication::applicationName();
    body += "</h3> Version ";
    body += QApplication::applicationVersion();
    body += "<p>A very basic character map for Maemo.</p>"
            "&copy; 2014 Peter Pichler</p>"
            "<p>This program is distributed under <a href=\"http://www.gnu.org/licenses/\">GPL3.</a></p>";

    QMessageBox::about(this, title, body);
}

void MainWindow::buttonChar_clicked(uint id)
{
    const QString character(id);

    QString unicode("U+");
    if (id < 0x1000) unicode.append("0");
    if (id < 0x0100) unicode.append("0");
    if (id < 0x0010) unicode.append("0");
    unicode.append(QString::number(id, 16));

    labelCharacter->setText(character);
    labelUnicode->setText(unicode);
    labelDescription->setText(gucharmap_get_unicode_name(id));
    edit->insert(character);
}

void MainWindow::buttonCopy_clicked()
{
    edit->selectAll();
    edit->copy();
}

void MainWindow::buttonClear_clicked()
{
    labelCharacter->clear();
    labelUnicode->clear();
    labelDescription->clear();
    edit->clear();
}
