/*
 * imagecache.c - routines to abstract image caching and asynchronous image
 * loading.
 *
 * Copyright (C) 2008 Benoit Goudreault-Emond (bgoudreaultemond@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>

#include "conf.h"

#include "imagecache.h"


typedef struct {
    gchar* filename;
    GdkPixbuf* image;
    gboolean is_double_page;
    gboolean is_manga_mode;
    double w, h;
    GThread* thread;
    gboolean marked;
} CacheEntry;

/* thread parameter */
typedef struct {
    CacheEntry* entry;
    CacheEntry* second_entry;
    gboolean is_manga_mode;
} ThreadParam;


static GArray* pagelist = NULL;
static GArchiveReader* current_reader = NULL;
static gboolean current_reader_is_reentrant;

static void
free_image(CacheEntry* entry)
{
    if(entry->thread) {
	if(debug)
	    fprintf(stderr, "Freeing thread image for %s\n", entry->filename);
	GdkPixbuf* temp = (GdkPixbuf*)g_thread_join(entry->thread);
	if(temp)
	    g_object_unref(temp);
	entry->thread = NULL;
    }
    if(entry->image) {
	if(debug)
	    fprintf(stderr, "Freeing synchronous image for %s\n", entry->filename);
	g_object_unref(entry->image);
	entry->image = NULL;
    }
}

static void
clear_pagelist(GArray* pl)
{
  int i;
  CacheEntry* entry;
  for(i = 0; i < pl->len; ++i) {
      entry = &g_array_index(pagelist, CacheEntry, i);
      free_image(entry);
      g_free(entry->filename);
  }
  g_array_free(pl, TRUE);
}

static int
sort_thunk(gconstpointer a,
	   gconstpointer b,
	   gpointer user_data)
{
    GCompareFunc cmp = (GCompareFunc)user_data;
    CacheEntry* lhs = (CacheEntry*)a;
    CacheEntry* rhs = (CacheEntry*)b;
    return (cmp)(&lhs->filename, &rhs->filename);
}

static void
sort_pagelist(GArray* pl)
{
    g_array_sort_with_data(pl, 
			   sort_thunk,
			   g_archive_reader_translate_strategy(pref.sort_order));
}

static void
add_to_pagelist(const gchar* filename, gpointer user_data)
{
    GArray* pl = (GArray*)user_data;
    gchar* duped = g_strdup(filename);
    CacheEntry entry;
    entry.filename = duped;
    entry.image = NULL;
    entry.is_double_page = FALSE;
    entry.w = entry.h = 0.0;
    entry.thread = NULL;
    entry.marked = FALSE;
  
    g_array_append_val(pl, entry);
}

/* Archive readers are not reentrant, so we  */
static GdkPixbuf*
get_page(GArchiveReader* reader,
	 const gchar* filename,
	 double* w,
	 double* h)
{
    static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
    gboolean needs_lock = !current_reader_is_reentrant;
    GdkPixbuf* retval;
    
    if(needs_lock)
	g_static_mutex_lock(&mutex);
    retval = g_archive_reader_get_page(reader, filename, w, h);
    if(needs_lock)
	g_static_mutex_unlock(&mutex);
    return retval;
}


/* NOTE: to avoid race conditions, this routine never sets image. */
static GdkPixbuf*
do_load(CacheEntry* entry,
	CacheEntry* second_entry,
	gboolean is_manga_mode,
	double* out_w,
	double* out_h)
{
    GdkPixbuf* pxm = NULL;
    GdkPixbuf* pxm1 = NULL;
    GdkPixbuf* final = NULL;
    double w, h, w1 = 0, h1 = 0;
    
    if(debug)
	fprintf(stderr, "Loading file %s\n", entry->filename);
    
    pxm = g_archive_reader_get_page(current_reader, entry->filename, &w, &h);
    if(pxm == NULL)
	return NULL;

    if(second_entry) {
	int bps, bps1;
	if(debug)
	    fprintf(stderr, "Loading file %s\n", second_entry->filename);
	pxm1 = g_archive_reader_get_page(current_reader, second_entry->filename, &w1, &h1);
	if(pxm1 == NULL) {
	    g_object_unref(pxm);
	    return NULL;
	}
	bps = gdk_pixbuf_get_bits_per_sample(pxm);
	bps1 = gdk_pixbuf_get_bits_per_sample(pxm1);
	
	final = gdk_pixbuf_new(GDK_COLORSPACE_RGB,
			       FALSE,
			       MAX(bps, bps1),
			       w + w1,
			       MAX(h, h1));
	gdk_pixbuf_fill(final, 0xffffffff);
	if(is_manga_mode) {
	    gdk_pixbuf_copy_area(pxm1,
				 0, 0, w1, h1,
				 final,
				 0, 0);
	    g_object_unref(pxm1);
	    gdk_pixbuf_copy_area(pxm,
				 0, 0, w, h,
				 final,
				 w1, 0);
	    g_object_unref(pxm);
	} else {
	    gdk_pixbuf_copy_area(pxm,
				 0, 0, w, h,
				 final,
				 0, 0);
	    g_object_unref(pxm);
	    gdk_pixbuf_copy_area(pxm1,
				 0, 0, w1, h1,
				 final,
				 w, 0);
	    g_object_unref(pxm1);
	}
    } else {
	final = pxm;
    }

    *out_w = w + w1;
    *out_h = MAX(h, h1);
    
    return final;
}

static void
collect_garbage(GArray* pl)
{
    int i;
    for(i = 0; i < pl->len; ++i) {
	CacheEntry* entry = &g_array_index(pl, CacheEntry, i);
	if(!entry->marked)
	    free_image(entry);
    }
}

static gpointer
thread_loader(gpointer data) 
{
    ThreadParam* param_ptr = (ThreadParam*)data;
    ThreadParam param = *param_ptr;
    g_free(param_ptr);
    if(debug)
	fprintf(stderr, "Loading from thread\n");
    return do_load(param.entry,
		   param.second_entry,
		   param.is_manga_mode,
		   &param.entry->w,
		   &param.entry->h);
}

/* NOTE: this routine and load_asynch are the ONLY places allowed to
 * set entry->image; this is to avoid race conditions
 */
static GdkPixbuf*
load_synch(CacheEntry* entry,
	   CacheEntry* second_entry,
	   gboolean is_manga_mode,
	   double* w,
	   double* h)
{
    GdkPixbuf* retval = NULL;
    double this_w = 0.0, this_h = 0.0;
    if(entry->thread) {
	entry->image = g_thread_join(entry->thread);
	if(debug)
	    fprintf(stderr, "Got from thread: %p\n", entry->image);
    }
    entry->thread = NULL;
    if(entry->image && 
       ((second_entry != NULL) == entry->is_double_page) &&
       entry->is_manga_mode == is_manga_mode) {
	if(debug)
	    fprintf(stderr, "Cache hit for %s\n", entry->filename);
	retval = entry->image;
    } else {
	if(debug)
	    fprintf(stderr, "Cache miss for %s\n", entry->filename);
	retval = do_load(entry, second_entry, is_manga_mode, &this_w, &this_h);
	if(retval) {
	    entry->w = this_w;
	    entry->h = this_h;
	    entry->is_double_page = second_entry ? TRUE : FALSE;
	    entry->is_manga_mode = is_manga_mode;
	    if(entry->image)
		g_object_unref(entry->image);
	    entry->image = retval;
	}
    }
    if(w) *w = entry->w;
    if(h) *h = entry->h;
    entry->marked = FALSE;
    return retval;
}

static void
load_asynch(GArray* pl,
	    gboolean is_double_page,
	    gboolean is_manga_mode)
{
    /* find all marked entries */
    int i;
    ThreadParam* param;
    for(i = 0; i < pl->len; ++i) {
	CacheEntry* entry = &g_array_index(pl, CacheEntry, i);
	if(entry->marked) {
	    entry->marked = FALSE;
	    if(entry->is_double_page == is_double_page &&
	       entry->is_manga_mode == is_manga_mode) {
		if(entry->thread)
		    entry->image = (GdkPixbuf*)g_thread_join(entry->thread);
		entry->thread = NULL;
		if(entry->image)
		    continue;
	    } else
		free_image(entry);
	    /* if we are here, we need to spawn a thread to load the image */
	    param = g_new(ThreadParam, 1);
	    param->entry = entry;
	    if(is_double_page && i + 1 < pl->len)
		param->second_entry = &g_array_index(pl, CacheEntry, i + 1);
	    else
		param->second_entry = NULL;
	    param->is_manga_mode = is_manga_mode;
	    entry->is_double_page = is_double_page;
	    entry->is_manga_mode = is_manga_mode;
	    entry->thread = g_thread_create(thread_loader,
					    param,
					    TRUE,
					    NULL);
	}
    }
}

void
image_cache_load(GArchiveReader* reader)
{
    if(current_reader)
	g_object_unref(current_reader);
    current_reader = reader;
    g_object_ref(reader);
    current_reader_is_reentrant = g_archive_reader_is_reentrant(current_reader);
    
    if(pagelist)
	clear_pagelist(pagelist);
    pagelist = g_array_new(FALSE, TRUE, sizeof(CacheEntry));
    g_archive_reader_get_file_list(reader,
				   add_to_pagelist,
				   pagelist);
    if(pagelist->len != 0)
	sort_pagelist(pagelist);
}

GdkPixbuf* 
image_cache_request_page(int page_number,
			 gboolean is_double_page,
			 gboolean is_manga_mode,
			 double* w,
			 double* h,
			 const gchar** page_title)
{
    CacheEntry* entry;
    CacheEntry* second_entry = NULL;
    int nr = is_double_page ? 2 : 1;
    int effective_next_page = 0;
    GdkPixbuf* retval;

    if(w) *w = 0;
    if(h) *h = 0;
    if(!pagelist || page_number >= pagelist->len || page_number < 0)
	return NULL;

    /* look for the entry */
    entry = &g_array_index(pagelist, CacheEntry, page_number);
    if(page_title)
	*page_title = entry->filename;
    if(is_double_page && page_number + 1 < pagelist->len) {
	second_entry = &g_array_index(pagelist, CacheEntry, page_number + 1);
    }
    /* Mark it for loading. */
    entry->marked = TRUE;
    /* if we requested caching, mark the page(s) to cache as well. */
    if(pref.cache_next) {
	int i;
	CacheEntry* asynch_entry;
	effective_next_page = page_number;
	for(i = 0; i < pref.cache_next; ++i) {
	    effective_next_page += nr;
	    if(effective_next_page >= pagelist->len)
		effective_next_page = page_number + 1;
	    if(effective_next_page >= pagelist->len)
		break;
	    asynch_entry = &g_array_index(pagelist, CacheEntry, effective_next_page);
	    /* mark for load */
	    asynch_entry->marked = TRUE;
	}
    }
    /* unload pages we won't need */
    collect_garbage(pagelist);
    /* force load of the current page */
    retval = load_synch(entry, second_entry, is_manga_mode, w, h);
    /* load others asynchronously */
    load_asynch(pagelist, is_double_page, is_manga_mode);
    return retval;
}

int 
image_cache_length(void)
{
    return pagelist ? pagelist->len : 0;
}

void
image_cache_resort(void) {
    int i;
    CacheEntry* entry;
    if(!pagelist) 
	return;
    /* free all images */
    for(i = 0; i < pagelist->len; ++i) {
	entry = &g_array_index(pagelist, CacheEntry, i);
	free_image(entry);
    }
    /* resort */
    sort_pagelist(pagelist);
    /* let the next page request fix it */
}

void
image_cache_free(void)
{
    clear_pagelist(pagelist);
}

void
image_cache_dump_data(void)
{
    int i;
    if(!debug)
	return;
    if(!pagelist) {
	fprintf(stderr, "Empty cache\n");
	return;
    }
    for(i = 0; i < pagelist->len; ++i) {
	CacheEntry* entry = &g_array_index(pagelist, CacheEntry, i);
	fprintf(stderr, 
		"%d: file %s, image %s, is double: %c, is manga: %c, w: %f, h: %f, thread: %c, marked: %c\n",
		i, entry->filename, entry->image ? "loaded" : "not loaded",
		entry->is_double_page ? 'T' : 'F',
		entry->is_manga_mode ? 'T' : 'F',
		entry->w,
		entry->h,
		entry->thread ? 'T' : 'F',
		entry->marked ? 'T' : 'F');
    }
}
