#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{OptionController}.
"""

from common.carmanconfig import CarmanConfig
from main.configurebt import ConfigureGPS
#from maps.loadtrackctrl import LoadTrackController
from main.messagedlgctrl import MessageDlgController
from maps.manageaccounts import ManageAccounts
from maps.managebuddiesctrl import ManageBuddiesController
from maps.managemapsctrl import ManageMapsController
from maps.selectmapservice import SelectMapService
#from maps.tracklistview import TrackListView
from models.gpsmodel import GPSModel
from models.infosharingmodel import InfoSharingModel
from options.globaloptview import GlobalOptionsView


class OptionController(object):
    """
    Implements features related to option button in maps module.

    @ivar _parent: L{MapsController}
    @ivar is_model: L{InfoSharingModel}
    """

    def __init__(self, parent):
        self._parent = parent
        self.is_model = InfoSharingModel()

    def show(self):
        """
        Creates and shows a menu options adding callbacks to each option.
        """
        view = GlobalOptionsView(self._parent)

        if CarmanConfig().is_service:
            view.add_item("Manage account", self.__manage_accounts_cb)
            if self.is_model.service and \
                    self.is_model.service.account_is_connected():
                view.add_item("Manage buddies", self.__manage_buddies_cb)

        view.add_item("Configure GPS", self.__configure_gps)
#        view.add_item("Load track", self.__load_track)
        view.add_item("Manage maps", self.__manage_maps)
        view.add_item("Select map service", self.__select_map_repository)
        view.add_global_items()
        view.show()

    def __select_map_repository(self, obj, item, param):
        """
        Callback called when 'Select map service' option is clicked.

        @type obj: L{GlobalOptionsView}
        @param obj: The options map screen that can be controled through the 'Select map service' screen.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        view = SelectMapService(obj)
        view.show()

    def __configure_gps(self, obj, item, param):
        """
        Callback called when 'Configure GPS' option is clicked.

        @type obj: L{GlobalOptionsView}
        @param obj: The options map screen that can be controled through the 'Configure GPS' screen.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        def item_selected(obj2, item, param):
            def reconnect():
                obj.hide(True)
                obj2.hide(True)
                model.disconnect()
                model.connect(param)

            status = model.Status()
            if status == GPSModel.FIXING or status == GPSModel.FIXED:
                msg = MessageDlgController(confirm_cb=reconnect)
                msg.show_message("GPS is already connected<br>Do you " \
                    "want disconnect?", title="CONFIGURE GPS", buttons=2)
            else:
                obj2.hide()
                obj.hide(True)
                model.connect(param)

        model = GPSModel()
        view = ConfigureGPS(item_selected)
        view.show()

    """
    def __load_track(self, obj, item, param):
        def track_selected(obj2, item, param):
            if param != current:
                loadtrack = LoadTrackController(obj2)
                loadtrack.load(param)
            else:
                obj2.hide()
                obj.hide(True)
                self._parent.unload_track_aux()

        current = self._parent.get_selected_track()
        tracklist = TrackListView(obj, True, track_selected)
        tracklist.show(current)
    """

    def __manage_maps(self, obj, item, param):
        """
        Callback called when 'Manage maps' option is clicked.

        @type obj: L{GlobalOptionsView}
        @param obj: The options map screen that can be controled through the 'Manage maps' screen.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        manage_maps = ManageMapsController(obj)
        manage_maps.show()

    def __manage_accounts_cb(self, obj, item, param):
        """
        Callback called when 'Manage accounts' option is clicked.

        @type obj: L{GlobalOptionsView}
        @param obj: The options map screen that can be controled through the 'Manage accounts' screen.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        if not self.is_model.service:
            self.is_model.instantiate_isservice()
        manage_accounts = ManageAccounts(obj)
        manage_accounts.show()

    def __manage_buddies_cb(self, obj, item, param):
        """
        Callback called when 'Manage buddies' option is clicked.

        @type obj: L{GlobalOptionsView}
        @param obj: The options map screen that can be controled through the 'Manage buddies' screen.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: object
        @param param: Not used.
        """
        manage_buddies = ManageBuddiesController(obj)
        manage_buddies.show()
