#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
"""
Implements L{Transition}.
"""

import time, ecore
from common.carlog import ERROR

class Transition(object):
    """
    Implements the view transitions.

    @type canvas: C{evas.Canvas}
    @param canvas: Evas canvas object.
    @type scr_src: C{edje.Edje}
    @param scr_src: Source screen.
    @type scr_dst: C{edje.Edje}
    @param scr_dst: Destination screen.
    @type duration: int
    @param duration: time duration of the transition.
   """
    def __init__(self, canvas, scr_src, scr_dst, duration):
        self.canvas = canvas
        self.scr_src = scr_src
        self.scr_dst = scr_dst
        self.animation = None
        self.duration = duration <= 0 and 1 or duration
    # __init__

    def __fade_cb(self):
        """
        Fade transition callback.
        """
        progress = (time.time() - self.start_time) / self.duration
        color = 255 - int(round((progress > 1 and 1 or progress) * 255))
        self.clip_src.color_set(color, color, color, color)
        if progress >= 1:
            self.scr_src.hide()
            self.scr_src.clip_unset()
            self.clip_src.delete()
            self.ret_cb()
            return False
        else:
            return True
    # __fade_cb

    def fade(self, ret_cb):
        """
        Initializes the fade transition.

        @type ret_cb: callback.
        @param ret_cb: Transition finalized callback.
        """
        if not callable(ret_cb):
            ERROR("Callback function not valid for transition")
            return

        self.clip_src = self.canvas.Rectangle(size=self.canvas.size)
        self.scr_src.clip_set(self.clip_src)
        self.clip_src.show()
        self.scr_dst.show()
        self.scr_dst.lower()

        self.ret_cb = ret_cb
        self.start_time = time.time()
        self.animation = ecore.animator_add(self.__fade_cb)
    # fade

    def __scroll_cb(self):
        """
        Scroll transition callback.
        """
        progress = (time.time() - self.start_time) / self.duration
        pos = int(round((progress > 1 and 1 or progress) * self.width))
        if self.move_to_right:
            self.scr_src.move(pos, 0)
            self.scr_dst.move(pos - self.width, 0)
        else:
            self.scr_src.move(-pos, 0)
            self.scr_dst.move(self.width - pos, 0)
        if progress >= 1:
            self.scr_src.hide()
            self.scr_src.move(0, 0)
            self.ret_cb()
            return False
        else:
            return True
    # __scroll_cb

    def scroll(self, move_to_right, ret_cb):
        """
        Initializes the scroll transition.

        @type move_to_right: boolean
        @param move_to_right: scroll direction.
        @type ret_cb: callback
        @param ret_cb: transition finalized callback.
        """
        if not callable(ret_cb):
            ERROR("Callback function not valid for transition")
            return

        self.width = self.scr_src.size_get()[0]
        if move_to_right:
            self.scr_dst.move(-self.width, 0)
        else:
            self.scr_dst.move(self.width, 0)
        self.scr_dst.show()

        self.ret_cb = ret_cb
        self.move_to_right = move_to_right
        self.start_time = time.time()
        self.animation = ecore.animator_add(self.__scroll_cb)
    # scroll

    def __fall_cb(self):
        """
        Fall transition callback
        """
        progress = (time.time() - self.start_time) / self.duration
        pos = int(round((progress > 1 and 1 or progress) * self.height))
        self.scr_dst.move(0, pos - self.height)
        if progress >= 1:
            self.scr_src.hide()
            self.ret_cb()
            return False
        else:
            return True
    # __fall_cb

    def fall(self, ret_cb):
        """
        Initializes the fall transition.

        @type ret_cb: callback
        @param ret_cb: transition finalized callback.
        """
        if not callable(ret_cb):
            ERROR("Callback function not valid for transition")
            return

        self.height = self.scr_src.size_get()[1]
        self.scr_src.lower()
        self.scr_dst.move(0, -self.height)
        self.scr_dst.show()

        self.ret_cb = ret_cb
        self.start_time = time.time()
        self.animation = ecore.animator_add(self.__fall_cb)
    # fall

    def stop(self):
        """
        Stops the transition animation.
        """
        if self.animation:
            self.animation.delete()
    # stop

# Transition
