#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import evas
import logging
import evas.decorators

from terra.core.manager import Manager
from terra.core.controller import Controller
from terra.ui.grid import CellRenderer
from terra.ui.base import EdjeWidget
from terra.ui.screen import Screen

log = logging.getLogger("canola.ui.album_grid")
mger = Manager()
KineticGridWidget = mger.get_class("Widget/KineticGrid")
OptionsControllerMixin = mger.get_class("OptionsControllerMixin")
AlbumGridOptionsModelFolder = mger.get_class("Model/Options/Folder/Media/Audio/Local/AlbumArt")


class CellRendererWidget(EdjeWidget, CellRenderer):
    def __init__(self, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, "grid_item_album", parent, theme)
        self._model = None
        self._img = self.evas.FilledImage()
        self.part_swallow("contents", self._img)
        self.signal_emit("cover,hide", "")

    @evas.decorators.del_callback
    def on_del(self):
        self._img.delete()

    def theme_changed(self):
        EdjeWidget.theme_changed(self)
        self.force_redraw()

    def force_redraw(self):
        m = self._model
        self._model = None
        self.value_set(m)

    def value_set(self, v):
        if self._model is v or v is None:
            return
        self._model = v
        self.part_text_set("album", v.name or "")
        self.part_text_set("artist", v.artist or "")
        if v.cover_small is None:
            self.signal_emit("cover,hide", "")
        else:
            if not os.path.exists(v.cover_small):
                self.signal_emit("cover,hide", "")
            else:
                try:
                    self._img.file_set(v.cover_small)
                    self._img.reload()
                    self.signal_emit("cover,show", "")
                except Exception, e:
                    log.error("could not load image %r: %s", v.cover_small, e)
                    self.signal_emit("cover,hide", "")


class AlbumGrid(Screen):
    view_name = "album_grid"

    def __init__(self, canvas, main_window, title, elements, theme=None):
        Screen.__init__(self, canvas, "album_grid", main_window, title, theme)
        self.elements = elements
        self.callback_clicked = None
        self._setup_gui_grid()

    def _setup_gui_grid(self):
        def renderer_new(canvas):
            return CellRendererWidget(self)
        self._grid = KineticGridWidget(self, renderer_new, self.elements)
        self._grid.clicked_cb_set(self._cb_clicked)
        self.part_swallow("contents", self._grid)

    def _cb_clicked(self, grid, index):
        self.callback_clicked(self, index)

    def model_updated(self):
        self._grid.model_updated()

    def loaded(self):
        if not self.elements:
            self.part_text_set("message", "No items found.")
            self.signal_emit("message,show", "")

    def delete(self):
        self._grid.delete()
        self.elements = None
        Screen.delete(self)

    def append(self, child):
        self._grid.append(child)

    def extend(self, children):
        self._grid.extend(children)

    def theme_changed(self):
        Screen.theme_changed(self)
        self._grid.theme_changed()

    def force_redraw(self):
        self._grid.force_redraw()


class AlbumGridController(Controller, OptionsControllerMixin):
    terra_type = "Controller/Folder/Media/Audio/Local/AlbumArt"
    child_screen = Manager().get_class("Controller/Folder/Media/Audio/AlbumArt")

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.animating = False
        self.model.load()
        self._setup_view()
        self.model.updated = False

        # should be after setup UI
        self.model.changed_callback_add(self._update_ui)

        self._check_model_loaded()
        OptionsControllerMixin.__init__(self)

    def do_resume(self):
        if self.model.updated:
            self.model.updated = False
            self.view.force_redraw()

    def _check_model_loaded(self):
        if self.model.is_loaded:
            self.view.loaded()
        else:
            self.model.callback_loaded = self._model_loaded

    def _model_loaded(self, model):
        self.view.loaded()
        self.model.callback_loaded = None

    def _setup_view(self):
        title = self.model.name
        self.view = AlbumGrid(self.evas, self.parent.view, title,
                              self.model.children)
        self.view.callback_clicked = self.cb_on_clicked

    def _update_ui(self, model):
        self.view.model_updated()

    def delete(self):
        # should be before deleting UI
        self.model.changed_callback_del(self._update_ui)
        self.view.delete()
        self.model.unload()

    def back(self):
        if self.animating:
            return

        def end(*ignored):
            self.animating = False

        self.animating = True
        self.parent.back(end)

    def cb_on_animation_ended(self, *ignored):
        self.animating = False

    def cb_on_clicked(self, view, index):
        if self.animating:
            return

        def end(*ignored):
            self.animating = False

        self.model.current = index
        controller = self.child_screen(self.model, self.view.evas, self.parent)
        self.animating = True
        self.parent.use(controller, end)

    def go_home(self):
        self.parent.go_home()

    def force_view_redraw(self):
        """Force renderers to be redrawn.

        This may be used after some property (title, cover, ...) changed
        and view must be updated.
        """
        self.view.force_redraw()

    def reset_model(self):
        """Call reset on model children.

        This may be used after covers are fetched from internet and
        one wants the objects to try to access the covers.
        """
        self.view.freeze()
        for c in self.model.children:
            c.reset()
        self.view.thaw()

    def options_model_get(self):
        return AlbumGridOptionsModelFolder(None, self)
