#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import logging

from terra.core.plugin_prefs import PluginPrefs
from terra.core.model import Model, ModelFolder
from terra.core.manager import Manager
from terra.utils.encoding import to_utf8

mger = Manager()
db = mger.canola_db
log = logging.getLogger("canola.fs")

def exception_logger(func):
    def wrapper(*a, **ka):
        try:
            return func(*a, **ka)
        except Exception, e:
            log.error("Exception: %s", e, exc_info=True)
    return wrapper

class DirectoryModel(ModelFolder):
    """ModelFolder containing file system entries.

    The behavior can be changed by means of directory_factory and file_factory,
    the former will be used to create directories and defaults to this own
    class so the behavior is recursive, while the later to create files must
    be specified.  The result of the factory is NOT used, the factory should
    add the items to parent (default behavior of giving L{Model} a parent).

    Signature:

        C{factory(full_path, entry_name, parent)}

    @see: L{DBFileModel}
    """
    terra_type = "Model/Folder/Directory"
    __slots__ = ("path", "directory_factory", "file_factory")

    def __init__(self, path, name, parent=None):
        self.path = path
        ModelFolder.__init__(self, name, parent)
        self.directory_factory = self.__class__

    def do_load(self):
        isdir = os.path.isdir
        pjoin = os.path.join
        base = self.path
        dfactory = self.directory_factory
        ffactory = self.file_factory
        dirs = os.listdir(base)
        dirs.sort()
        for f in dirs:
            p = pjoin(base, f)
            if isdir(p):
                dfactory(p, f, self)
            else:
                ffactory(p, f, self)


class FileModel(Model):
    __slots__ = ("path",)
    def __init__(self, path, name, parent=None):
        self.path = path
        Model.__init__(self, name, parent)


class DBFileModel(FileModel):
    """Model of file entry associated with DB.

    This provides a factory to be used with L{DirectoryModel} that will
    lookup the given path in the DB using C{stmt_get_id} SQL query and if
    one entry is found a new item is created, otherwise no instance will
    be created.
    """
    __slots__ = ("id",)

    stmt_get_id = "SELECT id FROM files WHERE path = ? AND dtime = 0"

    def __init__(self, id, path, name, parent=None):
        self.id = id
        FileModel.__init__(self, path, name, parent)

    @classmethod
    @exception_logger
    def factory(cls, path, name, parent=None):
        try:
            row = db.execute(cls.stmt_get_id, (buffer(path),))[0]
        except IndexError:
            return None

        return cls(row[0], path, name, parent)


def gen_stmt_get_id(extra_table):
    """Create stmt_get_id to join files with extra_table.

    This will create a SQL statement to get one item from database by using
    path and dtime=0 (no deleted files).
    """
    return "SELECT id FROM files, %s USING(id) WHERE path = ? AND dtime = 0" % \
           (extra_table,)


def memoizer_getter(name, getter):
    """Create a property getter function that will remember last value.

    This will create a function that uses C{name} as instance attribute to
    remember last calculated value, making following lookups faster.

    If value changes, the old will still be used until the attribute C{name}
    is deleted.

    @see: L{memoizer_setter()}
    """
    def f(self):
        try:
            return getattr(self, name)
        except AttributeError:
            v = getter(self)
            setattr(self, name, v)
            return v
    return f


def memoizer_setter(name, setter):
    """Create a property setter function that will forget about last value.

    This will set the value and remember this new value so L{memoizer_getter()}
    will use it.

    @see: L{memoizer_getter()}
    """
    def f(self, v):
        setter(self, v)
        setattr(self, name, v)
    return f


def memoizer_property(name, fget=None, fset=None):
    """Create a property that remember last value, making lookups faster.

    This kind of property is useful for expensive operations like math or
    database queries.

    @param name: attribute name to use to remember values.
    @param fget: getter function.
    @param fset: setter function.
    """
    if fget:
        fget = memoizer_getter(name, fget)
    if fset:
        fset = memoizer_setter(name, fset)
    return property(fget, fset)


def gen_prop_db_generic(attr_name, stmt_get, stmt_set=None, key_attr="id"):
    """Creates a property that lookups and possible set DB fields.

    It will use L{memoizer_property()} on C{attr_name} to speed up.

    @param attr_name: attribute name to use to remember values.
    @param stmt_get: SQL statement to use to get values.
    @param stmt_set: SQL statement to use to set values.
    """
    def getter(self):
        try:
            row = db.execute(stmt_get, (getattr(self, key_attr),))[0]
        except IndexError:
            return None

        return to_utf8(row[0])

    if stmt_get is None:
        return property(getter)

    def setter(self, value):
        db.execute(stmt_set, (value, getattr(self, key_attr)))

    return memoizer_property(attr_name, exception_logger(getter),
                             exception_logger(setter))


def gen_prop_db(attr_name, table, name, key_attr="id"):
    """Creates a property to query a value from DB.

    Generate simple queries for SELECT and UPDATE the table field.

    @param attr_name: attribute name to use to remember values.
    @param table: database table name.
    @param name: database table field name.
    """
    stmt_get = "SELECT %s FROM %s WHERE id = ?" % (name, table)
    stmt_set = "UPDATE %s SET %s = ? WHERE id = ?" % (table, name)
    return gen_prop_db_generic(attr_name, stmt_get, stmt_set, key_attr)


def gen_prop_db_join(attr_name, tables_fields, name):
    """Creates a property to query a value from DB using left join.

    Generate left join queries for SELECT the table field. UPDATE is NOT
    supported.

    @param attr_name: attribute name to use to remember values.
    @param tables_fields: pair of TABLE.FIELD to use for join.
    @param name: database table field name.
    """

    if len(tables_fields) != 2 or \
       "." not in tables_fields[0] or \
       "." not in tables_fields[1]:
        raise ValueError("tables_fields must be a pair of TABLE.FIELD")
    ta = tables_fields[0].split(".")[0]
    tb = tables_fields[1].split(".")[0]
    t = "%s, %s" % (ta, tb)
    c = "%s = %s" % (tables_fields[1], tables_fields[0])
    stmt_get = "SELECT %s FROM %s WHERE %s.id = ? AND %s" % (name, t, ta, c)
    return gen_prop_db_generic(attr_name, stmt_get)


def get_cover_dir():
    canola_prefs = PluginPrefs("settings")
    try:
        path = canola_prefs["cover_path"]
    except KeyError:
        path = canola_prefs["cover_path"] = \
            os.path.expanduser("~/.canola/covers/")
        canola_prefs.save()

    if not os.path.exists(path):
        os.makedirs(path)
    return path


def gen_prop_cover(filename):
    def getter(self):
        try:
            return os.path.join(self.cover_dir, self.artist, self.album, filename)
        except Exception:
            return None

    return property(getter)

class MediaDBFileModel(DBFileModel):
    __slots__ = ("uri",)

    def __init__(self, id, path, name, parent=None):
        DBFileModel.__init__(self, id, path, name, parent)
        self.uri = self.path


class AudioDBFileModel(MediaDBFileModel):
    terra_type = "Model/Media/Audio/FS"
    __slots__ = ("_title", "_trackno", "_rating", "_playcnt", "_album",
                 "_artist", "_genre")
    stmt_get_id = gen_stmt_get_id("audios")
    cover_dir = get_cover_dir()

    title = gen_prop_db("_title", "audios", "title")
    trackno = gen_prop_db("_trackno", "audios", "trackno")
    rating = gen_prop_db("_rating", "audios", "rating")
    playcnt = gen_prop_db("_playcnt", "audios", "playcnt")
    album_id = gen_prop_db("_album_id", "audios", "album_id")
    album = gen_prop_db("_album", "audio_albums", "name", "album_id")
    artist = gen_prop_db_generic("_artist",
                                 "SELECT audio_artists.name FROM "
                                 "audios, audio_albums, audio_artists WHERE "
                                 "audios.id = ? AND "
                                 "audio_albums.id = audios.album_id AND "
                                 "audio_artists.id = audio_albums.artist_id")
    genre_id = gen_prop_db("_genre_id", "audios", "genre_id")
    genre = gen_prop_db("_genre", "audio_genres", "name", "genre_id")
    cover = gen_prop_cover("cover.jpg")
    cover_small = gen_prop_cover("cover-small.jpg")
    start_pos = None
    last_pos = None


class AudioDBDirectoryModel(DirectoryModel):
    terra_type = "Model/Folder/Directory/Media/Audio"
    file_factory = AudioDBFileModel.factory


class VideoDBFileModel(MediaDBFileModel):
    terra_type = "Model/Media/Video/FS"
    __slots__ = ("_title", "_artist")
    stmt_get_id = gen_stmt_get_id("videos")

    title = gen_prop_db("_title", "videos", "title")
    artist = gen_prop_db("_artist", "videos", "artist")


class VideoDBDirectoryModel(DirectoryModel):
    terra_type = "Model/Folder/Directory/Media/Video"
    file_factory = VideoDBFileModel.factory


class ImageDBFileModel(DBFileModel):
    terra_type = "Model/Media/Image/FS"
    __slots__ = ("_title", "_width", "_height")
    stmt_get_id = gen_stmt_get_id("images")

    title = gen_prop_db("_title", "images", "title")
    width = gen_prop_db("_width", "images", "width")
    height = gen_prop_db("_height", "images", "height")


class ImageDBDirectoryModel(DirectoryModel):
    terra_type = "Model/Folder/Directory/Media/Image"
    file_factory = ImageDBFileModel.factory
