#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

from terra.core.terra_object import TerraObject
from terra.core.model import Model
from terra.core.manager import Manager

db = Manager().canola_db

import logging
log = logging.getLogger("canola.lazy_list")

class LazyDBList(TerraObject):
    """Basic class to create a list of children to a ModelFolder,
    creating children on demand from a database.

    You should subclass this and reimplement _create_item().

    """
    terra_type = "Utils/LazyDBList"

    def __init__(self, parent, query, *query_args):
        self.parent = parent

        if not isinstance(query, tuple):
            self.query = "SELECT * FROM %s LIMIT ? OFFSET ?" % query
            self.count_query = "SELECT COUNT(*) FROM %s" % query
        else:
            result_column = query[0]
            t_query = query[1]
            self.query = "SELECT %s FROM %s LIMIT ? " \
                         "OFFSET ?" % (result_column, t_query)
            self.count_query = "SELECT COUNT(*) FROM " \
                               "(SELECT %s FROM %s)" % (result_column, t_query)

        self.query_args = query_args
        self._count = None
        self._elements = None

    def _create_item(self, row):
        raise NotImplementedError("You must implement _create_item.")

    def _on_item_delete(self, item):
        pass

    def __setitem__(self, key, item):
        self._elements[key] = item

    def __len__(self):
        if self._count is None:
            try:
                row = db.execute(self.count_query, *self.query_args)[0]
                self._count = row[0]
            except Exception, e:
                log.error("Error loading number of elements for list: %s", e)
                self._count = 0
        return self._count

    def __getitem__(self, idx):
        if self._elements is None:
            self._elements = [None] * len(self)

        if isinstance(idx, slice):
            start = idx.start or 0
            stop = idx.stop or len(self)
            idx_slice = slice(start, stop, None)
        else:
            idx_slice = slice(idx, idx + 5, None)
        limit = idx_slice.stop - idx_slice.start

        if None in self._elements[idx_slice]:
            if self.query_args:
                args = self.query_args[0] + (limit, idx_slice.start)
            else:
                args = (limit, idx_slice.start)

            rows = db.execute(self.query, args)
            index = idx_slice.start
            for row in rows:
                if self._elements[index] is None:
                    self._elements[index] = self._create_item(row)
                index += 1

        return self._elements[idx]

    def __delitem__(self, idx):
        if self._elements is None:
            return

        if isinstance(idx, slice):
            stop = idx.stop or len(self)
            start = idx.start or 0
            size = stop - start
            for item in self._elements[idx]:
                self._on_item_delete(item)
        else:
            self._on_item_delete(self._elements[idx])
            size = 1
        del self._elements[idx]
        self._count -= size

    def reset(self):
        self._elements = None
        self._count = None


class HybridList(TerraObject):
    "Merges two lists with maybe different behaviours into one."

    terra_type = "Utils/HybridList"

    def __init__(self, first, second):
        self._first = first
        self._second = second

    def __setitem__(self, key, item):
        if key < len(self._first):
            self._first[key] = item
        else:
            self._second[key - len(self._first)] = item

    def __len__(self):
        return len(self._first) + len(self._second)

    def __normalize_slice(self, idx):
        start = idx.start or 0
        if start < 0:
            start += len(self)
        stop = idx.stop or len(self)
        if stop < 0:
            stop += len(self)
        idx = slice(start, stop)
        return idx

    def __getitem__(self, idx):
        if isinstance(idx, slice):
            return self.__get_slice(idx)
        return self.__get_one_item(idx)

    def __get_one_item(self, idx):
        if idx < 0:
            idx += len(self)
        if idx < len(self._first):
            return self._first[idx]
        return self._second[idx - len(self._first)]

    def __get_slice(self, idx):
        idx = self.__normalize_slice(idx)

        if idx.start >= idx.stop:
            return []
        elif idx.stop <= len(self._first):
            return self._first[idx]
        elif idx.start >= len(self._first):
            offset = len(self._first)
            offset_idx = slice(idx.start - offset, idx.stop - offset)
            return self._second[offset_idx]
        return self._first[idx.start:] + \
               self._second[:idx.stop - len(self._first)]

    def __delitem__(self, idx):
        if isinstance(idx, slice):
            self.__del_slice(idx)
        else:
            self.__del_one_item(idx)

    def __del_one_item(self, idx):
        if idx < 0:
            idx += len(self)
        if idx < len(self._first):
            del self._first[idx]
        else:
            del self._second[idx - len(self._first)]

    def __del_slice(self, idx):
        idx = self.__normalize_slice(idx)

        if idx.start >= idx.stop:
            pass
        elif idx.stop <= len(self._first):
            del self._first[idx]
        elif idx.start >= len(self._first):
            offset = len(self._first)
            offset_idx = slice(idx.start - offset, idx.stop - offset)
            del self._second[offset]
        else:
            del self._first[idx.start:]
            del self._second[:idx.stop - len(self._first)]

    def __str__(self):
        return str(self[:])


class AudioModel(Model):
    """Base class to guide developers to build AudioModels
    playable by the Player plugin.

    """
    terra_type = "Model/Media/Audio"

    def __init__(self, name=None, parent=None):
        if not name:
            Model.__init__(self, "", parent)
        else:
            Model.__init__(self, name, parent)
        self.id = None
        self.uri = ""
        self.genre = ""
        self.trackno = 0
        self.rating = None
        self.playcnt = 0
        self.artist = ""
        self.album = ""
        self.cover = None
        self.start_pos = None

    def __eq__(self, other):
        return isinstance(other, AudioModel) and \
               self.title == other.title and \
               self.uri == other.uri and \
               self.genre == other.genre and \
               self.trackno == other.trackno and \
               self.rating == other.rating and \
               self.artist == other.artist and \
               self.album == other.album and \
               self.cover == other.cover


class VideoModel(Model):
    """Base class to guide developers to build VideoModels
    playable by the Player plugin.

    """
    terra_type = "Model/Media/Video"

    def __init__(self, name=None, parent=None):
        if not name:
            Model.__init__(self, "", parent)
        else:
            Model.__init__(self, name, parent)
        self.title = name
        self.artist = ""
        self.uri = ""
        self.size = 0
        self.thumbnail = ""
