#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging

import evas
import edje.decorators

from terra.ui.screen import Screen
from terra.ui.throbber import ThrobberWidget

log = logging.getLogger("canola.audio.screen")

class PlayerScreen(Screen):
    view_name = "player"

    def __init__(self, evas, main_window):
        Screen.__init__(self, evas, "player_view", main_window, "Now Playing")
        self.canvas = evas
        self.video_window = None
        self.tb_disabled = False
        self.tracking_disabled = False
        self.throbber = ThrobberWidget(self, self.theme,
                                       type=ThrobberWidget.WHITE)
        self.media_area = self.part_swallow_get("player_view/media_area")
        self.media_controls = \
            self.part_swallow_get("player_view/media_controls")

        # setup initial callbacks
        self.callback_prev = self.prev
        self.callback_next = self.next
        self.callback_seek = self.seek
        self.callback_handle_play_pause = None
        self.callback_trackbar_mouse_up = None
        self.callback_trackbar_mouse_down = None
        self.callback_set_volume = self.set_volume
        self.callback_get_duration = self.get_duration
        self.callback_theme_changed = None
        self.callback_update_trackbar = self.update_trackbar


    def prev(self):
        raise NotImplementedError("You must setup a callback to prev")

    def next(self):
        raise NotImplementedError("You must setup a callback to next")

    def seek(self, value):
        raise NotImplementedError("You must setup a callback to seek")

    def play(self):
        raise NotImplementedError("You must setup a callback to play")

    def pause(self):
        raise NotImplementedError("You must setup a callback to pause")

    def set_volume(self, value):
        raise NotImplementedError("You must setup a callback to "
                                  "set the volume")

    def get_duration(self):
        raise NotImplementedError("You must setup a callback to "
                                  "get the duration")

    def update_trackbar(self):
        raise NotImplementedError("You must setup a callback to "
                                  "update the trackbar")

    def theme_changed(self):
        Screen.theme_changed(self)
        self.media_area = self.part_swallow_get("player_view/media_area")
        self.media_controls = \
            self.part_swallow_get("player_view/media_controls")
        if self.callback_theme_changed:
            self.callback_theme_changed()

    def throbber_start(self):
        self.throbber.start()
        if self.video_window:
            self.video_window.hide()

    def throbber_stop(self):
        self.throbber.stop()
        if self.video_window:
            self.video_window.show()

    def set_playing(self, playing):
        if playing:
            signal = "media,state,play"
        else:
            signal = "media,state,pause"
        self.media_controls.signal_emit(signal, "")

    def setup_volume(self, volume=0):
        self.volume = volume
        v = volume / 100.0
        vc = self.part_swallow_get("player_view/volume_control")
        vc.part_drag_value_set("knob", 0.0, 1 - v)

    def setup_view(self):
        # initial data
        tb = self.part_swallow_get("player_view/track_bar")
        tb.part_text_set("track_position", "00:00")
        tb.part_text_set("track_left", "- 00:00")

    def change_trackbar_state(self, enable=True):
        tb = self.part_swallow_get("player_view/track_bar")
        tl = tb.part_swallow_get("time_label_left")
        if enable:
            self.tb_disabled = False
            tb.signal_emit("drag,enable", "trackbar_base")
            tl.signal_emit("time_left,show", "")
        else:
            self.tb_disabled = True
            tb.signal_emit("drag,disable", "trackbar_base")
            tl.signal_emit("time_left,hide", "")

    def change_back_trackbar_state(self, enable=True):
        tb = self.part_swallow_get("player_view/track_bar")
        if enable:
            tb.signal_emit("show_back_trackbar", "")
        else:
            tb.signal_emit("hide_back_trackbar", "")

    def set_tracking_state(self, enable):
        tb = self.part_swallow_get("player_view/track_bar")
        if enable:
            tb.signal_emit("tracking,enable", "")
        else:
            tb.signal_emit("tracking,disable", "")
        self.tracking_disabled = not enable

    @edje.decorators.signal_callback("drag,set",
                                     "player_view/volume_control:knob")
    def volume_bar_cb(obj, emission, source):
        vc = obj.part_swallow_get("player_view/volume_control")
        trash, volume_factor = vc.part_drag_value_get("knob")
        obj.volume = 100 * (1 - volume_factor)
        obj.callback_set_volume(obj.volume)

    def _send_pos_duration(self, pos, duration):
        tb = self.part_swallow_get("player_view/track_bar")
        tlp = tb.part_swallow_get("time_label_pos")
        tll = tb.part_swallow_get("time_label_left")
        tlp.message_send(0, duration)
        tll.message_send(0, duration)
        tlp.message_send(0, pos)
        tll.message_send(0, pos)

    @edje.decorators.signal_callback("trackbar_base,current_pos",
                                     "player_view/track_bar:*")
    def position_cb(obj, emission, source):
        if not obj.tb_disabled:
            try:
                pos = float(source.split(":")[1].replace(",", "."))
                duration = obj.callback_get_duration()
                obj._send_pos_duration(pos, duration)
            except Exception, e:
                log.warning("Error while updating pos @ view: %s", e)

    @edje.decorators.signal_callback("trackbar_base,drag_set",
                                     "player_view/track_bar:*")
    def trackbar_cb(obj, emission, source):
        if not obj.tb_disabled:
            try:
                pos = float(source.split(":")[1].replace(",", "."))
                duration = obj.callback_get_duration()
                tb = obj.part_swallow_get("player_view/track_bar")
                tb.part_drag_value_set("knob", pos, 0.0)
                obj.callback_seek(int(pos * duration))

                # set duration and position labels
                obj._send_pos_duration(pos, duration)
            except Exception, e:
                log.warning("Error during trackbar callback: %s", e)

    @edje.decorators.signal_callback("action,clicked",
                                     "player_view/media_controls:previous")
    def prev_button_cb(obj, emission, source):
        obj.callback_prev()

    @edje.decorators.signal_callback("action,clicked",
                                     "player_view/media_controls:play_pause")
    def play_pause_button_cb(obj, emission, source):
        obj.callback_handle_play_pause()

    @edje.decorators.signal_callback("action,clicked",
                                     "player_view/media_controls:next")
    def next_button_cb(obj, emission, source):
        obj.callback_next()

    @edje.decorators.signal_callback("mouse,down,1",
                                     "player_view/track_bar:*")
    def trackbar_mouse_down(obj, emission, source):
        if obj.callback_trackbar_mouse_down:
            obj.callback_trackbar_mouse_down()

    @edje.decorators.signal_callback("mouse,up,1",
                                     "player_view/track_bar:*")
    def trackbar_mouse_up(obj, emission, source):
        if obj.callback_trackbar_mouse_up:
            obj.callback_trackbar_mouse_up()

    def raise_volume(self, value):
        volume =  self.volume + value
        vc = self.part_swallow_get("player_view/volume_control")
        if volume <= 100.0:
            self.volume = volume
            v = volume / 100.0
            vc.part_drag_value_set("knob", 0.0, 1 - v)
        else:
            self.volume = 100.0
            vc.part_drag_value_set("knob", 0.0, 0)
        return True

    def lower_volume(self, value):
        volume =  self.volume - value
        vc = self.part_swallow_get("player_view/volume_control")
        if volume >= 0.0:
            self.volume = volume
            v = volume / 100.0
            vc.part_drag_value_set("knob", 0.0, 1 - v)
        else:
            self.volume = 0.0
            vc.part_drag_value_set("knob", 0.0, 1)
        return True

    def handle_key_down(self, event):
        # Handle keys while focus is on canvas
        if event.keyname == "F7":
            self.raise_volume(5)
        elif event.keyname == "F8":
            self.lower_volume(5)
        elif event.keyname == "Left":
            self.callback_prev()
        elif event.keyname == "Right":
            self.callback_next()
        elif event.keyname == "Return":
            self.callback_handle_play_pause()
        else:
            self._parent_widget.handle_key_down(event)
        return True

    @evas.decorators.del_callback
    def delete_cb(self):
        self.throbber.delete()
        self.tb_disabled = False
        self.callback_prev = None
        self.callback_next = None
        self.callback_seek = None
        self.callback_handle_play_pause = None
        self.callback_trackbar_mouse_up = None
        self.callback_trackbar_mouse_down = None
        self.callback_set_volume = None
        self.callback_get_duration = None
        self.callback_theme_changed = None
        self.callback_update_trackbar = None
