/* 
   Copyright (C) 2011 Ove Kaaven

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

#include "lookup.h"
#include "settings.h"
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <curl/curl.h>
#include <libxml/HTMLparser.h>
#ifdef HAVE_EBOOK
#include <libebook/e-book.h>
#endif

static CURL* hcurl;
#ifdef HAVE_EBOOK
static EBook* book;
#endif

#ifndef HTML_PARSE_NOIMPLIED
#define HTML_PARSE_NOIMPLIED 0
#endif

#define parse_opts (HTML_PARSE_NOIMPLIED | HTML_PARSE_RECOVER | HTML_PARSE_NOERROR | HTML_PARSE_NOWARNING)

static size_t fetch_doc_cb(void *ptr, size_t size, size_t nmemb, void *data);

void prepare_lookup(void)
{
  curl_global_init(CURL_GLOBAL_ALL);
  hcurl = curl_easy_init();
  curl_easy_setopt(hcurl, CURLOPT_WRITEFUNCTION, fetch_doc_cb);

  /* for now, don't verify SSL certificates */
  curl_easy_setopt(hcurl, CURLOPT_SSL_VERIFYPEER, 0L);
  curl_easy_setopt(hcurl, CURLOPT_SSL_VERIFYHOST, 0L);
  /* only IPv4 supported on Maemo */
  curl_easy_setopt(hcurl, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);

#ifdef HAVE_EBOOK
  book = e_book_new_system_addressbook(NULL);
  if (book) {
    if (!e_book_open(book, TRUE, NULL)) {
      printf("Could not open system addressbook\n");
      g_object_unref(book);
      book = NULL;
    }
  }
#endif
}

void cleanup_lookup(void)
{
#ifdef HAVE_EBOOK
  if (book) {
    g_object_unref(book);
    book = NULL;
  }
#endif
  if (hcurl) {
    curl_easy_cleanup(hcurl);
    hcurl = NULL;
  }
}

static xmlNodePtr get_element(xmlNodePtr node)
{
  while (node && node->type != XML_ELEMENT_NODE) {
    node = node->next;
  }
  return node;
}

static char *get_query(xmlNodePtr conf, const char *id)
{
  xmlNodePtr cur = conf ? conf->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char *)cur->name, "query")) {
      xmlChar *q_data = xmlNodeGetContent(cur);
      char *q_full = g_strconcat((char *)q_data, id, NULL);
      xmlFree(q_data);
      return q_full;
    }
    cur = cur->next;
  }
  return NULL;
}

static xmlNodePtr get_search_tree(xmlNodePtr conf, const char *type)
{
  xmlNodePtr cur = conf ? conf->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char *)cur->name, type)) {
      return get_element(cur->children);
    }
    cur = cur->next;
  }
  return NULL;
}

static size_t fetch_doc_cb(void *ptr, size_t size, size_t nmemb, void *data)
{
  static const char *cst = ";charset=";
  htmlParserCtxtPtr *parser = data;
  int err;

  if (!*parser) {
    char *url = NULL, *ctype = NULL, *cs = NULL;
    xmlCharEncoding enc = 0;

    curl_easy_getinfo(hcurl, CURLINFO_EFFECTIVE_URL, &url);
    curl_easy_getinfo(hcurl, CURLINFO_CONTENT_TYPE, &ctype);

    /* find encoding from Content-Type header */
    cs = strstr(ctype, cst);
    if (cs) {
      enc = xmlParseCharEncoding(cs + strlen(cst));
    }

    /* create parser using known information */
    *parser = htmlCreatePushParserCtxt(NULL, NULL, NULL, 0, url, enc);
    if (!*parser) {
      fprintf(stderr, "HTML Parser could not be created\n");
      return 0;
    }
    htmlCtxtUseOptions(*parser, parse_opts);
  }

  /* parse data */
  err = htmlParseChunk(*parser, ptr, nmemb, 0);
  if (err) {
    /* this is usually errors about malformed HTML, ignore it */
    /* printf("HTML Parser Error %d\n", err); */
  }
  return nmemb;
}

/* fetch over HTTP using libcurl */
static xmlDocPtr fetch_doc(const char *url)
{
  htmlParserCtxtPtr parser = NULL;
  htmlDocPtr doc;
  CURLcode err;
  int ok;

  if (!url) return NULL;

  curl_easy_setopt(hcurl, CURLOPT_WRITEDATA, &parser);
  curl_easy_setopt(hcurl, CURLOPT_URL, url);
  err = curl_easy_perform(hcurl);
  if (err) {
    if (err != CURLE_WRITE_ERROR) {
      fprintf(stderr, "CURL Error %d for query %s\n", err, url);
    }
    if (parser) {
      htmlFreeParserCtxt(parser);
    }
    return NULL;
  }

  if (!parser) {
    long rc;
    curl_easy_getinfo(hcurl, CURLINFO_RESPONSE_CODE, &rc);
    /* It's possible that the site sent a redirect, currently we don't follow them. */
    /* On the site I tested, redirects only happen if the number was not found. */
    fprintf(stderr, "No data (code %ld) for query %s\n", rc, url);
    return NULL;
  }

  htmlParseChunk(parser, NULL, 0, 1);
  doc = parser->myDoc;
  ok = parser->wellFormed;
  htmlFreeParserCtxt(parser);

  if (!ok) {
    /* lots of bad HTML out there, ignore it */
    /* printf("Bad HTML\n"); */
  }

  return doc;
}

/* fetch from local disk (only for testing) */
static xmlDocPtr load_doc(const char *fname) __attribute__((unused));
static xmlDocPtr load_doc(const char *fname)
{
  return htmlReadFile(fname, NULL, parse_opts);
}

static int match_tag(xmlNodePtr cur, xmlNodePtr tag)
{
  xmlAttrPtr prop = tag->properties;
  while (prop) {
    if (!strcmp((char *)prop->name, "tag")) {
      xmlChar *val = xmlNodeGetContent(prop->children);
      int c = strcmp((char *)cur->name, (char *)val);
      xmlFree(val);
      if (c) break;
    } else {
      xmlChar *data = xmlGetProp(cur, prop->name);
      if (data) {
        xmlChar *val = xmlNodeGetContent(prop->children);
        int c = strcmp((char *)data, (char *)val);
        xmlFree(val);
        xmlFree(data);
        if (c) break;
      } else {
        break;
      }
    }
    prop = prop->next;
  }
  return !prop;
}

static xmlNodePtr search_for_tag(xmlNodePtr pos, xmlNodePtr tag)
{
  xmlNodePtr cur = get_element(pos->children);
  xmlNodePtr ret = NULL;
  while (cur) {
    xmlNodePtr next = get_element(cur->next);
    if (match_tag(cur, tag)) {
      ret = cur;
      break;
    } else {
      ret = search_for_tag(cur, tag);
      if (ret) {
        break;
      }
    }
    cur = next;
  }
  return ret;
}

static xmlNodePtr delete_tag(xmlNodePtr pos, xmlNodePtr tag)
{
  xmlNodePtr cur = get_element(pos->children);
  while (cur) {
    xmlNodePtr next = get_element(cur->next);
    if (match_tag(cur, tag)) {
      xmlUnlinkNode(cur);
      xmlFreeNode(cur);
    } else {
      delete_tag(cur, tag);
    }
    cur = next;
  }
  return pos;
}

static xmlNodePtr search_in_doc(xmlNodePtr pos, xmlNodePtr tag)
{
  if (!strcmp((char *)tag->name, "find")) {
    return search_for_tag(pos, tag);
  }
  if (!strcmp((char *)tag->name, "delete")) {
    return delete_tag(pos, tag);
  }
  return pos;
}

static char *search_doc(xmlNodePtr conf, xmlDocPtr doc, const char *type)
{
  xmlNodePtr pos = doc ? xmlDocGetRootElement(doc) : NULL;
  xmlNodePtr cur = get_search_tree(conf, type);
  xmlChar *data = NULL;
  char **toks, **stok, **dtok, *res;

  while (pos && cur) {
    pos = search_in_doc(pos, cur);
    cur = get_element(cur->next);
  }
  if (!pos) {
    return NULL;
  }
  data = xmlNodeGetContent(pos);
  if (!data) return NULL;

  /* in order to compress whitespace, split into tokens */
  toks = g_strsplit_set((char *)data, " \r\n\t", 0);
  xmlFree(data);

  /* remove empty tokens */
  stok = toks;
  dtok = toks;
  while (*stok) {
    if (**stok) {
      *dtok++ = *stok++;
    } else {
      g_free(*stok);
      stok++;
    }
  }
  *dtok = NULL;

  /* recreate string */
  res = g_strjoinv(" ", toks);
  g_strfreev(toks);

  return res;
}

static char *search_locdb(xmlNodePtr region, const char *number)
{
  FILE *f = get_region_text(region, number);
  char *ret = NULL;
  int ch = '\n';

  if (!f) return NULL;

  while (!feof(f)) {
    /* match prefix */
    const char *nn = number;
    int diff = 0;
    while (!diff) {
      ch = getc(f);
      if (ch == EOF || ch == '\n' || ch == '\t') break;
      diff = ch - *nn;
      if (*nn) nn++;
    }
    if (diff) {
      /* no match, skip line */
      while (ch != EOF && ch != '\n') {
        ch = getc(f);
      }
    }
    else {
      if (ch != '\t') break;
      /* match found */
      GString *str = g_string_new(NULL);
      ch = getc(f);
      while (ch != EOF && ch != '\n') {
        g_string_append_c(str, ch);
        ch = getc(f);
      }
      ret = g_string_free(str, FALSE);
      break;
    }
  }
  fclose(f);
  return ret;
}

static EContact *lookup_book(const char *number)
{
  GList *contacts = NULL;
  EBookQuery *query = e_book_query_vcard_field_test("TEL", E_BOOK_QUERY_IS, number);
  gboolean ok = e_book_get_contacts(book, query, &contacts, NULL);
  e_book_query_unref(query);
  if (contacts) {
    /* return first matching contact and free the rest */
    GList *first = g_list_first(contacts);
    EContact *contact = first->data;
    contacts = g_list_delete_link(contacts, first);
    g_list_foreach(contacts, (GFunc)g_object_unref, NULL);
    g_list_free(contacts);
    return contact;
  }
  return NULL;
}

char *lookup(const char *number, const char *mcc)
{
  GString *ret = g_string_new(NULL);
  xmlNodePtr region = get_region(number, mcc);
  const char *domnumber = skip_prefix(region, number);
  EContact *contact;

  /* try to find contact in local address book */
  contact = lookup_book(number);
  if (!contact && domnumber != number)
    contact = lookup_book(domnumber);

  if (contact) {
    /* If the caller is in the addressbook, we don't need to show
     * basic contact details (the call dialog already shows the name,
     * and the user probably does not need help remembering the address),
     * however the user may want to remember notes. */
    char *name = e_contact_get(contact, E_CONTACT_FULL_NAME);
    char *note = e_contact_get(contact, E_CONTACT_NOTE);
    if (name) {
      /* Will add the name to the return string for console logging purposes,
       * but the GUI notification will suppress this line since it starts
       * with "in addressbook". */
      g_string_append(ret, "in addressbook: ");
      g_string_append(ret, name);
    }
    if (note && *note) {
      if (ret->len) g_string_append_c(ret, '\n');
      g_string_append(ret, "Note: ");
      g_string_append(ret, note);
    }
    g_free(note);
    g_free(name);
    g_object_unref(contact);
  } else {
    char *name = NULL;
    char *address = NULL;
    /* find contact in online directory service */
    xmlNodePtr dir = get_region_directory(region, domnumber);
    const char *locnumber = skip_prefix(dir, domnumber);
    char *q = get_query(dir, locnumber);
    xmlDocPtr doc = fetch_doc(q);
    /* xmlDocPtr doc = load_doc("test.html"); */
    g_free(q);

    name = search_doc(dir, doc, "name");
    address = search_doc(dir, doc, "address");
    xmlFreeDoc(doc);

    if (name) {
      g_string_append(ret, "Name: ");
      g_string_append(ret, name);
      g_free(name);
    }
    if (address) {
      if (ret->len) g_string_append_c(ret, '\n');
      g_string_append(ret, "Address: ");
      g_string_append(ret, address);
      g_free(address);
    }
    else {
      /* If a directory lookup did not find the address, see if
       * we can at least find an approximate location. */
      xmlChar *rname = get_region_name(region);
      char *loc = search_locdb(region, domnumber);
      if (loc || rname) {
        if (ret->len) g_string_append_c(ret, '\n');
        g_string_append(ret, "Location: ");
        if (loc) {
          g_string_append(ret, loc);
          /* Probably not necessary to add country if the location was found.
           * Besides, the region name cannot currently differentiate countries
           * within the same numbering plan, such as USA and Canada. */
  #if 0
          if (rname) {
            g_string_append(ret, ", ");
            g_string_append(ret, (char *)rname);
          }
  #endif
        }
        else {
          /* Location was not found, so do show region name. */
          g_string_append(ret, (char *)rname);
        }
      }
      g_free(loc);
      xmlFree(rname);
    }
  }

  return g_string_free(ret, !ret->len);
}
