/*
 *  calendar home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <libintl.h>
#include <locale.h>
#include <ctime>
#include <clockd/libtime.h>
#include <hildon/hildon.h>
#include <CMulticalendar.h>
#include "cal-home-calendar-util.h"



static time_t 
cal_home_advance_date_by_time_period(struct tm* start,
				     const CalTimePeriod& ctp)
{
  switch(ctp)
  {
  case CAL_TIME_PERIOD_WEEK:
    start->tm_mday+=7;
    break;
  case CAL_TIME_PERIOD_2WEEK:
    start->tm_mday+=2*7;
    break;
  case CAL_TIME_PERIOD_3WEEK:
    start->tm_mday+=3*7;
    break;
  case CAL_TIME_PERIOD_MONTH:
    start->tm_mon+=1;
    break;
  case CAL_TIME_PERIOD_2MONTH:
    start->tm_mon+=2;
    break;
  case CAL_TIME_PERIOD_3MONTH:
    start->tm_mon+=3;
    break;
  case CAL_TIME_PERIOD_6MONTH:
    start->tm_mon+=6;
    break;
  case CAL_TIME_PERIOD_12MONTH:
    start->tm_year+=1;
    break;
  default:
   start->tm_year+=1; 
  }
  return mktime(start);
}

void
cal_home_time_today_midnight(struct tm* today)
{
  time_get_local(today);
  today->tm_hour = today->tm_min = today->tm_sec = 0;
}

static void
cal_home_time_today_now(struct tm* today)
{
  time_get_local(today);
  today->tm_sec = 0;
}

static int
cal_row_count_for_cal_size(const gint& cal_size)
{
  if(cal_size == -1)
    return 9;
  else
    return cal_size;
}

static GdkPixbuf* 
get_calendar_pixbuf(CComponent* component)
{
  int error = 0;
  //(*event_iter)->getCalendarId(), (*event_iter)->getType());  
  int eventType = component->getType();
  CCalendar* cal = CMulticalendar::MCInstance()->getCalendarById(component->getCalendarId(), error);

  GdkPixbuf* pixbuf = NULL;
  switch(cal->getCalendarColor())
  {
  case COLOUR_DARKBLUE:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_darkblue", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_DARKGREEN:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_darkgreen", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_DARKRED:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_darkred", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_ORANGE:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_orange", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_VIOLET:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_violet", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_YELLOW:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_yellow", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_BLUE:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_blue", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_RED:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_red", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  case COLOUR_GREEN:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_green", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
    break;
  default:
    pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
				      "calendar_colors_white", 26,GTK_ICON_LOOKUP_NO_SVG, NULL);
  }
  delete cal;
  if(eventType == 4 || component->getAlarm())
    {
      cairo_t* cr;
      cairo_surface_t* surface = cairo_image_surface_create_for_data(gdk_pixbuf_get_pixels(pixbuf),
								     CAIRO_FORMAT_RGB24,
								     26, 26,
								     gdk_pixbuf_get_rowstride(pixbuf));
      GdkPixbuf* birthday;
      if(eventType==4)
	birthday = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					   "calendar_birthday", 16, GTK_ICON_LOOKUP_NO_SVG, NULL);
      else
	birthday = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					   "calendar_alarm", 16, GTK_ICON_LOOKUP_NO_SVG, NULL);
      cr = cairo_create(surface);
      gdk_cairo_set_source_pixbuf(cr, birthday, 5, 5);
      cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
      cairo_paint(cr);
      cairo_surface_destroy(surface);
      cairo_destroy(cr);
      g_object_unref(birthday);
    }
    
  return pixbuf;
}

static void
cal_fill_event_list_ui(CalHomePlugin* desktop_plugin)
{
  GtkTreeIter iter;
  gtk_container_foreach (GTK_CONTAINER (desktop_plugin->event_list_container), (GtkCallback) gtk_widget_destroy, NULL);
  if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(desktop_plugin->event_list),
				   &iter))
  {
    do
    {
      gchar* time;
      gchar* summary;
      GdkPixbuf *pixbuf;
      gtk_tree_model_get(GTK_TREE_MODEL(desktop_plugin->event_list),
			 &iter,
			 0, &time,
			 1, &summary,
			 2, &pixbuf,
			 -1);
      GtkWidget* line = gtk_hbox_new(FALSE, 0);

      GtkWidget* time_label = gtk_label_new(time);
      GtkWidget* cal_image = gtk_image_new_from_pixbuf(pixbuf);

      hildon_helper_set_logical_font(time_label,
				     "HomeSystemFont");
      hildon_helper_set_logical_color(time_label,
				      GTK_RC_FG,
				      GTK_STATE_NORMAL,
				      "ActiveTextColor");
      GtkWidget* summary_label = gtk_label_new(summary);
      hildon_helper_set_logical_font(summary_label,
				     "HomeSystemFont");
      hildon_helper_set_logical_color(summary_label,
				      GTK_RC_FG,
				      GTK_STATE_NORMAL,
				      "DefaultTextColor");

      gtk_misc_set_alignment (GTK_MISC (time_label), 0, 0); 
      gtk_misc_set_alignment (GTK_MISC (summary_label), 0, 0);
      gtk_misc_set_alignment (GTK_MISC (cal_image), 1, 0);
      gtk_label_set_ellipsize(GTK_LABEL(summary_label), PANGO_ELLIPSIZE_END);
      gtk_box_pack_start(GTK_BOX(line), time_label, FALSE, FALSE, 8);
      gtk_box_pack_start(GTK_BOX(line), summary_label, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(line), cal_image, FALSE, FALSE, 8);
      gtk_box_pack_start(GTK_BOX(desktop_plugin->event_list_container), line, TRUE, FALSE, 4);
      
      g_free(time);
      g_free(summary);
    }while(gtk_tree_model_iter_next(GTK_TREE_MODEL(desktop_plugin->event_list),
				    &iter));
  }
}

static void
cal_delete_ccomponentset_items(SortedCComponentSet& ccomponent_set)
{
  
  for(SortedCComponentSet::const_iterator iter = ccomponent_set.begin();
      iter!=ccomponent_set.end();
      ++iter)
  {
    delete (*iter);
  }
}

static gboolean 
cal_date_on_today(const time_t& t,
		  CalHomePlugin* desktop_plugin)
{
  struct tm* time = localtime(&t);

  return time->tm_year == desktop_plugin->date_today.tm_year &&
    time->tm_mon == desktop_plugin->date_today.tm_mon &&
    time->tm_mday== desktop_plugin->date_today.tm_mday;
}

static void
cal_condense_ccomponents_set_to_list_store(const SortedCComponentSet& ccomponent_set,
					   GtkListStore* list_store,
					   CalHomePlugin* desktop_plugin,
					   gsize up_to)
{
  struct tm end_date;
  struct tm end_date_x;
  cal_home_time_today_now(&end_date);
  cal_home_time_today_midnight(&end_date_x);
  end_date_x.tm_mday++;
  time_t end_time_x = mktime(&end_date_x);
  time_t end_time = cal_home_advance_date_by_time_period(&end_date,
							 desktop_plugin->cal_time_period);
  time_t date_today_time = mktime(&desktop_plugin->date_today);
  GtkTreeIter iter;
  int earliest_update_end_time = end_time_x;
  desktop_plugin->today_events = 0;
  for(SortedCComponentSet::const_iterator event_iter = ccomponent_set.begin();
      event_iter != ccomponent_set.end() && up_to>0;
      ++event_iter)
  {
    gtk_list_store_append(list_store,
			  &iter);
    string summary = (*event_iter)->getSummary();
    string component = (*event_iter)->toString();
    time_t start = (*event_iter)->getDateStart();

    struct tm* time = localtime(&start);
    char buf[255];
    if(cal_date_on_today((*event_iter)->getDateStart(), desktop_plugin) || 
       start < date_today_time)
    {
      ++desktop_plugin->today_events;
      
      if(earliest_update_end_time>(*event_iter)->getDateEnd())
	earliest_update_end_time = (*event_iter)->getDateEnd()+60;
      if(date_today_time < (*event_iter)->getDateStart() &&
	 earliest_update_end_time > (*event_iter)->getDateStart())
	earliest_update_end_time = (*event_iter)->getDateStart()+60;

      if((*event_iter)->getAllDay() || start < date_today_time)
      {      
	gtk_list_store_set(list_store, &iter,
			   0, dgettext("calendar", "cal_va_allday_home"),
			   -1);
      }
      else
      {
	
	if(desktop_plugin->has_24_time_format)
	  strftime(buf, 255,dgettext("hildon-libs","wdgt_va_24h_time"), time);
	else
	{
	  if(time->tm_hour>11)
	  {
	    strftime(buf, 255,dgettext("hildon-libs","wdgt_va_12h_time_pm"), time);
	  }
	  else
	  {
	    strftime(buf, 255,dgettext("hildon-libs","wdgt_va_12h_time_am"), time);
	  }
	}
	gtk_list_store_set(list_store, &iter,
			   0, buf,
			   -1);
      }
    }
    else
    {
      strftime(buf, 255,dgettext("hildon-libs","wdgt_va_date_short"), time);
      gtk_list_store_set(list_store, &iter,
			 0, buf,
			 -1);
    }
    GdkPixbuf *pixbuf = get_calendar_pixbuf(*event_iter);

    gtk_list_store_set(list_store, &iter,
		       1, summary.c_str(), 
		       2, pixbuf,
		       -1);
    
    g_object_unref(pixbuf);
    --up_to;
  }
  struct tm now;
  cal_home_time_today_now(&now);
  
  if(desktop_plugin->time_out_handler)
  {
    g_source_remove(desktop_plugin->time_out_handler);
  }
  desktop_plugin->time_to_update = earliest_update_end_time;
}

static void
cal_read_ccompnents_for_calendar(CCalendar* calendar,
				 time_t start_time,
				 time_t end_time,
				 SortedCComponentSet& all_components)
{
  int error = 0;
  int offset = 0;
  int size = 0;
  int limit = 100;
  do
  {
    vector<CComponent*> calendars_components = calendar->getAllEventsTodos(start_time,
									   end_time,
									   limit,
									   offset,
									   error);
  
    size = calendars_components.size();
    offset+=limit;
    time_t last_start_date = start_time;
    for(int k = 0;k<calendars_components.size();++k)
    {
      time_t x = calendars_components[k]->getDateStart();
      CComponent* component = calendars_components[k];
      vector<time_t> instance_times =
	component->getInstanceTimes(last_start_date, 
				    end_time);
    
      // Remove Task Components now
      // others are cleaned up later
      if(calendars_components[k]->getType()==2)
      {
	delete calendars_components[k];
      }
      else
	if(instance_times.empty())
	  all_components.insert(component);
	else
	{
	  for(vector<time_t>::const_iterator iter = instance_times.begin();
	      iter!=instance_times.end();
	      ++iter)
	  {
	    // clone entry with instance time as start time
	    CComponent* c = new CComponent(*component);
	    c->setDateStart(*iter);
	    all_components.insert(c);
	  }
	}
    }
  }while(size !=0 );
}

static void
cal_read_ccomponents_all_visible(SortedCComponentSet& all_components,
				 CalHomePlugin* desktop_plugin)
{
  time_t start_time = mktime(&desktop_plugin->date_today);
  struct tm end_date = desktop_plugin->date_today;
  time_t end_time = cal_home_advance_date_by_time_period(&end_date,
							 desktop_plugin->cal_time_period);

  for(int i=0;i<desktop_plugin->number_visible_cals;++i)
  {
    int error = 0;
    CCalendar* cal = CMulticalendar::MCInstance()->getCalendarById(desktop_plugin->visible_cals[i], error);
    if(error == 500)
    {
      cal_read_ccompnents_for_calendar(cal,
				       start_time,
				       end_time,
				       all_components);
    }
    delete cal;
  }
}

void 
cal_home_load_db_events(CalHomePlugin* desktop_plugin)
{
  cal_home_time_today_now(&desktop_plugin->date_today);
  gtk_list_store_clear(desktop_plugin->event_list);
  SortedCComponentSet all_components;
  cal_read_ccomponents_all_visible(all_components,
				   desktop_plugin);
  cal_condense_ccomponents_set_to_list_store(all_components,
					     desktop_plugin->event_list,
					     desktop_plugin,
					     cal_row_count_for_cal_size(desktop_plugin->cal_row_count));
  cal_delete_ccomponentset_items(all_components);
  cal_fill_event_list_ui(desktop_plugin);
  desktop_plugin->update_view = FALSE;
  gtk_widget_show_all(desktop_plugin->event_list_container);
}
