/*
 *  calendar home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <libintl.h>
#include <locale.h>
#include <string.h>
#include <ctime>
#include <clockd/libtime.h>
#include <dbus/dbus-glib-lowlevel.h>
extern "C"{

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
}
#include <CMulticalendar.h>
#include <CalInterface.h>
#include <CComponent.h>
#include <gconf/gconf-client.h>	

#include "cal-home-widget.h"	
#include "cal-home-load-and-store.h"	
#include "cal-home-calendar-util.h"	
#include "cal-home-settings-dlg.h"	

extern "C" {
HD_DEFINE_PLUGIN_MODULE(CalHomePlugin, cal_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)
}


#define CAL_BG_IMAGE_PATH "/usr/share/icons/hicolor/64x64/hildon/calendar_applet.png"
#define GCONF_TIME_FORMAT_PATH "/apps/clock/time-format"

static const char * 	CALENDAR_DBUS_LISTENER_SIGNAL = "dbChange";

static const int CAL_WIDGET_ENTRY_ROW_HEIGHT = 36;
static const int CAL_WIDGET_TITLE_HEIGHT = 64;
static const int CAL_WIDGET_BOTTON_HEIGHT = 8;
static const int CAL_WIDGET_WIDTH = 352;

void
cal_widget_resize_for_cal_size(CalHomePlugin* desktop_plugin)
{
  CalSize count = (CalSize)gtk_tree_model_iter_n_children(GTK_TREE_MODEL(desktop_plugin->event_list), NULL); 
  gtk_widget_set_size_request(desktop_plugin->event_list_container, 
			      CAL_WIDGET_WIDTH, 
			      CAL_WIDGET_ENTRY_ROW_HEIGHT*count);
  switch(desktop_plugin->cal_size)
  {
  case CAL_SIZE_AUTO:
    gtk_window_resize(GTK_WINDOW (desktop_plugin), 
		      CAL_WIDGET_WIDTH, 
		      (CAL_WIDGET_ENTRY_ROW_HEIGHT*count)+CAL_WIDGET_TITLE_HEIGHT+CAL_WIDGET_BOTTON_HEIGHT);
    break;
  case CAL_SIZE_LARGE:
    gtk_window_resize(GTK_WINDOW (desktop_plugin), 
		      CAL_WIDGET_WIDTH, 
		      (CAL_WIDGET_ENTRY_ROW_HEIGHT*LARGE_ROW_COUNT)+CAL_WIDGET_TITLE_HEIGHT+CAL_WIDGET_BOTTON_HEIGHT);
    break;
  case CAL_SIZE_MEDIUM:
    gtk_window_resize(GTK_WINDOW (desktop_plugin), 
		      CAL_WIDGET_WIDTH, 
		      (CAL_WIDGET_ENTRY_ROW_HEIGHT*MEDIUM_ROW_COUNT)+CAL_WIDGET_TITLE_HEIGHT+CAL_WIDGET_BOTTON_HEIGHT);
    break;
  case CAL_SIZE_SMALL:
    gtk_window_resize(GTK_WINDOW(desktop_plugin), 
		      CAL_WIDGET_WIDTH,
		      (CAL_WIDGET_ENTRY_ROW_HEIGHT*SMALL_ROW_COUNT)+CAL_WIDGET_TITLE_HEIGHT+CAL_WIDGET_BOTTON_HEIGHT);
    break;
  }
}

static gboolean
cal_24_time_format()
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return TRUE;
  
  gboolean is24Format = TRUE;
  is24Format = gconf_client_get_bool(client, GCONF_TIME_FORMAT_PATH, NULL);
  g_object_unref(client);
  return is24Format;
}

static void
render_day_on_title(CalHomePlugin* desktop_plugin)
{
  cairo_t* cr;
  GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file(CAL_BG_IMAGE_PATH, NULL);

  if(!pixbuf)
  {
    return;
  }

  int img_width = gdk_pixbuf_get_width(pixbuf);
  int img_height = gdk_pixbuf_get_height(pixbuf);
  
  cairo_surface_t *surface
    = cairo_image_surface_create_for_data(gdk_pixbuf_get_pixels(pixbuf),
                                          CAIRO_FORMAT_RGB24,
                                          img_width,
                                          img_height,
                                          gdk_pixbuf_get_rowstride(pixbuf));

  cr = cairo_create(surface);

  PangoLayout* layout;
  PangoFontDescription* desc;
  
  layout = pango_cairo_create_layout(cr);

  struct tm time;
  int  ret = time_get_local(&time);
  char buf[255];

  strftime(buf, 255,"%A", &time);
  gtk_label_set_text(GTK_LABEL(desktop_plugin->day_label), buf);

  strftime(buf, 255,"%B", &time);
  gtk_label_set_text(GTK_LABEL(desktop_plugin->month_label), buf);

  sprintf(buf,"%d", time.tm_mday);
  pango_layout_set_text(layout, buf, -1);
  desc = pango_font_description_from_string("Sans Bold 19");
  pango_layout_set_font_description(layout ,desc);
  pango_font_description_free(desc);
  int width = 0;
  int height = 0;
  pango_layout_get_pixel_size(layout, &width, &height);

  cairo_move_to(cr, 32-(width/2.0),33-(height/2.0));
  cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 1.0); 
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  pango_cairo_show_layout(cr, layout);
  gtk_image_set_from_pixbuf(GTK_IMAGE(desktop_plugin->day_image),
 			    pixbuf); 
  g_object_unref(pixbuf); 
  g_object_unref(layout); 
  cairo_surface_destroy(surface);
  cairo_destroy(cr);
}

static gboolean
cal_update_time(gpointer data);

static void
register_time_out_handler(CalHomePlugin* desktop_plugin)
{
  struct tm now;
  time_get_local(&now);
  desktop_plugin->time_out_handler = g_timeout_add_seconds((23-now.tm_hour)*3600 - 50 * now.tm_min - now.tm_sec, 
							   cal_update_time, 
							   (gpointer) desktop_plugin);
}

static void
cal_update_if_visible_and_date_changed(CalHomePlugin* desktop_plugin);

static gboolean
cal_update_time(gpointer data)
{
  cal_update_if_visible_and_date_changed((CalHomePlugin*)data);
  register_time_out_handler((CalHomePlugin*)data);
  return FALSE;
}

static void 
handle_time_format_changed(GConfClient* client,
			   guint cnxn_id,
			   GConfEntry* entry,
			   gpointer user_data)
{
  CalHomePlugin* desktop_plugin = CAL_HOME_PLUGIN(user_data);
  desktop_plugin->has_24_time_format = cal_24_time_format();
  desktop_plugin->update_view = TRUE;
}

static void
cal_update_if_visible_and_date_changed(CalHomePlugin* desktop_plugin)
{
  struct tm now;
  time_get_local(&now);
  gboolean time_differ = 
    (now.tm_mday != desktop_plugin->date_today.tm_mday ||
     now.tm_mon != desktop_plugin->date_today.tm_mon ||
     now.tm_year != desktop_plugin->date_today.tm_year);

  if(!time_differ)
  {
    return;
  }
  gboolean on;
  g_object_get(GTK_WIDGET(desktop_plugin), "is-on-current-desktop", &on, NULL);

  if(on)
  {
    render_day_on_title(desktop_plugin);
    cal_home_load_db_events(desktop_plugin);
    //    gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));    
  }
  else
  {
    desktop_plugin->update_view = TRUE;
  }
}

static void
handle_time_changed(DBusGProxy *object, gint time, CalHomePlugin* desktop_plugin)
{
  cal_update_if_visible_and_date_changed(desktop_plugin);
}

static DBusHandlerResult
handle_db_change(DBusConnection *bus, 
		 DBusMessage *msg, 
		 gpointer data)
{
  if(dbus_message_is_signal(msg,CALENDAR_SERVICE, CALENDAR_DBUS_LISTENER_SIGNAL))
  {  
    CAL_HOME_PLUGIN(data)->update_view = TRUE;
  }
  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void
cal_on_current_desktop(GtkWidget* widget,
		       gpointer data)
{
  CalHomePlugin* desktop_plugin = CAL_HOME_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
  {
    if(!desktop_plugin->time_out_handler)
    {
      register_time_out_handler(desktop_plugin);
    }
    if(desktop_plugin->update_view)
    {
      render_day_on_title(desktop_plugin);
      cal_home_load_db_events(desktop_plugin);
    }
  }
  else if(desktop_plugin->time_out_handler)
  {
    g_source_remove(desktop_plugin->time_out_handler);
    desktop_plugin->time_out_handler = 0;
  }
}

static void
cal_home_show_calendar(DBusGProxy* proxy,
		       int type)
{
  
  time_t tt = time(NULL); 
  int msgID = tt; 
  const char* text = "DUMMY";  
  if(proxy)
  {
    dbus_g_proxy_call_no_reply(proxy, 
			       CALENDAR_LAUNCH_METHOD,
			       G_TYPE_UINT, type, 
			       G_TYPE_INT, msgID, 
			       G_TYPE_STRING, text,
			       DBUS_TYPE_INVALID); 
  }  
}

static gboolean
cal_home_button_release(GtkWidget* widget, GdkEventButton *event, CalHomePlugin* desktop_plugin)
{
  cal_home_show_calendar(desktop_plugin->dbus_calendar_proxy,
			 desktop_plugin->launch_view);
  return TRUE;
}

static gboolean
cal_home_button_release_show_agenda(GtkWidget* widget, GdkEventButton *event, CalHomePlugin* desktop_plugin)
{
  cal_home_show_calendar(desktop_plugin->dbus_calendar_proxy,
			 3);
  return TRUE;
}

static gboolean
cal_home_button_release_show_week_view(GtkWidget* widget, GdkEventButton *event, CalHomePlugin* desktop_plugin)
{
  cal_home_show_calendar(desktop_plugin->dbus_calendar_proxy,
			 2);
  return TRUE;
}

static gboolean
cal_home_button_release_show_month_view(GtkWidget* widget, GdkEventButton *event, CalHomePlugin* desktop_plugin)
{
  cal_home_show_calendar(desktop_plugin->dbus_calendar_proxy,
			 1);
  return TRUE;
}

static GtkWidget*
build_ui(CalHomePlugin *desktop_plugin)
{
  GtkWidget *contents = gtk_event_box_new();
  GtkWidget* box  = gtk_vbox_new(FALSE, 0);

  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  desktop_plugin->event_list_container  = gtk_vbox_new(TRUE, 0);
  GtkWidget* title = gtk_hbox_new(FALSE, 0);
  
  desktop_plugin->day_image = gtk_image_new_from_icon_name("calendar_applet", 
							   HILDON_ICON_SIZE_THUMB); 

  desktop_plugin->day_label = gtk_label_new("");
  desktop_plugin->month_label = gtk_label_new("");

  hildon_helper_set_logical_font(desktop_plugin->day_label,
				 "SystemFont");
  hildon_helper_set_logical_color(desktop_plugin->day_label,
				  GTK_RC_FG,
				  GTK_STATE_NORMAL,
				  "DefaultTextColor");

  gtk_label_set_ellipsize(GTK_LABEL(desktop_plugin->month_label), PANGO_ELLIPSIZE_END);
  hildon_helper_set_logical_color (desktop_plugin->month_label, GTK_RC_FG, GTK_STATE_NORMAL, "ActiveTextColor");
  hildon_helper_set_logical_font(desktop_plugin->month_label,
				 "SmallSystemFont");
  
  GtkWidget* day_box = gtk_event_box_new();
  GtkWidget* week_day_box = gtk_event_box_new();
  GtkWidget* month_box = gtk_event_box_new();

  gtk_event_box_set_visible_window(GTK_EVENT_BOX(day_box), FALSE);
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(week_day_box), FALSE);
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(month_box), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(day_box), 0);
  gtk_container_set_border_width(GTK_CONTAINER(week_day_box), 0);
  gtk_container_set_border_width(GTK_CONTAINER(month_box), 0);

  gtk_container_add(GTK_CONTAINER(day_box), desktop_plugin->day_image);
  gtk_container_add(GTK_CONTAINER(week_day_box), desktop_plugin->day_label);
  gtk_container_add(GTK_CONTAINER(month_box), desktop_plugin->month_label);

  gtk_misc_set_alignment (GTK_MISC (desktop_plugin->day_label), 0.5, 0.8);
  gtk_misc_set_alignment (GTK_MISC (desktop_plugin->month_label), 0.8, 1.0);
  
  gtk_box_pack_start(GTK_BOX(title), day_box, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(title), week_day_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(title), month_box, TRUE, TRUE, 3);

  g_signal_connect(GTK_CONTAINER(day_box), 
		   "button-release-event", 
		   G_CALLBACK(cal_home_button_release_show_agenda), 
		   desktop_plugin);

  g_signal_connect(GTK_CONTAINER(week_day_box), 
		   "button-release-event", 
		   G_CALLBACK(cal_home_button_release_show_week_view), 
		   desktop_plugin);

  g_signal_connect(GTK_CONTAINER(month_box), "button-release-event", 
		   G_CALLBACK(cal_home_button_release_show_month_view), 
		   desktop_plugin);

  gtk_box_pack_start(GTK_BOX(box), title, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box), desktop_plugin->event_list_container, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(contents), box);
  gtk_widget_show_all(contents);
  return GTK_WIDGET(contents);
}

static void
cal_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  CalHomePlugin* desktop_plugin = CAL_HOME_PLUGIN(widget);
  desktop_plugin->iD = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
  cal_home_read_settings(desktop_plugin);
  render_day_on_title(desktop_plugin);
  cal_home_load_db_events(desktop_plugin);
  cal_widget_resize_for_cal_size(desktop_plugin);
  GTK_WIDGET_CLASS(cal_home_plugin_parent_class)->realize(widget);

}

static void
cal_render_background_small(cairo_t* cr, CalHomePlugin* desktop_plugin)
{
  cairo_translate(cr, 0, 64); 
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -172); 
  cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, 5*36); 
  cairo_fill(cr); 
  cairo_translate(cr, 0, -64); 
}

static void
cal_render_background_medium(cairo_t* cr, CalHomePlugin* desktop_plugin)
{
  cairo_translate(cr, 0, 64); 
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -64); 
  cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, 5*36); 
  cairo_fill(cr); 
  cairo_translate(cr, 0, -64); 
}

static void
cal_render_background_large(cairo_t* cr, CalHomePlugin* desktop_plugin)
{
  cairo_translate(cr, 0, 396-(5*36+8));
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -64);
  cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, 5*36+8);
  cairo_fill(cr);
  cairo_translate(cr, 0, -(396-(5*36+8)));
  cairo_translate(cr, 0, 64); 
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -64); 
  cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, 4*36); 
  cairo_fill(cr); 
  cairo_translate(cr, 0, -64); 
}

static void
cal_render_background_auto(cairo_t* cr, CalHomePlugin* desktop_plugin)
{
  CalSize count = (CalSize)gtk_tree_model_iter_n_children(GTK_TREE_MODEL(desktop_plugin->event_list), NULL); 
  if(count == SMALL_ROW_COUNT)
  {
    cal_render_background_small(cr, desktop_plugin);
  }
  else if(count == MEDIUM_ROW_COUNT)
  {
    cal_render_background_medium(cr, desktop_plugin);
  }
  else if(count == LARGE_ROW_COUNT)
  {
    cal_render_background_large(cr, desktop_plugin);
  }
  else
  {
    if(count>5)
    {
      int height = (CAL_WIDGET_ENTRY_ROW_HEIGHT*count)+CAL_WIDGET_TITLE_HEIGHT+CAL_WIDGET_BOTTON_HEIGHT;
      cairo_translate(cr, 0, height-(5*36+8));
      gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -64);
      cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, 5*36+8);
      cairo_fill(cr);
      cairo_translate(cr, 0, -(height-(5*36+8)));


      cairo_translate(cr, 0, 64); 
      gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -64 -(9-count)*36);
      cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, (count-5)*36); 
      cairo_fill(cr); 
      cairo_translate(cr, 0, -64); 
    }
    else
    {
      cairo_translate(cr, 0, 64);
      gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, -64 - (5-count)*36);
      cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, count*36+8);
      cairo_fill(cr);
      cairo_translate(cr, 0, -64);
    }
  }
}

static void 
cal_expose(GtkWidget* widget, GdkEventExpose *event, gpointer data)
{
  CalHomePlugin* desktop_plugin = CAL_HOME_PLUGIN(data);
  cairo_t* cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->applet_bg, 0, 0);
  cairo_rectangle(cr, 0, 0, CAL_WIDGET_WIDTH, 64);
  cairo_fill(cr);
  if(desktop_plugin->cal_size == CAL_SIZE_MEDIUM)
  {
    cal_render_background_medium(cr, desktop_plugin);
  }
  else if(desktop_plugin->cal_size == CAL_SIZE_LARGE)
  {
    cal_render_background_large(cr, desktop_plugin);
  }
  else if(desktop_plugin->cal_size == CAL_SIZE_SMALL)
  {
    cal_render_background_small(cr, desktop_plugin);
  }
  else if(desktop_plugin->cal_size == CAL_SIZE_AUTO)
  {
    cal_render_background_auto(cr, desktop_plugin);
  }
  cairo_destroy(cr);
}

static void
register_gconf_notify_on_timeformat_changed(CalHomePlugin* desktop_plugin)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    desktop_plugin->gconf_notify_handler =0;
    return;
  }
  GError* error = NULL;
  guint handle_id = gconf_client_notify_add(client, GCONF_TIME_FORMAT_PATH,
					    handle_time_format_changed, desktop_plugin, NULL, &error);
  if(error==NULL)
  {
    desktop_plugin->gconf_notify_handler = handle_id;
  }
  else
  {
    g_error_free(error);
  }
  g_object_unref(client);
}

static void
register_dbus_signal_on_time_changed(CalHomePlugin *desktop_plugin)
{
  desktop_plugin->dbus_system_conn = hd_home_plugin_item_get_dbus_g_connection(&desktop_plugin->hitem, 
									       DBUS_BUS_SYSTEM, 
									       NULL );

  if(desktop_plugin->dbus_system_conn)
  {
    desktop_plugin->dbus_clockd_proxy = dbus_g_proxy_new_for_name(desktop_plugin->dbus_system_conn,
								  CLOCKD_SERVICE,
								  CLOCKD_PATH,
								  CLOCKD_INTERFACE);
    dbus_g_proxy_add_signal(desktop_plugin->dbus_clockd_proxy,
			    CLOCKD_TIME_CHANGED,
			    G_TYPE_INT, G_TYPE_INVALID);

    dbus_g_proxy_connect_signal(desktop_plugin->dbus_clockd_proxy,
				CLOCKD_TIME_CHANGED,
				G_CALLBACK(handle_time_changed), desktop_plugin, NULL);
  }
}

static void
register_dbus_signal_on_db_changed(CalHomePlugin *desktop_plugin)
{
  desktop_plugin->dbus_session_conn = hd_home_plugin_item_get_dbus_g_connection(&desktop_plugin->hitem, 
										DBUS_BUS_SESSION, 
										NULL );
  if(desktop_plugin->dbus_session_conn)
  {
    desktop_plugin->dbus_calendar_proxy = dbus_g_proxy_new_for_name(desktop_plugin->dbus_session_conn,
								    CALENDAR_SERVICE,
								    CALENDAR_PATH,
								    CALENDAR_INTERFACE);

  }
  dbus_bus_add_match(dbus_g_connection_get_connection(desktop_plugin->dbus_session_conn), 
		     "type='signal',interface='com.nokia.calendar'", 
		     NULL);

  dbus_connection_add_filter(dbus_g_connection_get_connection(desktop_plugin->dbus_session_conn), 
			     handle_db_change, 
			     desktop_plugin, 
			     NULL);
}

static void
cal_home_init_default_calendar_list_setting(CalHomePlugin *desktop_plugin)
{
  desktop_plugin->number_visible_cals = CMulticalendar::MCInstance()->getNoofCalendars();
  desktop_plugin->visible_cals = g_new0(gint, desktop_plugin->number_visible_cals);
  vector<CCalendar*> calendars = CMulticalendar::MCInstance()->getListCalFromMc();
  for(int i=0;i<desktop_plugin->number_visible_cals;++i)
  {
    desktop_plugin->visible_cals[i] = calendars[i]->getCalendarId();
  }
  CMulticalendar::MCInstance()->releaseListCalendars(calendars);
}

static void
cal_load_original_bg(CalHomePlugin* desktop_plugin)
{
  desktop_plugin->applet_bg = NULL;
  GdkPixbuf* bg = gdk_pixbuf_new_from_file("/usr/share/themes/default/images/CalendarAppletBackground.png", NULL);
  if(bg == NULL)
    return;
  GdkPixbuf* stride = gdk_pixbuf_new_subpixbuf(bg, 0, 0, CAL_WIDGET_WIDTH, 252);
  desktop_plugin->applet_bg = gdk_pixbuf_copy(stride);
  g_object_unref(bg);
}

static void
cal_home_plugin_init(CalHomePlugin *desktop_plugin)
{
  cal_load_original_bg(desktop_plugin);
  
  desktop_plugin->event_list = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_STRING, GDK_TYPE_PIXBUF);
  desktop_plugin->dbus_system_conn = NULL;
  desktop_plugin->dbus_clockd_proxy = NULL;
  desktop_plugin->dbus_session_conn = NULL;
  desktop_plugin->dbus_calendar_proxy = NULL;
  desktop_plugin->launch_view = 1;
  desktop_plugin->cal_time_period = CAL_TIME_PERIOD_12MONTH;
  desktop_plugin->cal_size = CAL_SIZE_MEDIUM;
  desktop_plugin->has_24_time_format = cal_24_time_format();
  desktop_plugin->update_view = TRUE;

  GtkWidget *contents = build_ui(desktop_plugin);
  gtk_window_set_default_size (GTK_WINDOW (desktop_plugin), CAL_WIDGET_WIDTH, 252);

  cal_home_init_default_calendar_list_setting(desktop_plugin);

  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);

  hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);

  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(cal_show_settings_dialog), NULL);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(cal_on_current_desktop), NULL);
  g_signal_connect(GTK_CONTAINER(contents), "button-release-event", G_CALLBACK(cal_home_button_release), desktop_plugin);

  g_signal_connect(G_OBJECT(contents), "expose_event", G_CALLBACK(cal_expose), desktop_plugin);
  register_time_out_handler(desktop_plugin);
  register_gconf_notify_on_timeformat_changed(desktop_plugin);  
  register_dbus_signal_on_time_changed(desktop_plugin);
  register_dbus_signal_on_db_changed(desktop_plugin);
}

static void
cal_plugin_finalize(GObject *object)
{
  CalHomePlugin *desktop_plugin = CAL_HOME_PLUGIN(object);
  if(desktop_plugin->time_out_handler)
  {
    g_source_remove(desktop_plugin->time_out_handler);
    desktop_plugin->time_out_handler = 0;
  }

  if(desktop_plugin->dbus_clockd_proxy)
    g_object_unref(desktop_plugin->dbus_clockd_proxy);

  if(desktop_plugin->dbus_calendar_proxy)
    g_object_unref(desktop_plugin->dbus_calendar_proxy);

  dbus_g_connection_unref(desktop_plugin->dbus_system_conn);  
  dbus_g_connection_unref(desktop_plugin->dbus_session_conn);  
  g_free(desktop_plugin->iD);
  G_OBJECT_CLASS (cal_home_plugin_parent_class)->finalize (object);
}

static void
cal_home_plugin_class_init(CalHomePluginClass *klass) 
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = cal_realize;
  G_OBJECT_CLASS(klass)->finalize = cal_plugin_finalize;
}

static void
cal_home_plugin_class_finalize(CalHomePluginClass *klass) 
{
}

