/*
 *  Birthday application for Maemo.
 *  Copyright (C) 2010 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include <libosso.h>

#include <glib.h>
#include <glib/gi18n.h>

#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include <libebook/e-book.h>
#include <libosso-abook/osso-abook.h>

enum
{
	COLUMN_AVATAR = 0,
	COLUMN_DISPLAY,
	COLUMN_FULLNAME,
	COLUMN_NEXT_BIRTHDAY,
	NUM_COLS
};

static unsigned int
calc_age (EContactDate *bdate)
{
    struct tm tm_bday;
    struct tm *tm_age;
    time_t t_age;
    int age = 0;

    tm_bday.tm_sec = 0;
    tm_bday.tm_min = 0;
    tm_bday.tm_hour = 0;
    tm_bday.tm_mday = bdate->day;
    tm_bday.tm_mon = bdate->month - 1;
    tm_bday.tm_year = bdate->year - 1900;

    t_age = time (NULL) - mktime (&tm_bday);
    tm_age = gmtime (&t_age);
    age = tm_age->tm_year - 70;

    if (age < 0)
	age = 0;

    return age;
}

static unsigned int
calc_next_bday (EContactDate *bdate)
{
    struct tm tm_current_bday, tm_next_bday;
    struct tm *tm_current_date;
    time_t t_current_date, t_current_bday, t_next_bday, t_next_bday_in;

    t_current_date = time (NULL);
    tm_current_date = localtime (&t_current_date);

    tm_current_bday.tm_sec = 0;
    tm_current_bday.tm_min = 0;
    tm_current_bday.tm_hour = 0;
    tm_current_bday.tm_mday = bdate->day;
    tm_current_bday.tm_mon = bdate->month - 1;
    tm_current_bday.tm_year = tm_current_date->tm_year;
    t_current_bday = mktime (&tm_current_bday);

    if (t_current_date > t_current_bday) {
	tm_next_bday.tm_sec = 0;
	tm_next_bday.tm_min = 0;
	tm_next_bday.tm_hour = 0;
	tm_next_bday.tm_mday = bdate->day;
	tm_next_bday.tm_mon = bdate->month - 1;
	tm_next_bday.tm_year = tm_current_date->tm_year + 1;
	t_next_bday = mktime (&tm_next_bday);
    } else {
	t_next_bday = t_current_bday;
    }
    t_next_bday_in = t_next_bday - t_current_date;
    return (t_next_bday_in / 86400) + 1;
}

static gchar *
get_text_font_by_name (const gchar *name)
{
	GtkSettings *settings;
	GtkStyle *style;

	settings = gtk_settings_get_default ();
	style = gtk_rc_get_style_by_paths (settings, name, NULL, G_TYPE_NONE);
	return pango_font_description_to_string (style->font_desc);
}

static gchar *
get_text_color_by_name (const gchar *name)
{
	GtkSettings *settings;
	GtkStyle *style;
	GdkColor color;

	settings = gtk_settings_get_default ();
	style = gtk_rc_get_style_by_paths (settings, "GtkButton", "osso-logical-colors", G_OBJECT_TYPE(gtk_button_new()));
	gtk_style_lookup_color (style, name, &color);
	return gdk_color_to_string (&color);
}

static GtkListStore *
create_bday_liststore(GList *contacts)
{
	GtkListStore *store;
	GtkTreeIter iter;
	GList *contact;
	gchar *text_font = NULL;
	gchar *text_color = NULL;

	text_font = get_text_font_by_name ("SmallSystemFont");
	text_color = get_text_color_by_name ("SecondaryTextColor");

	store = gtk_list_store_new(NUM_COLS,
				   GDK_TYPE_PIXBUF,	/* COLUMN_AVATAR */
				   G_TYPE_STRING,	/* COLUMN_DISPLAY */
				   G_TYPE_STRING,	/* COLUMN_FULLNAME */
				   G_TYPE_INT);		/* COLUMN_NEXT_BIRTHDAY */

	for (contact = contacts; contact != NULL; contact = contact->next) {
		EContactDate *bdate;

		bdate = e_contact_get (E_CONTACT (contact->data), E_CONTACT_BIRTH_DATE);
		if (bdate) {
			EContactPhoto *photo;
			GError *error = NULL;
			GdkPixbuf *avatar;
			gchar *avatar_filename = NULL;
			gchar *fullname = NULL;
			guint age = 0, next_birthday = 0;
			gchar *display_column = NULL;
			gchar *next_birthday_text = NULL;

#if 0
			photo = e_contact_get (E_CONTACT (contact->data), E_CONTACT_PHOTO);
			if (photo) {
				avatar_filename = g_filename_from_uri (photo->data.uri, NULL, NULL);
				if (avatar_filename) {
					gint height = 0;

					avatar = gdk_pixbuf_new_from_file (avatar_filename, &error);
					height = gdk_pixbuf_get_height (avatar);
					if (height != 48) {
						gint new_height = 48;
						gint new_width = (int) new_height * gdk_pixbuf_get_width (avatar) / height;
						gdk_pixbuf_scale_simple (avatar, new_width, new_height, GDK_INTERP_BILINEAR);
					}
					g_free (avatar_filename);
				}
				avatar_filename = NULL;
			} else {
				avatar = gdk_pixbuf_new_from_file ("/usr/share/icons/hicolor/48x48/hildon/general_default_avatar.png", &error);
			}
#endif
				avatar = gdk_pixbuf_new_from_file ("/usr/share/icons/hicolor/48x48/hildon/general_default_avatar.png", &error);

			fullname = e_contact_get (E_CONTACT (contact->data), E_CONTACT_FULL_NAME);
			age = calc_age(bdate);
			next_birthday = calc_next_bday(bdate);
			next_birthday_text = g_strdup_printf(ngettext ("next birthday in %d day", "next birthday in %d days", next_birthday), next_birthday);
			display_column = g_strdup_printf("%s <span font_desc=\"%s\" foreground=\"%s\"><sup>(%d)</sup>\n%02d.%02d.%04d, %s</span>",
							 fullname, text_font, text_color, age, bdate->day, bdate->month, bdate->year, next_birthday_text);

			gtk_list_store_append (store, &iter);
			gtk_list_store_set (store, &iter,
					    COLUMN_AVATAR, avatar,
					    COLUMN_DISPLAY, display_column,
					    COLUMN_FULLNAME, fullname,
					    COLUMN_NEXT_BIRTHDAY, next_birthday,
					    -1);

			if (next_birthday_text)
				g_free (next_birthday_text);
			next_birthday_text = NULL;

			e_contact_date_free (bdate);
		}
		bdate = NULL;
	}

	if (text_font)
		g_free (text_font);

	if (text_color)
		g_free (text_color);

	return store;
}

static GtkWidget *
create_main_window(GtkListStore *store)
{
	HildonProgram *program = NULL;
	GtkWidget *window, *main_vbox, *alignment, *label, *pannable, *tree_view;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	program = hildon_program_get_instance ();
	g_set_application_name (_("Birthday"));

	/* main window */
	window = hildon_stackable_window_new ();
	hildon_program_add_window (program, HILDON_WINDOW (window));

	/* aligment */
	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment),
				   HILDON_MARGIN_HALF, 0, HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (window), alignment);

	/* main vbox */
	main_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (alignment), main_vbox);

	/* no_search_result label */
	label = gtk_label_new (_("No contacts with set birthdate"));
	hildon_helper_set_logical_color (label, GTK_RC_FG,
					 GTK_STATE_NORMAL, "SecondaryTextColor");
	hildon_helper_set_logical_font (label, "LargeSystemFont");
	gtk_box_pack_start (GTK_BOX (main_vbox), label, TRUE, TRUE, 0);

	/* alignment for pannable area */
	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment),
				   0, 0, HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (main_vbox), alignment, TRUE, TRUE, 0);

	/* pannable for tree view */
	pannable = hildon_pannable_area_new ();
	g_object_set (G_OBJECT (pannable), "mov-mode", HILDON_MOVEMENT_MODE_VERT, NULL);
	gtk_container_add (GTK_CONTAINER (alignment), pannable);

	/* tree view */
	tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	gtk_container_add (GTK_CONTAINER (pannable), tree_view);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (store),
					      COLUMN_NEXT_BIRTHDAY, GTK_SORT_ASCENDING);

	/* display column */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_fixed_width (column, 696);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
					     "markup", COLUMN_DISPLAY,
					     NULL);
	g_object_set (G_OBJECT (renderer), "xpad", 10, NULL);

	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

	/* avatar column */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_fixed_width (column, 64);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_end (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
					     "pixbuf", COLUMN_AVATAR,
					     NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

	/* window signals */
	g_signal_connect (G_OBJECT (window), "destroy", G_CALLBACK (gtk_main_quit), NULL);

	gtk_widget_show_all (GTK_WIDGET(window));
	gtk_widget_hide (GTK_WIDGET(label));
	return window;
}

int main (int argc, char **argv)
{
	osso_context_t *osso_context;
	EBook *ebook;
	EBookQuery *query;
	GError *error = NULL;
	GtkWidget *window;
	GtkListStore *store;
	GList *contacts;

	hildon_gtk_init (&argc, &argv);

	/* initialize localization */
	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	/* initialize osso */
	osso_context = osso_initialize ("birthday", "0.1", TRUE, NULL);
	if (osso_context == NULL) {
		g_critical ("Error initializing osso");
		return 1;
	}

	/* init abook */
	if (!osso_abook_init (&argc, &argv, osso_context)) {
		g_critical ("Error initializing libosso-abook");
		goto exit;
	}

	ebook = e_book_new_system_addressbook (&error);
	if (!ebook) {
		g_warning ("Error opening system address book: %s", error->message);
		g_error_free (error);
		goto exit;
	}

	if (!e_book_open (ebook, TRUE, &error)) {
		g_warning ("Error opening system address book: %s", error->message);
		g_error_free (error);
		goto exit;
	}

	query = e_book_query_any_field_contains ("");

	if (!e_book_get_contacts (ebook, query, &contacts, &error)) {
		g_warning ("Error getting contacts: %s", error->message);
		g_error_free (error);
		goto exit;
	}

	store = create_bday_liststore (contacts);
	window = create_main_window (store);

	gtk_main ();

exit:
	osso_deinitialize (osso_context);
	return 0;
}
