{$INCLUDE valkyrie.inc}
// @abstract(Valkyrie Core - Valkyrie Utilities)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(7 May, 2004)
// @lastmod(23 Nov, 2005)
//
// Common utility functions, defines and objects for Valkyrie.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit vutil;
interface
uses SysUtils;

type
// Type for 32bit flags
     T32BitRange  = 0..31;
// 256 bit flags type
     TFlags       = set of Byte;
// Pointer to TFlags
     PFlags       = ^TFlags;
// 32 bit flags type
     TFlags32     = set of T32BitRange;
// Unique IDentification number. QWord may prove not enough?
     TUID         = QWord;
// Message IDentification number.
     TMSGID       = Word;
// Type identification number. Used to save/load.
     TIDN         = Word;
// Translation table of Printable characters
     TPrintableCharToByte = array[33..126] of Byte;
//
     TAnsiStringArray = array of AnsiString;
//
     TVersion     = array [1..4] of Byte;



     
const
  Black        = 0;  //< Color black
  Blue         = 1;  //< Color blue
  Green        = 2;  //< Color green
  Cyan         = 3;  //< Color cyan
  Red          = 4;  //< Color red
  Magenta      = 5;  //< Color magenta
  Brown        = 6;  //< Color brown
  LightGray    = 7;  //< Color light gray
  DarkGray     = 8;  //< Color dark gray
  LightBlue    = 9;  //< Color light blue
  LightGreen   = 10; //< Color light green
  LightCyan    = 11; //< Color light cyan
  LightRed     = 12; //< Color light red
  LightMagenta = 13; //< Color light magenta
  Yellow       = 14; //< Color yellow
  White        = 15; //< Color white
  
// Reads a single line from the given textfile and returns it.
// Used by many Valkyrie projects for versioning.
//
// @param( aFileName Text file with first line as version string )
// @returns( First line of the text file )
// @raises( EException if file not found or cannot be read )
function ReadVersion( const aFileName : string ) : string;

// A virtual die. number is the number of dice, sides the number of sides.
// Note that Randomize must be called before. If any of the parameters is
// zero, the result is zero as well.
//
// @param( aNumber number of dice to be rolled )
// @param( aSides number of sides on the dice )
// @returns( Result of the dice roll )
function Dice( aNumber : DWord; aSides : DWord ) : DWord;

// Converts string to a boolean.
//
// @param( aString String to be converted )
// @returns( @true if string starts with 't', 'T', 'y', 'Y', @false otherwise )
function StrToBool( const aString : AnsiString ) : Boolean;

// Converts boolean to a string.
//
// @param( aBoolean Boolean to be converted )
// @returns( 'TRUE' if aBoolean is @true, 'FALSE' if @false )
function BoolToStr(const aBoolean : boolean) : string;

// Reads a parameter from the string. Depending on the length of aParamChar:
// Lenght 1 returns delimeted word number aNumber, length 2 returns the word
// between the two characters, length three - parameter delimeted by character
// number 2, between characters 1 and 3.
//
// @deprecated
// Use ExtractDelimited and ExtractSubStr instead.
function Parameter( const aString : AnsiString; aNumber : Byte; const aParamChar: AnsiString ) : AnsiString; deprecated;

// Crops the String to given length. Returned string will be at most
// the passed length.
//
// @param( aString String to be cropped )
// @param( aLength Length the string is to be cropped to )
// @returns( String cropped to the desired length )
function CroppedString( const aString : Ansistring; aLength : Word ) : Ansistring;

// Sets a string to given length in place, cropping it if needed, and padding
// it with spaces or the passed character if length is greater than the one
// passed.
//
// @param( aString String to be padded to the desired length )
// @param( aLength Length the string is to be padded to )
// @param( aPadChar Padding character, space by default )
procedure Pad( var aString : AnsiString; aLength : byte; aPadChar : Char = ' ' );

// Produces a string to given length from the one passed, cropping it if needed,
// and padding it with spaces or the passed character if length is greater than
// the one passed.
//
// @param( aString Source string to be padded to the desired length )
// @param( aLength Length the string is to be padded to )
// @param( aPadChar Padding character, space by default )
function Padded( const aString : AnsiString; aLength : byte; aPadChar : Char = ' ' ) : AnsiString;

// Returns the string with the first letter capitalized.
// Deprecated, use Capitalized instead.
function CapLet(const s : Ansistring):Ansistring; deprecated;

// Produces a time string of the format DD.MM.YY HH:MM.
//
// @returns( DateTimeToStr( Now ) )
function TimeStamp : string;

// Splits a string into two substrings. Split is performed at the first split
// character position after split distance. The split character is not present
// in any of the output strings.
//
// If the split character is not present, the string is returned in the first
// string, if it is the first character, then the string is returned in the
// second one.
//
// @param( aString String to be splitted )
// @param( aFirstResult String to the left of the split character )
// @param( aSecondResult String to the right of the split character )
// @param( aSplitChar The character to split the string, space by default )
// @param( aSplitPos The position from which to search for the split character,
//         zero by default )
procedure Split( const aString : Ansistring;
                 out aFirstResult, aSecondResult : Ansistring;
                 aSplitChar : Char = ' '; aSplitDist : Byte = 0);

// Formats the string the Valkyrie way -- replacing @1..@X with the passed
// parameters. Supports conversion for integers and floats.
//
// @param( aString String to be formated )
// @param( aParams Parameters to be converted and injected into the string )
function VFormat( const aString : Ansistring; const aParams : array of Const ) : Ansistring;

// Returns the string with the first letter capitalized.
//
// @param( aString String to be capitalized )
// @returns( Capitalized string )
function Capitalized( const aString : Ansistring ) : Ansistring;

// Sequential non-existing filename return
function SequentialFilename( const Name, Ext : Ansistring; Numbers : Byte = 4) : Ansistring;

// Converts version string of the format "x.y.z trail..." to TVersion format
function StringToVersion( const VersionString : AnsiString ) : TVersion;

// Converts TVersion to string
function VersionToString( const Version : TVersion ) : Ansistring;

// Object for representing and passing a integer based rectangle. Useful
// for constructing GUI's. While storing the dimensions as two pairs of
// coordinates allows direct width and height manipulation.
//
// Width and height equal to zero mean that the coordinates overlap.
//
// @deprecated
// This class was specifically written for GUI's. You may want to use @link(TArea)
// for logic.
//
// @seealso(TArea)
type TRect = object
  // Pair of coordinates as integers
  x1,y1,x2,y2 : Integer;
  // Constructor, takes two pairs of integers representing coordinates.
  procedure Create(nx1,ny1,nx2,ny2 : Integer);
  // Sets the width to the desired value.
  procedure SetWidth(nw : Integer);
  // Sets the geight to the desired value.
  procedure SetHeight(nh : Integer);
  // Returns the width of the rectangle.
  function GetWidth : Integer;
  // Returns the width of the rectangle.
  function GetHeight : Integer;
  // Property for width
  property w : Integer read GetWidth write SetWidth;
  // Property for height
  property h : Integer read GetHeight write SetHeight;
  // Property for first coordinate x value
  property x : Integer read x1 write x1;
  // Property for first coordinate y value
  property y : Integer read y1 write y1;
end;

// Returns a rect of the given coordinates.
function NewRectXY(x1,y1,x2,y2 : Integer) : TRect;

// Returns a rect based on the passed coordinate and width and height value.
function NewRectWH(x1,y1,w,h : Integer) : TRect;

// Valkyrie's base exception class. Allows construction via VFormat.
type EException = class(Exception)
  // Create the exception, just a override of Exception.Create
  constructor Create( const aMessage : AnsiString );
  // Create the exception, formating the arguments.
  constructor Create( const aMessage : AnsiString; const aParams : array of Const );
end;


operator = ( a : TVersion; b : TVersion) : Boolean;
operator > ( a : TVersion; b : TVersion) : Boolean;
operator >= ( a : TVersion; b : TVersion) : Boolean;

implementation
uses strutils;

operator = ( a : TVersion; b : TVersion) : Boolean;
begin
  Exit( (a[1] = b[1]) and (a[2] = b[2]) and (a[3] = b[3]) and (a[4] = b[4]) );
end;

operator >= ( a : TVersion; b : TVersion) : Boolean;
begin
  if a[1] <> b[1] then Exit( a[1] > b[1] );
  if a[2] <> b[2] then Exit( a[2] > b[2] );
  if a[3] <> b[3] then Exit( a[3] > b[3] );
  if a[4] <> b[4] then Exit( a[4] > b[4] );
  Exit( True );
end;

operator > ( a : TVersion; b : TVersion) : Boolean;
begin
  if a[1] <> b[1] then Exit( a[1] > b[1] );
  if a[2] <> b[2] then Exit( a[2] > b[2] );
  if a[3] <> b[3] then Exit( a[3] > b[3] );
  if a[4] <> b[4] then Exit( a[4] > b[4] );
  Exit( False );
end;

function NewRectXY(x1, y1, x2, y2 : Integer) : TRect;
begin
  NewRectXY.x1 := x1;
  NewRectXY.y1 := y1;
  NewRectXY.x2 := x2;
  NewRectXY.y2 := y2;
end;

function NewRectWH(x1, y1, w, h : Integer) : TRect;
begin
  NewRectWH.x1 := x1;
  NewRectWH.y1 := y1;
  NewRectWH.x2 := x1+w-1;
  NewRectWH.y2 := y1+h-1;
end;

{ TRect }

procedure TRect.Create(nx1, ny1, nx2, ny2 : Integer);
begin
  x1 := nx1;
  y1 := ny1;
  x2 := nx2;
  y2 := ny2;
end;

procedure TRect.SetWidth(nw : Integer);
begin
  x2 := x1+nw-1;
end;

procedure TRect.SetHeight(nh : Integer);
begin
  y2 := y1+nh-1;
end;

function TRect.GetWidth : Integer;
begin
  Exit(x2-x1+1);
end;

function TRect.GetHeight : Integer;
begin
  Exit(y2-y1+1);
end;


function VFormat( const aString : Ansistring; const aParams : array of Const) : Ansistring;
var iCount : DWord;
begin
  VFormat := aString;
  for iCount := 1 to High( aParams ) + 1 do
  begin
    case aParams[ iCount - 1 ].vtype of
      vtInteger    : VFormat := AnsiReplaceStr( VFormat, '@' + IntToStr( iCount ), IntToStr( aParams[iCount-1].vinteger ) );
      vtString     : VFormat := AnsiReplaceStr( VFormat, '@' + IntToStr( iCount ), aParams[iCount-1].vstring^ );
      vtExtended   : VFormat := AnsiReplaceStr( VFormat, '@' + IntToStr( iCount ), FloatToStr( aParams[iCount-1].vextended^ ) );
      vtAnsiString : VFormat := AnsiReplaceStr( VFormat, '@' + IntToStr( iCount ), AnsiString( aParams[iCount-1].vansistring ) );
    end;
  end;
end;

function Capitalized( const aString: Ansistring ): Ansistring;
begin
  Capitalized    := aString;
  Capitalized[1] := UpCase( Capitalized[1] );
end;

function SequentialFilename(const Name, Ext: Ansistring; Numbers : Byte = 4 ): Ansistring;
var Count : DWord;
    Full  : AnsiString;
begin
  Count := 0;
  repeat
    Full := IntToStr(Count);
    if Length(Full) > Numbers then Exit('');
    Full := Name + AddChar( '0', IntToStr(Count), Numbers ) + Ext;
    Inc(Count);
  until not FileExists(Full);
  Exit(Full);
end;

function StringToVersion( const VersionString : AnsiString ) : TVersion;
var Temp  : AnsiString;
    CB,CE : Byte;
begin
  for CB := 1 to 4 do StringToVersion[CB] := 0;
  if Length(VersionString) = 0 then Exit;
  CB := 1;
  while (CB <= Length(VersionString)) and (not (VersionString[CB] in ['0','1'..'9'])) do Inc(CB);
  CE := CB;
  while (CE <= Length(VersionString)) and (VersionString[CE] in ['0','1'..'9','.']) do Inc(CE);
  Temp := MidStr(VersionString, CB, CE-CB);
  StringToVersion[1] := StrToIntDef( ExtractDelimited( 1, Temp, ['.'] ), 0 );
  StringToVersion[2] := StrToIntDef( ExtractDelimited( 2, Temp, ['.'] ), 0 );
  StringToVersion[3] := StrToIntDef( ExtractDelimited( 3, Temp, ['.'] ), 0 );
  StringToVersion[4] := StrToIntDef( ExtractDelimited( 4, Temp, ['.'] ), 0 );
end;

function VersionToString( const Version : TVersion ) : Ansistring;
begin
  VersionToString := IntToStr( Version[1] )+'.'+IntToStr( Version[2] )+'.'+IntToStr( Version[3] );
  if Version[4] <> 0 then VersionToString += '.'+IntToStr( Version[4] );
end;

procedure Split( const aString : Ansistring;
                 out aFirstResult, aSecondResult : Ansistring;
                 aSplitChar : Char = ' '; aSplitDist : Byte = 0 );
var iSplitCharPos : Word;
begin
  if aSplitDist = 0 then
    iSplitCharPos := Pos( aSplitChar, aString )
  else
  begin
    iSplitCharPos := aSplitDist;
    if iSplitCharPos < Length( aString )
      then while ( iSplitCharPos > 0 ) and ( aString[iSplitCharPos] <> aSplitChar ) do Dec( iSplitCharPos )
      else iSplitCharPos := 0;
  end;

  if iSplitCharPos = 0 then begin aFirstResult := aString; aSecondResult := '';      Exit; end;
  if iSplitCharPos = 1 then begin aFirstResult := '';      aSecondResult := aString; Exit; end;

  aFirstResult  := AnsiLeftStr ( aString, iSplitCharPos - 1 );
  aSecondResult := AnsiRightStr( aString, Length( aString ) - iSplitCharPos );
end;

function CroppedString( const aString : Ansistring; aLength : Word) : Ansistring;
begin
  CroppedString := aString;
  Delete( CroppedString, aLength + 1 ,255 );
end;

procedure Pad( var aString : AnsiString; aLength : Byte; aPadChar : Char = ' ' );
var iOldLength : Word;
begin
  iOldLength := Length( aString );
  SetLength( aString, aLength );
  if aLength > iOldLength then
    FillChar( aString[ iOldLength + 1 ], aLength - iOldLength, aPadChar );
end;

function Padded( const aString : AnsiString; aLength : Byte; aPadChar : Char = ' ' ) : AnsiString;
begin
  Padded := aString;
  Pad( Padded, aLength, aPadChar );
end;

function CapLet(const s : Ansistring):Ansistring;
begin
  Exit( Capitalized( s ) );
end;

function Parameter( const aString : AnsiString; aNumber : Byte; const aParamChar: AnsiString ) : AnsiString;
var iPos : Integer;
begin
  if Length( aParamChar ) = 1 then Parameter := ExtractDelimited( aNumber, aString, [ aParamChar[1] ]) else
  if Length( aParamChar ) = 2 then
  begin
    iPos := Pos( aParamChar[1], aString ) + 1;
    Parameter := ExtractSubStr( aString, iPos, [ aParamChar[2] ] )
  end
  else
  begin
    iPos := Pos( aParamChar[1], aString ) + 1;
    Parameter := ExtractDelimited( aNumber, ExtractSubStr( aString, iPos, [ aParamChar[3] ] ), [ aParamChar[2], aParamChar[3] ]);
  end;
end;

function StrToBool( const aString : AnsiString ) : Boolean;
begin
  if length( aString ) < 1 then Exit( False );
  Exit( aString [1] in ['t','T','y','Y'] );
end;


function Dice( aNumber : DWord; aSides : DWord ) : DWord;
var iCount : DWord;
begin
  Dice := 0;
  if aNumber * aSides = 0 then Exit(0);
  if aSides = 1 then Exit( aNumber );
  for iCount := 1 to aNumber do
     Dice += Random( aSides ) + 1;
end;

function TimeStamp : string;
begin
  Exit(DateTimeToStr(Now));
end;

function BoolToStr( const aBoolean : boolean ) : string;
begin
  if aBoolean then Exit('TRUE') else exit('FALSE');
end;

function ReadVersion(const aFileName : string) : string;
var iTextFile : Text;
begin
  {$PUSH}
  {$I-}
  Assign(iTextFile,aFileName);
  Reset(iTextFile);
  Readln(iTextFile,ReadVersion);
  Close(iTextFile);
  {$POP} {restore $I}
  if IOResult <> 0 then raise EException.Create('Version file '+aFileName+' not found!');
end;

{ EException }

constructor EException.Create(const aMessage: AnsiString);
begin
  inherited Create( aMessage );
end;

constructor EException.Create(const aMessage: AnsiString; const aParams: array of const);
begin
  inherited Create( VFormat( aMessage, aParams ) );
end;

begin
  Randomize;
end.

//* 23.11.2004 Removed a lot of useless stuff.
// 2006-NOV-09 Removed TrimString (use SysUtils.Trim) and added new Split
//*

