{$INCLUDE valkyrie.inc}
unit vui;
{$H+}
interface
uses Classes, SysUtils, strutils, vmath, vutil, voutput, vinput, vnode, vds, vini;

type
TWindowStyle = record
  BackColor     : Byte;
  MainColor     : Byte;
  BoldColor     : Byte;
  TitleColor    : Byte;
  FrameColor    : Byte;
  InactiveColor : Byte;
  WrongColor    : Byte;
  Frame         : string[8];
  LetterChoicePre : string[8];
  LetterChoicePost: string[8];
  MenuType           : (mtChoice,mtLetter,mtHybrid);
  MenuHighlightType  : (mhInvert,mhBold,mhNormal);
end;
PWindowStyle = ^TWindowStyle;

const
VDefaultWindowStyle : TWindowStyle = (
  BackColor       : Black;
  MainColor       : LightGray;
  BoldColor       : White;
  TitleColor      : White;
  FrameColor      : DarkGray;
  InactiveColor   : DarkGray;
  WrongColor      : Red;
  Frame           : 'ĳĳڿ'; //'-|-|/++/'
  LetterChoicePre : '[';
  LetterChoicePost: ']';
  MenuType     : mtChoice;
  MenuHighlightType  : mhNormal;
);
LetterList = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';



type TUIElement = class(TNode,IUIElement)
  constructor Create(newParent : TUIElement); reintroduce;
  procedure Draw; virtual;
  protected
  Style : PWindowStyle;
  private
  function ParentIsElement : boolean;
end;

type

{ TTextUIArea }

TUIArea = class(TUIElement)
  constructor Create(newParent : TUIElement; newDimensions : TRect); virtual;
  constructor Create(newParent : TUIElement); reintroduce;
  function getAbsoluteDimensions : TRect; virtual;
  function AbsoluteDimensions : TRect;
  function ParentIsArea : boolean;
  public
  Dimensions : TRect;
end;

TUIContent = class(TUIArea)
  constructor Create(newParent : TUIElement; newDimensions : TRect; newText : Ansistring = nil);
  constructor Create(newParent : TUIElement; newText : Ansistring = nil);
  constructor Create(newParent : TUIElement; Line : byte; newText : Ansistring = nil);
  procedure SetText(newText : AnsiString); virtual;
  public
  LineBreak : Boolean;
  protected
  Text : Ansistring;
end;

TUILabel = class(TUIContent)
end;

TUIWindow = class(TUIArea)
  constructor Create(newParent : TUIElement; newDimensions : TRect; newTitle : AnsiString = ''); reintroduce;
  procedure Hide;
  procedure Show;
  procedure Top;
  public
  Title           : Ansistring;
  HBorder,VBorder : Byte;
  protected
  function getAbsoluteDimensions : TRect; override;
  protected
  Framed          : Boolean;
  Visible         : Boolean;
end;

TUISeparator = class(TUIArea)
  constructor Create(newParent : TUIArea; isHorizontal : Boolean; newPosition : Byte; newInset : Byte = 2; newLengthBonus : ShortInt = 0); reintroduce;
  protected
  Horizontal   : Boolean;
  Position     : Byte;
  Inset        : Byte;
  AreaA, AreaB : TUIArea;
  LengthBonus  : ShortInt;
  public
  property Top    : TUIArea read AreaA;
  property Bottom : TUIArea read AreaB;
  property Left   : TUIArea read AreaA;
  property Right  : TUIArea read AreaB;
end;

type TUIStringListArea = class(TUIArea)
  // Initializes the widget -- the string array passed is the one to be viewed.
  constructor Create(newParent : TUIElement; setStringList : TStringArray = nil); reintroduce;
  // Rewinds to the first strings (sets position 0)
  procedure First;
  // Rewinds to the last strings (sets position to Count-Height)
  procedure Recent;
  // Sets the string list
  procedure SetList(setStringList : TStringArray);
  protected
  Pos           : DWord;
  StringList    : TStringArray;
  public
  // Determinates the current position in viewing of the list. Position = 0 means that
  // Strings from 1 to Height will be shown.
  property Position : DWord read Pos write Pos;
end;

type TUIInputLine = class(TUIArea)
  // Initializes the widget
  constructor Create(newParent : TUIElement; newLength : Byte); reintroduce;
  // Runs the input line, returns the break key.
  function Run(BreakKeys : TKeyFilter = [VKEY_ENTER]) : Byte; virtual;
  // Returns the value of the string
  function Return : AnsiString;
  // Sets the string
  procedure SetLine(const newValue : ShortString);
  protected
  procedure Update; virtual;
  protected
  Value     : ShortString;
  Len       : Byte;
  Position  : Byte;
  EditChars : TKeyFilter;
end;

type TUIViewer = class(TUIArea)
  // Initializes the widget
  constructor Create(newParent : TUIElement; newTitle : Ansistring); reintroduce;
  // Runs the input line, returns the break key.
  function Run : Byte;
  // Sets a StringArray for viewing
  procedure SetTitle(newTitle : AnsiString);
  // Sets a StringArray for viewing
  procedure SetText(StringList : TStringArray);
  // Sets keys exiting run loop
  procedure SetExitKeys(aExitKeys : TKeyFilter);
  // Returns a pointer to the active stringlist
  function GetText : TStringArray;
  // Sets the view to recently added
  procedure Recent;
  // Sets the view to first added
  procedure First;
  protected
  procedure Update; virtual;
  function TextLines : DWord;
  protected
  StringArea : TUIStringListArea;
  ExitKeys   : TKeyFilter;
  Title      : AnsiString;
  public
  property Area : TUIStringListArea read StringArea;
end;

//
type TUIFullViewer = class(TUIViewer)
    constructor Create(newParent : TUIElement; newTitle : AnsiString ); reintroduce;
    procedure Load(const FName : string);
    procedure Load(stream : TStream);
    procedure SetArray(stringArray : TStringArray);
    destructor Destroy; override;
    protected
    Owner      : Boolean;
    StringList : TStringArray;
  end;


const
// If set, messages will be continued in the "current" line, and breaked apart.
    UIMSG_CONTINUE    = 1;
// If set, when buffer will overflow there will be a more message generated
// and the window will wait for enter.
    UIMSG_MORE        = 2;
// If set, then only active messages will be drawn.
    UIMSG_UPDATECLEAR = 3;
// If set, then message coloring active.
    UIMSG_COLORING    = 4;


type TUIMessageHighlight = record
    Wildcard : Ansistring;
    Color    : Byte;
  end;

TUIHighlightsArray = specialize TArray<TUIMessageHighlight>;


{ TUIMessages }

TUIMessages = class(TUIArea)
  // Creates a new message system. Pos and Size are used to place the Messages
  // on the textmode screen. BufferSize determines how many messages will be
  // stored by the system.
  constructor Create(newParent : TUIElement; nBufferSize : Word = 1000; const nOptions : TFlags32 = []; const nMoreText : string = '@B[more]'); reintroduce;
  // Adds a new message to the system. By default the message will be active.
  procedure Add(MessageText : Ansistring); reintroduce;
  // Clears the buffer.
  procedure ClearAll;
  // Updates the messages -- all messages are marked as "read" - inactive.
  // Should be run immidately AFTER Updating the screen.
  procedure Update;
  // Prints a more message and waits for enter.
  procedure More; virtual; abstract;
  // Get's a added message. The messages are indexed from the last added to
  // the last one in the buffer. That is -- Get(0) returns the last added
  // message, Get(1) returns the previous and so on...
  function Get(Index : Word) : Ansistring;
  // Destroys last message.
  procedure KillLast;
  // Loads Color highlights from INI file (see vini.pas)
  procedure LoadColors(INIFile : TINI);
  // Frees memory of the message system.
  destructor Destroy; override;
  // Returns amount of stored messages.
  function Size : DWord;
  // Callback for key-value
  procedure AddHighlight( Key, Value : Variant );
  protected
  // Holds the messages
  FMessages    : TMessageBuffer;
  // Holds the data for highlighting.
  FHighlights  : TUIHighlightsArray;
    // Number of highlights.
  FHighCount   : Word;
  // Holds the AMOUNT of active messages
  FActive      : Word;
  // Set of options (see flags)
  FOptions     : TFlags32;
  // The more message.
  FMoreText    : string[20];
  // The more message length.
  FMoreLength  : Byte;
end;

type IMap = interface
  function getASCII(X,Y : LongInt) : TPicture;
  function getColor(X,Y : LongInt) : DWord;
  function getTile(X,Y : LongInt; Layer : byte = 0) : DWord;
end;

const UIMAP_FIXED     = 1;
      UIMAP_CENTERING = 2;
      UIMAP_TRUECOLOR = 3;
      UIMAP_CURSOR    = 4;
      UIMAP_MARKING   = 5;

type TUIMap = class(TUIArea)
  constructor Create(newParent : TUIElement; newMap : IMap = nil; newOptions : TFlags32 = []); reintroduce;
  procedure setXY(X,Y : LongInt);
  procedure setMap(newMap : IMap);
  procedure Mark(X,Y : LongInt; sgn : char; atr : byte);
  procedure Update;
  function ScreenX(nWorldX : LongInt) : LongInt; inline;
  function ScreenY(nWorldY : LongInt) : LongInt; inline;
  function WorldX(nScreenX : LongInt) : LongInt; inline;
  function WorldY(nScreenY : LongInt) : LongInt; inline;
  protected
  MarkMap  : array of array of TPicture;
  FMap     : IMap;
  FADim    : TRect;
  FOptions : TFlags32;
  FX,FY    : LongInt;
  FShiftX  : LongInt;
  FShiftY  : LongInt;
end;

//TODO :
//  -- documentation
//  -- implementation
//  -- lettered version
//  -- scrolling if greater than space
//  -- HOME/END handling
//  -- different COLORed options!

{ TGenericTextUIMenu }

generic TGenericTextUIMenu<_TValue> = class(TUIArea)
  public type
    // Option for TTextMenu.
    TTextUIMenuField = record
      // Name of the option.
      name   : Ansistring;
      // Wether the option is active (you can hover over it)
      active : Boolean;
      // Value stored
      Value  : _TValue;
      // Color of the item
      Color  : Byte;
    end;
    // Calback procedure for TGenericTextUIMenu.
    TTextUIMenuCallback = procedure(choice : _TValue) of object;
  var public
  // Initialize the menu at position xp,yp, with optionaly : a initial choice,
  // a callback to be run at each change and wether Escape can be pressed.
  constructor Create(newParent : TUIElement; TerminationKeys : TKeyFilter = [VKEY_ENTER]; cback : TTextUIMenuCallback = nil; BrowseInactive : Boolean = False; initial : byte = 1); reintroduce;
  // Adds a choice to the menu system.
  procedure Add(name : Ansistring; value : _TValue; active : boolean = True; color : Byte = 0); reintroduce;
  // Runs the menu, and returns the termination key pressed
  function Run : byte; virtual;
  // Set max amount of elements (scroll)
  procedure SetScroll( ScrollValue : Byte );
  // Returns the value
  function Return : _TValue;
  // Sets active flag on given item
  procedure SetActive(Index : DWord; newActive : Boolean = True);
  protected
  procedure Update; virtual;
  procedure Draw; override;
  procedure ResizeValues(Value : DWord);
  protected
  Options     : TFlags32;
  Fields      : array of TTextUIMenuField;
  Callback    : TTextUIMenuCallback;
  Count       : DWord;
  Scroll      : DWord;
  ScrollSize  : DWord;
  Size        : DWord;
  Choice      : DWord;
  FirstActive : DWord;
  LastActive  : DWord;
  Escape      : Boolean;
  BInactive   : Boolean;
  TermKeys    : TKeyFilter;
  TermInactive: Boolean;
  public
  property Items : DWord read Count;
  property TerminateIncactive : Boolean read TermInactive write TermInactive;
end;

// specialization for TTextMenu
type TByteTextUIMenu = specialize TGenericTextUIMenu<Byte>;
     TByteTextUIMenuCallback = procedure(choice : Byte) of object;

// A UI menu with bytebased choices.
type

{ TUIMenu }

TUIMenu = class(TByteTextUIMenu)
    constructor Create(newParent : TUIElement; TerminationKeys : TKeyFilter = [VKEY_ENTER]; cback : TByteTextUIMenuCallback = nil; BrowseInactive : Boolean = False; initial : byte = 1); reintroduce;
    // Adds a choice to the menu system.
    procedure Add(name : Ansistring; active : boolean = True; color : byte = 0);
    // Runs the menu, and returns the value
    function Run : byte; override;
    public
    EscapeKey : Byte;
    EnterKey  : Byte;
  end;


implementation

constructor TUIArea.Create(newParent : TUIElement; newDimensions: TRect);
begin
  inherited Create(newParent);
  Dimensions := newDimensions;
end;

constructor TUIArea.Create(newParent : TUIElement);
var PAD : TRect;
begin
  inherited Create(newParent);
  Dimensions.X1 := 1;
  Dimensions.Y1 := 1;
  Dimensions.X2 := ScreenSizeX;
  Dimensions.Y2 := ScreenSizeY;
  if ParentIsArea then
  begin
    PAD := TUIArea(Parent).getAbsoluteDimensions;
    Dimensions.X2 := PAD.GetWidth;
    Dimensions.Y2 := PAD.GetHeight;
  end;
end;

function TUIArea.getAbsoluteDimensions: TRect;
begin
  Exit(AbsoluteDimensions);
end;

function TUIArea.AbsoluteDimensions: TRect;
var PAD : TRect;
begin
  if not ParentIsArea then Exit(Dimensions);
  PAD := TUIArea(Parent).getAbsoluteDimensions;
  Exit(NewRectXY(Dimensions.X1+PAD.X1-1,Dimensions.Y1+PAD.Y1-1,
                 Dimensions.X2+PAD.X1-1,Dimensions.Y2+PAD.Y1-1));
end;

function TUIArea.ParentIsArea: boolean;
begin
  Exit((Parent <> nil) and (Parent.InheritsFrom(TUIArea)));
end;

{ TUIElement }

constructor TUIElement.Create(newParent: TUIElement);
begin
  inherited Create;
  if newParent <> nil then newParent.Add(Self);
  Style := @VDefaultWindowStyle;
end;

procedure TUIElement.Draw;
var Scan : TNode;
begin
  Scan := Child;
  repeat
    if Scan = nil then Break;
    if Scan is TUIElement then
      TUIElement(Scan).Draw;
    Scan := Scan.Next;
  until Scan = Child;
end;

function TUIElement.ParentIsElement: boolean;
begin
  Exit((Parent <> nil) and Parent.InheritsFrom(TUIElement));
end;


constructor TUIContent.Create(newParent: TUIElement; newDimensions: TRect; newText : Ansistring = nil);
begin
  inherited Create(newParent, newDimensions);
  Text := newText;
  LineBreak := True;
end;

constructor TUIContent.Create(newParent: TUIElement; newText : Ansistring = nil);
begin
  inherited Create(newParent);
  Text := newText;
  LineBreak := True;
end;

constructor TUIContent.Create(newParent: TUIElement; Line: byte; newText: Ansistring);
begin
  inherited Create(newParent);
  Dimensions.Y1 := Line;
  Text := newText;
  LineBreak := True;
end;

procedure TUIContent.SetText(newText: AnsiString);
begin
  Text := newText;
end;


{ TTextUIWindow }

constructor TUIWindow.Create(newParent: TUIElement; newDimensions: TRect; newTitle : AnsiString = '');
begin
  inherited Create(newParent, newDimensions);
  HBorder := 3;
  VBorder := 2;
  Framed  := True;
  Visible := True;
  Title   := newTitle;
end;

procedure TUIWindow.Hide;
begin
  Visible := False;
end;

procedure TUIWindow.Show;
begin
  Visible := True;
end;

procedure TUIWindow.Top;
var ParentStore : TNode;
begin
  ParentStore := Parent;
  Detach;
  ParentStore.Add(Self);
end;

function TUIWindow.getAbsoluteDimensions: TRect;
begin
  Result:=inherited getAbsoluteDimensions;
  Result.X1 += HBorder;
  Result.X2 -= HBorder;
  Result.Y1 += VBorder;
  Result.Y2 -= VBorder;
end;

{ TUISeparator }

constructor TUISeparator.Create(newParent : TUIArea; isHorizontal : Boolean; newPosition : Byte; newInset : Byte; newLengthBonus : ShortInt);
begin
  inherited Create(newParent);
  Inset := newInset;
  Position := newPosition;
  Horizontal := isHorizontal;
  LengthBonus := newLengthBonus;
  if Horizontal then
  begin
    AreaA := TUIArea.Create(Self,NewRectXY(1,1,Dimensions.x2,Position-Inset));
    AreaB := TUIArea.Create(Self,NewRectXY(1,Position+Inset,Dimensions.x2,Dimensions.y2));
  end
  else
  begin
    AreaA := TUIArea.Create(Self,NewRectXY(1,1,Position-Inset,Dimensions.y2));
    AreaB := TUIArea.Create(Self,NewRectXY(Position+Inset,1,Dimensions.x2,Dimensions.y2));
  end;
end;


{ TUIStringListArea }

constructor TUIStringListArea.Create(newParent: TUIElement; setStringList: TStringArray);
begin
  inherited Create(newParent);
  StringList := setStringList;
  Position := 0;
end;

procedure TUIStringListArea.First;
begin
  Position := 0;
end;

procedure TUIStringListArea.Recent;
begin
  if StringList <> nil then
    Position := Max(StringList.Count-Dimensions.GetHeight+1,0);
end;

procedure TUIStringListArea.SetList(setStringList: TStringArray);
begin
  StringList := setStringList;
end;

{ TUIInputLine }

constructor TUIInputLine.Create(newParent: TUIElement; newLength: Byte);
begin
  inherited Create(newParent);
  Len       := newLength;
  Value     := '';
  Position  := 0;
  EditChars := [32..125]-[Ord('@')];
end;

function TUIInputLine.Run(BreakKeys: TKeyFilter): Byte;
var Key : byte;
begin
  Output.ShowCursor;
  Update;
  repeat
    Key := Input.GetKey;
    case Key of
      VKEY_End   : Position := Length(Value);
      VKEY_Home  : Position := 0;
      VKEY_Left  : if Position > 0 then Dec(Position);
      VKEY_Right : if Position < Length(Value) then Inc(Position);
      VKEY_BackSpace : if (Length(Value) > 0) and (Position > 0) then
                       begin
                         Delete(Value,Position,1);
                         Dec(Position);
                       end;
      VKEY_Delete : if (Length(Value) > 0) and (Position < Length(Value)) then
                         Delete(Value,Position+1,1);
      else if key in EditChars then
        if Length(Value) < len then
        begin
          insert(chr(key),Value,Position+1);
          Inc(Position);
        end;
    end;
    Update;
  until Key in BreakKeys;
  Exit(Key);
end;

function TUIInputLine.Return: AnsiString;
begin
  Exit(Value);
end;

procedure TUIInputLine.SetLine(const newValue: ShortString);
begin
  Value := newValue;
  Position := Length(newValue);
end;

procedure TUIInputLine.Update;
begin
end;

{ TUIViewer }

constructor TUIViewer.Create(newParent: TUIElement; newTitle: Ansistring);
begin
  inherited Create(newParent);
  Title := newTitle;
  ExitKeys := [VKEY_ENTER,VKEY_ESCAPE];
end;

function TUIViewer.Run : Byte;
var Key : Byte;
    Last : DWord;
begin
  Output.HideCursor;
  Last := TextLines;
  Update;
  repeat
    Key := Input.GetKey([VKEY_HOME,VKEY_END,VKEY_UP,VKEY_DOWN,VKEY_PAGEUP,VKEY_PAGEDOWN]+ExitKeys);
    case Key of
      VKEY_END      : StringArea.Recent;
      VKEY_HOME     : StringArea.First;
      VKEY_UP       : if StringArea.Position > 0    then StringArea.Position := StringArea.Position-1;
      VKEY_DOWN     : if StringArea.Position < Last then StringArea.Position := StringArea.Position+1;
      VKEY_PAGEUP   : if StringArea.Position > 0    then StringArea.Position := Max(0   ,StringArea.Position-(Dimensions.GetHeight-6));
      VKEY_PAGEDOWN : if StringArea.Position < Last then StringArea.Position := Min(Last,LongInt(StringArea.Position)+(Dimensions.GetHeight-6));
    end;
    Update;
  until Key in ExitKeys;
  Exit(Key);
end;

procedure TUIViewer.SetTitle(newTitle: AnsiString);
begin
  Title := newTitle;
end;

procedure TUIViewer.SetText(StringList: TStringArray);
begin
  StringArea.SetList(StringList);
end;

procedure TUIViewer.SetExitKeys(aExitKeys: TKeyFilter);
begin
  ExitKeys := aExitKeys;
end;

function TUIViewer.GetText: TStringArray;
begin
  Exit(StringArea.StringList);
end;

procedure TUIViewer.Recent;
begin
  StringArea.Recent;
end;

procedure TUIViewer.First;
begin
  StringArea.First;
end;

procedure TUIViewer.Update;
begin
end;

function TUIViewer.TextLines: DWord;
begin
  if StringArea.StringList = nil then Exit(0);
  TextLines := Max(StringArea.StringList.Count-StringArea.Dimensions.GetHeight+1,0);
end;

{ TTextMenu }

constructor TGenericTextUIMenu.Create(newParent : TUIElement; TerminationKeys : TKeyFilter = [VKEY_ENTER]; cback : TTextUIMenuCallback = nil; BrowseInactive : Boolean = False; initial : byte = 1);
begin
  inherited Create(newParent,NewRectXY(1,1,1,1));
  TermInactive := False;
  FirstActive := 0;
  LastActive  := 0;
  Scroll      := 0;
  ScrollSize  := 0;
  Count := 0;
  Size := 20;
  Choice := initial;
  Callback := cback;
  BInactive := BrowseInactive;
  TermKeys := TerminationKeys;
  SetLength(Fields,Size);
end;

procedure TGenericTextUIMenu.Add(name: Ansistring; value : _TValue; active: boolean; color : byte);
begin
  Inc(Count);
  Inc(Dimensions.y2);
  ResizeValues(Count);
  Fields[Count].name := name;
  Fields[Count].active := active;
  Fields[Count].value := value;
  if color <> 0
    then Fields[Count].color := color
    else Fields[Count].color := Style^.MainColor;

  if Active then LastActive := Count;
  if FirstActive = 0 then if Active then FirstActive := Count;
  if Output.Length(name) > Dimensions.w then Dimensions.SetWidth(Output.Length(Name));
end;

function TGenericTextUIMenu.Run: byte;
var Key  : byte;
    Keys    : TKeyFilter;
    Letters : TKeyFilter;
begin
  Output.HideCursor;
  while (not Fields[Choice].Active) and (Choice < Count) do Inc(Choice);
  if not Fields[Choice].active then Exit(0);
  if Assigned(Callback) then Callback(Fields[Choice].Value);
  if Style^.MenuType = mtLetter
    then begin Keys := TermKeys; Choice := 0; end
    else Keys := [VKEY_UP, VKEY_DOWN]+TermKeys;
  Update;
  Letters := [];
  if Style^.MenuType <> mtChoice then
    for Key := 1 to Count do
    if Fields[Key].Active then
    begin
      Include(Keys,Ord(LetterList[Key]));
      Include(Letters,Ord(LetterList[Key]));
    end;
  repeat
    Key := Input.GetKey(Keys);

    if BInactive then
    case Key of
      VKEY_UP     : if Choice = 1     then Choice := Count else Dec(Choice);
      VKEY_DOWN   : if Choice = Count then Choice := 1     else Inc(Choice);
    end else
    case Key of
      VKEY_UP     : if Choice = FirstActive
                      then Choice := LastActive
                      else repeat Dec(Choice) until Fields[Choice].Active;
      VKEY_DOWN   : if Choice = LastActive
                      then Choice := FirstActive
                      else repeat Inc(Choice) until Fields[Choice].Active;
    end;
    if (Key in Letters) then Choice := Pos(Chr(Key),LetterList);
    if Assigned(Callback) then Callback(Choice);
    Update;
    if TermInactive and (Key in TermKeys) then Break;
  until ((Key in TermKeys) and (Fields[Choice].Active or (Style^.menuType = mtLetter))) or (Key in Letters);
  Exit(Key);
end;

procedure TGenericTextUIMenu.SetScroll(ScrollValue: Byte);
begin
  ScrollSize := ScrollValue;
end;

function TGenericTextUIMenu.Return: _TValue;
begin
  Exit(Fields[Choice].Value);
end;

procedure TGenericTextUIMenu.SetActive(Index: DWord; newActive: Boolean);
begin
  if Index > Count then Exit;
  Fields[Index].active:= newActive;
end;

procedure TGenericTextUIMenu.Update;
begin
end;

procedure TGenericTextUIMenu.Draw;
var AD : TRect;
    cn : DWord;
    clr : Byte;
    text  : string;
begin
  AD := getAbsoluteDimensions;
  //Output.ClearRectColor(AD,Style^.BackColor);
  for cn := 1 to Count do
    with Fields[cn] do
    begin
      clr := Style^.MainColor;
      text  := name;
      if Style^.MenuType <> mtChoice then
        text := Style^.LetterChoicePre + LetterList[cn] + Style^.LetterChoicePost + text;
      if (not Active) then if cn = Choice then Clr := Style^.WrongColor
                                          else Clr := Style^.InactiveColor
                      else if cn = Choice then Clr := Style^.BoldColor
                                          else Clr := Color;
      Output.DrawString(AD.x1,AD.y1+LongInt(cn-1),Clr,text)
    end;
  if Assigned(Callback) then Callback(Fields[Choice].Value);
end;

procedure TGenericTextUIMenu.ResizeValues(Value : DWord);
begin
  if Value < Size then Exit;
  Size := Size*2;
  SetLength(Fields,Size);
end;

{ TUIMenu }

constructor TUIMenu.Create(newParent: TUIElement; TerminationKeys: TKeyFilter;
  cback: TByteTextUIMenuCallback; BrowseInactive: Boolean; initial: byte);
begin
  inherited Create(newParent,TerminationKeys, cback, BrowseInactive, initial );
  EscapeKey := VKEY_ESCAPE;
  EnterKey := VKEY_ENTER;
end;

procedure TUIMenu.Add(name: Ansistring; active: boolean; color : byte);
begin
  inherited Add(name,Count+1,active,color);
end;

function TUIMenu.Run : byte;
begin
  Result := inherited Run;
  if Result = EscapeKey then Exit(0);
  if (Result = EnterKey) and (Style^.menuType = mtLetter) then Exit(0);
  Result := Return;
end;

{ TUIMessages }

constructor TUIMessages.Create(newParent: TUIElement; nBufferSize: Word; const nOptions: TFlags32; const nMoreText: string
  );
begin
  inherited Create(newParent);
  FActive     := 0;
  FOptions    := nOptions;
  FMoreText   := nMoreText;
  FMoreLength := Output.Length(FMoreText);
  FHighlights := nil;
  FHighCount  := 0;
  FMessages   := TMessageBuffer.Create(nBufferSize,Dimensions.GetWidth);
end;

procedure TUIMessages.Add(MessageText: Ansistring);
var Temp,Rest,Line : Ansistring;
    Count          : Word;
begin
  if (UIMSG_COLORING in FOptions) and (FHighCount > 0) then
  begin
    Temp := MessageText;
    for Count := 1 to FHighCount do
      with FHighlights[Count] do
      if IsWild(MessageText,Wildcard,False) then
        MessageText := Output.ColorToVCode(Color)+Temp+'@>';
  end;

  if (UIMSG_CONTINUE in FOptions) and (FActive <> 0) then
    begin MessageText := Get(0) + ' ' + MessageText; KillLast; end;

  if (Length(MessageText) >= Dimensions.GetWidth) and (FActive = Dimensions.GetHeight-1) and (UIMSG_MORE in FOptions) then
  begin
    Split(MessageText,Line,Rest,' ',Dimensions.GetWidth-(FMoreLength+1));
    Inc(FActive);
    FMessages.Add(Line);
    Draw;
    More;
    Update;
    MessageText := Rest;
  end;
  Inc(FActive,FMessages.Add(MessageText));
end;

procedure TUIMessages.ClearAll;
begin
  FActive := 0;
  FMessages.Clear;
end;

procedure TUIMessages.Update;
begin
  FActive := 0;
end;

function TUIMessages.Get(Index: Word): Ansistring;
begin
  Exit(FMessages.Get(Index));
end;

procedure TUIMessages.KillLast;
begin
  FMessages.KillLast;
  if FActive > 0 then Dec(FActive);
end;

var MTemp : TUIMessages;

procedure ColorCallbackProcedure(const nkey, nvalue : shortstring);
var color : byte;
    h     : TUIMessageHighlight;
begin
with MTemp do
begin
  color := Output.StringToColor(Copy2Symb(nkey,'_'));
  if color = 0 then CritError('Message highlight code "'+nkey+'" in ini file is not valid!');
  Inc(FHighCount);
  h.color    := color;
  h.Wildcard := nvalue;
  FHighlights[FHighCount] := h;
end;
end;


procedure TUIMessages.LoadColors(INIFile : TINI);
begin
  MTemp := Self;
  FHighlights := TUIHighlightsArray.Create(10);
  INIFile.QuerySection(@ColorCallbackProcedure);
  Include(FOptions,UIMSG_COLORING);
end;

destructor TUIMessages.Destroy;
begin
  FreeAndNil(FHighlights);
  FreeAndNil(FMessages);
  inherited Destroy;
end;

function TUIMessages.Size: DWord;
begin
  Exit(FMessages.Size);
end;

procedure TUIMessages.AddHighlight(Key, Value: Variant);
var h : TUIMessageHighlight;
begin
  if FHighlights = nil then
  begin
    FHighlights := TUIHighlightsArray.Create(30);
    Include(FOptions,UIMSG_COLORING);
  end;
  Inc(FHighCount);
  h.color    := Value;
  h.Wildcard := Key;
  FHighlights[FHighCount] := h;
end;

{ TUIMap }

constructor TUIMap.Create(newParent: TUIElement; newMap: IMap; newOptions: TFlags32);
var x,y : Word;
begin
  inherited Create(newParent);
  FADim := getAbsoluteDimensions;
  FOptions := newOptions;
  SetMap(newMap);
  if UIMAP_MARKING in FOptions then
  begin
    SetLength(MarkMap,FADim.GetWidth);
    for x := 0 to FADim.GetWidth-1 do
    begin
      SetLength(MarkMap[x],FADim.GetHeight);
      for y := 0 to FADim.GetHeight-1 do
        MarkMap[x,y] := 0;
    end;
  end;
  FX := 1;
  FY := 1;
  FShiftX := 0;
  FShiftY := 0;
end;

procedure TUIMap.setXY(X, Y: LongInt);
begin
  FADim := getAbsoluteDimensions;
  FX := X;
  FY := Y;
  FShiftX := 0;
  FShiftY := 0;
  if UIMAP_CENTERING in FOptions then
  begin
    FShiftX := FX - FADim.GetWidth div 2;
    FShiftY := FY - FADim.GetHeight div 2;
  end;
end;

procedure TUIMap.SetMap(newMap: IMap);
begin
  FMap := newMap;
end;

procedure TUIMap.Mark(X, Y: LongInt; sgn: char; atr: byte);
begin
  X := ScreenX(X)-FADim.x1;
  Y := ScreenY(Y)-FADim.y1;
  if (X < 0) or (Y < 0) or (X >= FADim.GetWidth) or (Y >= FADim.GetHeight) then Exit;
  MarkMap[x,y] := Ord(sgn) + 256*atr;
end;

procedure TUIMap.Update;
var x,y : Word;
begin
  if not (UIMAP_MARKING in FOptions) then Exit;
  SetLength(MarkMap,Dimensions.GetWidth);
  for x := 0 to FADim.GetWidth-1 do
  begin
    SetLength(MarkMap[x],FADim.GetHeight);
    for y := 0 to FADim.GetHeight-1 do
      MarkMap[x,y] := 0;
  end;
end;

function TUIMap.ScreenX(nWorldX: LongInt): LongInt;
begin
  ScreenX := nWorldX-FShiftX+FADim.x1-1;
end;

function TUIMap.ScreenY(nWorldY: LongInt): LongInt;
begin
  ScreenY := nWorldY-FShiftY+FADim.y1-1;
end;

function TUIMap.WorldX(nScreenX: LongInt): LongInt;
begin
  WorldX := FShiftX+nScreenX-FADim.x1+1
end;

function TUIMap.WorldY(nScreenY: LongInt): LongInt;
begin
  WorldY := FShiftY+nScreenY-FADim.y1+1
end;

{ TUIFullViewer }

constructor TUIFullViewer.Create(newParent: TUIElement; newTitle: AnsiString);
begin
  inherited Create(newParent,newTitle);
  Owner      := False;
  StringList := nil;
end;

procedure TUIFullViewer.Load(const FName: string);
var RS : shortstring;
    F  : Text;
begin
  FreeAndNil(StringList);
  StringList := TStringArray.Create(25);
  Owner := True;
  Assign(F,FName);
  Reset(F);
  repeat
    Readln(F,RS);
    StringList.Push(RS);
  until EOF(F);
  Close(F);

  SetText(StringList);
end;

procedure TUIFullViewer.Load(stream: TStream);
var Count  : DWord;
    Amount : DWord;
    VS     : String;
begin
  Amount := Stream.ReadDWord;

  FreeAndNil(StringList);
  StringList := TStringArray.Create(Amount+1);
  Owner := True;

  for Count := 1 to Amount do
  begin
    VS := Stream.ReadAnsiString;
    StringList.Push(VS);
  end;

  SetText(StringList);
end;

procedure TUIFullViewer.SetArray(stringArray: TStringArray);
begin
  StringList := StringArray;
  Owner := False;

  SetText(StringList);
end;

destructor TUIFullViewer.Destroy;
begin
  if Owner then
    FreeAndNil(StringList);
  inherited Destroy;
end;

end.

