{$INCLUDE valkyrie.inc}
// @abstract(ASCII Output system for Valkyrie Output)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 7, 2004)
// @lastmod(Jan 14, 2006)
//
// Implements an basic output system for Valkyrie, based
// on FP Video. System aims to be platform independent,
// tested on FreeBSD, Linux, Win32 and GO32V2 targets.
//
// @preformatted(
// Additional Features:
//   -- Screen dump capture -- currently without file
//      saving.
//   -- GenRogue Movie Format support -- recording, playing
//      and saving.
// )
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit vtoutput;
interface
uses vnode, vutil, vds, voutput;

// Specialization for GMFBuffer recording
type TGMFBuffer = specialize TArray<TScreenDump>;

// The basic output class, published as the singleton @link(Output).
// Should be initialized and disposed via TSystems.
type

{ TTextModeOutput }

TTextModeOutput = class(TOutput)
       // Initializes the Video system.
       constructor Create( Cols : Word = 80; Rows : Word = 25; const Flags : TFlags = [] ); reintroduce;
       // Updates (redraws) the screen.
       procedure Update; override;
       // Hides the cursor.
       procedure HideCursor; override;
       // Shows the cursor.
       procedure ShowCursor; override;
       // Moves the cursor to thr desired position.
       procedure MoveCursor(x,y : byte); override;
       // Puts character char to screen coordinate x,y, with color atr.
       procedure DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0); override;
       // Draws a @link(TPicture).
       procedure DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0); override;
       // Clears the whole screen memory.
       procedure Clear; override;
       // Clears a rectangle of screen memory.
       procedure ClearRect(x1,y1,x2,y2 : byte); override;
       // Clears a rectangle of screen memory with given color.
       procedure ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0);override;
       // Creates a screenshot file at the given name
       function ScreenShot(const FileName : string; stype : byte = 0) : boolean; override;
       // Returns the extension for the screenshots files.
       function ScreenShotExt(stype : byte = 0) : string; override;
       // Catches a binary uncompressed Screen Dump
       procedure CatchScreenDump(ptr : TScreenDump);
       // Outputs a binary uncompressed Screen Dump
       procedure DrawScreenDump(dump : TScreenDump);
       // Clear the GMF (GenRogue Movie File) movie data.
       procedure GMFClear;
       // Start recording a gmf movie.
       procedure GMFRecordStart;
       // Stop recording a gmf movie.
       procedure GMFRecordStop;
       // Save a gmf movie.
       procedure GMFSave(fname : string);
       // Save a gmf frame to GMFBuffer.
       procedure GMFSaveFrame;
       // Deinitializes the Video system.
       destructor Destroy; override;
       // Sets the application title
       procedure SetTitle(Long,Short : Ansistring); override;
       private
       GMFBuffer : TGMFBuffer;
       GMFRecord : Boolean;
       GMFFrame  : DWord;
       WasUpdateCalled: boolean;
     end;
     
const VTMO_HIDECURSOR     = 1;
      VTMO_FORCECLEAR     = 2;
      VTMO_USECURRENTMODE = 3;


implementation

uses {$IFDEF WIN32}Windows, {$ENDIF}
     {$IFDEF UNIX} Unix,    {$ENDIF}
     Video, SysUtils, zstream;

var TargetVideoMode : TVideoMode = (
         Col   : 80;
         Row   : 25;
         Color : True;
         );
         
const ClearCell : Word = Ord(' ')+(LightGray shl 8);

constructor TTextModeOutput.Create(Cols : Word = 80; Rows : Word = 25; const Flags : TFlags = []);
begin
  inherited Create;
  InitVideo;
  ClearScreen;
  
  if VTMO_USECURRENTMODE in Flags then
    GetVideoMode(TargetVideoMode)
  else
  begin
    TargetVideoMode.Col := Cols;
    TargetVideoMode.Row := Rows;
  end;
  
  Log('Setting ASCII video to @1x@2 color.',[Cols,Rows]);
  if not SetVideoMode(TargetVideoMode)
     then Log('Failed to set ASCII @1x@2 color textmode.',[Cols,Rows])
     else Log('ASCII Video @1x@2 color set.',[Cols,Rows]);
  ScreenSizeX := ScreenWidth;
  ScreenSizeY := ScreenHeight;
  ScreenSize  := ScreenSizeX*ScreenSizeY;
  if VTMO_HIDECURSOR  in Flags then SetCursorType(crHidden);
  if VTMO_FORCECLEAR  in Flags then begin Clear; Update; end;
  GMFBuffer := nil;
  GMFFrame  := 1;
  GMFRecord := False;
  WasUpdateCalled := False;
end;

procedure TTextModeOutput.Update;
begin
  if UI <> nil then UI.Draw;
  UpdateScreen(not WasUpdateCalled);
  WasUpdateCalled := true;
  if GMFRecord then GMFSaveFrame;
end;

procedure TTextModeOutput.HideCursor;
begin
  SetCursorType(crHidden);
end;

procedure TTextModeOutput.ShowCursor;
begin
  SetCursorType(crUnderLine);
end;

procedure TTextModeOutput.MoveCursor(x,y : byte);
begin
  SetCursorPos(x-1,y-1);
end;

procedure TTextModeOutput.DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0);
begin
  VideoBuf^[(x-1)+(y-1)*ScreenSizeX] := Ord(chr)+(atr shl 8);
end;

procedure TTextModeOutput.DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0);
begin
  VideoBuf^[(x-1)+(y-1)*ScreenSizeX] := pic;
end;

procedure TTextModeOutput.Clear;
begin
  FillWord(VideoBuf^,ScreenSize,ClearCell);
end;

procedure TTextModeOutput.ClearRect(x1,y1,x2,y2 : byte);
var x,y : byte;
begin
  for y := y1 to y2 do
    for x := x1 to x2 do
      VideoBuf^[(x-1)+(y-1)*ScreenSizeX] := ClearCell;
end;

procedure TTextModeOutput.ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0);
var x,y : byte;
    CC  : Word;
begin
  CC := Ord(' ')+((atr*16) shl 8);
  for y := y1 to y2 do
    for x := x1 to x2 do
      VideoBuf^[(x-1)+(y-1)*ScreenSizeX] := CC;
end;

function TTextModeOutput.ScreenShot(const FileName : string; stype : byte = 0) : Boolean;
var T   : Text;
    x,y : Word;
    lc,c : Byte;
    ch  : Char;
    function Color2BB(Color : Byte) : string;
    begin
      Color := Color mod 16;
      case Color of
        Black        : Exit('#333');   DarkGray     : Exit('gray');
        Blue         : Exit('navy');   LightBlue    : Exit('blue');
        Green        : Exit('green');  LightGreen   : Exit('lime');
        Cyan         : Exit('teal');   LightCyan    : Exit('aqua');
        Red          : Exit('maroon'); LightRed     : Exit('red');
        Magenta      : Exit('purple'); LightMagenta : Exit('fuchsia');
        Brown        : Exit('olive');  Yellow       : Exit('yellow');
        LightGray    : Exit('silver'); White        : Exit('white');
      end;
    end;
begin
  {$PUSH}
  {$I-}
  Assign(T,FileName);
  Rewrite(T);
  lc := 0;
  if IOResult <> 0 then Exit;
  if stype = 1 then
  begin
    Write(T,'[hr][tt][color=#BBB]');
    for y := 1 to ScreenSizeY do
    begin
      for x := 1 to ScreenSizeX do
      begin
        c := (VideoBuf^[(x-1)+(y-1)*ScreenSizeX] div 256) mod 16;
        ch := Chr(VideoBuf^[(x-1)+(y-1)*ScreenSizeX] mod 256);
        if ch = '' then
          ch:='|';
        if ch = '' then
          ch:='-';
        if ch in ['','','',''] then
          ch:='+';
        if (ch = ' ') or (ch = #0) then
        begin
          Write(T,' ');
          Continue;
        end;
        if lc <> c then
        begin
//          if lc <> 0 then Write(T,'[/color]');
          Write(T,'[/color][color='+Color2BB(c)+']');
          lc := c;
        end;
        Write(T,ch);
      end;
      Writeln(T);
    end;
    Writeln(T,'[/color][/tt][hr]');
  end
  else
  begin
    for y := 1 to ScreenSizeY do
    begin
      for x := 1 to ScreenSizeX do
        Write(T,Chr(VideoBuf^[(x-1)+(y-1)*ScreenSizeX] mod 256));
      Writeln(T);
    end;
  end;
  Close(T);
  {$POP} {restore $I}
  Exit(True);
end;

function TTextModeOutput.ScreenShotExt(stype : byte = 0) : string;
begin
  Exit('txt');
end;

procedure TTextModeOutput.CatchScreenDump(ptr : TScreenDump);
begin
  System.Move(VideoBuf^,ptr^,SizeOf(Word)*ScreenSize);
end;

procedure TTextModeOutput.DrawScreenDump(dump : TScreenDump);
begin
  System.Move(dump^,VideoBuf^,SizeOf(Word)*ScreenSize);
end;


destructor TTextModeOutput.Destroy;
begin
  GMFClear;
  DoneVideo;
  {$IFDEF UNIX}Shell('reset');{$ENDIF}
  inherited Destroy;
end;

procedure TTextModeOutput.SetTitle(Long, Short: Ansistring);
begin
  {$IFDEF WIN32}
  SetConsoleTitle(PChar(Long));
  {$ENDIF}
end;

// Start recording a gmf movie.
procedure TTextModeOutput.GMFRecordStart;
begin
  if GMFRecord then Exit;
  if Assigned(GMFBuffer) then GMFClear;
  GMFRecord := True;
  GMFBuffer := TGMFBuffer.Create(100);
end;

// Stop recording a gmf movie.
procedure TTextModeOutput.GMFRecordStop;
begin
  if not GMFRecord then Exit;
  GMFRecord := False;
end;

// Save a gmf movie.
procedure TTextModeOutput.GMFSave(fname : string);
var cnt : dword;
    OST : TGZFileStream;
begin
  if GMFRecord then GMFRecordStop;
  if GMFBuffer = nil then begin LOG('Can''t save GMF, buffer = nil!'); Exit; end;
  if GMFFrame < 2 then begin LOG('Can''t save GMF, GMFFrame < 2!'); Exit; end;

  OST := TGZFileStream.Create(fname,gzOpenWrite);
  OST.WriteDWord(GMFFrame-1);
  OST.WriteDWord(ScreenSizeX);
  OST.WriteDWord(ScreenSizeY);
  for cnt := 1 to GMFBuffer.Count do
    if GMFBuffer[cnt] <> nil then
    begin
      OST.Write(GMFBuffer[cnt]^,SizeOf(Word)*ScreenSize);
    end;
  FreeAndNil(OST);
  GMFClear;
end;

procedure TTextModeOutput.GMFSaveFrame;
var buf   : TScreenDump;
begin
  if not GMFRecord then Exit;
  if GMFBuffer = nil then Exit;
  buf := nil;
  GetMem(buf,SizeOf(Word)*ScreenSize);
  System.Move(VideoBuf^,buf^,SizeOf(Word)*ScreenSize);
  GMFBuffer[GMFFrame] := buf;
  Inc(GMFFrame);
end;

// Clear the GMF movie data.
procedure TTextModeOutput.GMFClear;
var cnt : dword;
    ptr : TScreenDump;
begin
  if GMFRecord then GMFRecordStop;
  if GMFBuffer = nil then Exit;
  for cnt := 1 to GMFBuffer.MaxSize do
    if GMFBuffer[cnt] <> nil then
    begin
      ptr := GMFBuffer[cnt];
      FreeMem(ptr,SizeOf(Word)*ScreenSize);
      GMFBuffer[cnt] := nil;
    end;
  FreeAndNil(GMFBuffer);
  GMFFrame  := 1;
  GMFBuffer := nil;
end;


end.

{ $Log:
2008-APR-26 Patches by Michal Bielinski
2006-JAN-14 Updated to Valkyrie 2
2005-FEB-18 Fixed a few comments
2005-JAN-28 Changed to support other modes then 80x25
2004-NOV-10 Added support for GenRogue Movie files
2004-OCT-22 Added support for screen grabs
2004-MAY-18 Added ClearRectColor
}
