//  @abstract(FreePascal Valkyrie SDL -- System unit)
//  @author  (Kornel Kisielewicz <charon@magma-net.pl>)
// 
//  This library provides a trivial SDL Valkyrie System. In the future it will provide
//  a full wrapper for the SDL library.
//
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  ToDo@br
//   -- Init allowing different resolutions, flags and depths)@br
//   -- Full wrapper for SDL@br
//   -- Own SDL datatypes@br
//   -- OpenGL support

{$INCLUDE valkyrie.inc}
unit vsdl;
interface
uses Classes, vutil, vsystem, vnode, SDL, GL, GLU;

const
   // Resolution 640x480
   VSDL640x480    = 0;
   // Resolution 800x600
   VSDL800x600    = 1;
   // Resolution 1024x768
   VSDL1024x768   = 2;
   // Resolution 1280x960
   VSDL1280x960   = 3;
   // Resolution 1280x1024
   VSDL1280x1024  = 4;
   // OpenGL surface
   VSDLOPENGL     = 5;
   // Fullscreen mode
   VSDLFULLSCREEN = 6;
   // Double-buffer mode
   VSDLDOUBLEBUF  = 7;
   // Software surface
   VSDLSWSURFACE  = 8;
   // Hardware surface
   VSDLHWSURFACE  = 9;
   // 16 bits per pixel
   VSDL16BPP      = 10;
   // 24 bits per pixel
   VSDL24BPP      = 11;
   // 32 bits per pixel
   VSDL32BPP      = 12;


type

{ TTrueColor }

TColor = object
    SDLColor : TSDL_Color;
    procedure Init(nr,ng,nb : Byte; na : Byte = 255);
    procedure Init(nSDLColor : TSDL_Color);
    function RGB : Uint32;
    function RGBA : Uint32;
  end;

{ TMonoFont }


// Base system for Valkyrie
type

{ TValkyrieSDL }

TValkyrieSDL = class(TSystem)
    Screen : PSDL_Surface;
    Width  : Word;
    Height : Word;
    BPP    : Word;
    Flags  : DWord;
    Ratio  : Single;
    VFlags : TFlags;
    FScreen: Boolean;
    // Registers system execution.
    constructor Create(const vsdlflags : TFlags=[]);
    // Registers system execution.
    constructor Create(aWidth, aHeight, aBPP : Word; const vsdlflags : TFlags=[]);
    // Closes system execution.
    destructor Destroy; override;
    procedure ToggleFullScreen;
    procedure Update; virtual;
    private
    procedure Setup(const vsdlflags : TFlags=[]);
    procedure SetupOpenGL;
  end;
  
// System singleton
var ValkyrieSDL : TValkyrieSDL = nil;

function NewColor(r,g,b : Byte; a : Byte = 255) : TColor;
function NewColor(nSDLColor : TSDL_Color) : TColor;
function NewColor( Color16 : Byte ) : TColor;
function RWopsFromStream( Stream : TStream; Size : DWord ) : PSDL_RWops;

const VALKYRIESHUTDOWN : Boolean = False;

implementation

uses SysUtils;

const SDLColors : array[0..15] of array[0..2] of Byte = (
      ( 0,   0,   0 ),
      ( 0,   0,   160 ),
      ( 0,   160, 0 ),
      ( 0,   160, 160 ),
      ( 160, 0,   0 ),
      ( 160, 0,   160 ),
      ( 160, 160, 0 ),
      ( 200, 200, 200 ),
      ( 128, 128, 128 ),
      ( 0,   0,   255 ),
      ( 0,   255, 0 ),
      ( 0,   255, 255 ),
      ( 255, 0,   0 ),
      ( 255, 0,   255 ),
      ( 255, 255, 0 ),
      ( 255, 255, 255 )
      );

function RW_Stream_Seek( context: PSDL_RWops; offset: Integer; whence: Integer ) : Integer; cdecl;
var Stream  : TStream;
    SOffset : PtrUInt;
    SSize   : PtrUInt;
begin
  SOffset := PtrUInt(context^.mem.base);
  Stream  := TStream(context^.mem.here);
  SSize   := PtrUInt(context^.mem.stop);

  case whence of
    0 : Stream.Seek( SOffset+offset, soBeginning );
    1 : Stream.Seek( offset, soCurrent );
    2 : Stream.Seek( SOffset+SSize+offset, soCurrent );
  end;
  Exit( Stream.Position-SOffset );
end;

function RW_Stream_Read( context: PSDL_RWops; Ptr: Pointer; size: Integer; maxnum : Integer ): Integer; cdecl;
var Stream : TStream;
begin
  Stream := TStream(context^.mem.here);
  Exit( Stream.Read( Ptr^, Size * maxnum ) div Size );
end;

function RW_Stream_Write( context: PSDL_RWops; Ptr: Pointer; size: Integer; num: Integer ): Integer; cdecl;
var Stream : TStream;
begin
  Stream := TStream(context^.mem.here);
  Exit( Stream.Write( Ptr^, Size * num ) div Size );
end;

function RW_Stream_Close( context: PSDL_RWops ): Integer; cdecl;
var Stream : TStream;
begin
  if Context <> nil then
  begin
    Stream := TStream(context^.mem.here);
    FreeAndNil( Stream );
    SDL_FreeRW( context );
  end;
end;

function RWopsFromStream( Stream : TStream; Size : DWord ) : PSDL_RWops;
begin
  RWopsFromStream := SDL_AllocRW();
  if RWopsFromStream <> nil then
  begin
    RWopsFromStream^.seek := @RW_Stream_Seek;
    RWopsFromStream^.read := @RW_Stream_Read;
    RWopsFromStream^.write := @RW_Stream_Write;
    RWopsFromStream^.close := @RW_Stream_Close;
    RWopsFromStream^.mem.base := PUInt8( Stream.Position );
    RWopsFromStream^.mem.here := PUInt8( Stream );
    RWopsFromStream^.mem.stop := PUInt8( Size );
  end;
end;

function NewColor(r, g, b : Byte; a : Byte) : TColor;
begin
  NewColor.SDLColor.r := r;
  NewColor.SDLColor.g := g;
  NewColor.SDLColor.b := b;
  NewColor.SDLColor.unused := a;
end;

function NewColor(nSDLColor : TSDL_Color) : TColor;
begin
  NewColor.SDLColor := nSDLColor;
end;

function NewColor(Color16: Byte): TColor;
begin
  NewColor.SDLColor.r := SDLColors[Color16][0];
  NewColor.SDLColor.g := SDLColors[Color16][1];
  NewColor.SDLColor.b := SDLColors[Color16][2];
  NewColor.SDLColor.unused := 255;
end;

constructor TValkyrieSDL.Create(const vsdlflags : TFlags=[]);
begin
  inherited Create;
  Width := 800; Height := 600; BPP := 32; Flags := 0;
  if VSDL640x480    in vsdlflags then begin Width := 640;  Height := 480; end;
  if VSDL800x600    in vsdlflags then begin Width := 800;  Height := 600; end;
  if VSDL1024x768   in vsdlflags then begin Width := 1024; Height := 768; end;
  if VSDL1280x960   in vsdlflags then begin Width := 1280; Height := 960; end;
  if VSDL1280x1024  in vsdlflags then begin Width := 1280; Height := 1024;end;
  if VSDL16BPP      in vsdlflags then BPP := 16;
  if VSDL24BPP      in vsdlflags then BPP := 24;
  if VSDL32BPP      in vsdlflags then BPP := 32;
  Setup(vsdlflags);
end;

constructor TValkyrieSDL.Create(aWidth, aHeight, aBPP: Word; const vsdlflags: TFlags);
begin
  inherited Create;
  Width := aWidth; Height := aHeight; BPP := aBPP; Flags := 0;
  Setup(vsdlflags);
end;

destructor TValkyrieSDL.Destroy;
begin
  VALKYRIESHUTDOWN := True;
  SDL_Quit;
  inherited Destroy;
end;

procedure TValkyrieSDL.ToggleFullScreen;
begin
  if FScreen
    then Flags := Flags and (not SDL_FULLSCREEN)
    else Flags := Flags or SDL_FULLSCREEN;
  FScreen := not FScreen;
//  SDL_WM_ToggleFullScreen(Screen);
  screen := SDL_SetVideoMode(Width, Height, BPP, Flags);
end;

procedure TValkyrieSDL.Update;
begin
  if VSDLOPENGL in VFlags then SDL_GL_SwapBuffers;
end;

procedure TValkyrieSDL.Setup(const vsdlflags: TFlags);
begin
  VFlags := vsdlflags;
  if VSDLOPENGL     in vsdlflags then Flags := Flags or SDL_OPENGL;
  if VSDLFULLSCREEN in vsdlflags then Flags := Flags or SDL_FULLSCREEN;
  if VSDLDOUBLEBUF  in vsdlflags then Flags := Flags or SDL_DOUBLEBUF;
  if VSDLSWSURFACE  in vsdlflags then Flags := Flags or SDL_SWSURFACE;
  if VSDLHWSURFACE  in vsdlflags then Flags := Flags or SDL_HWSURFACE;

  FScreen := VSDLFULLSCREEN in vsdlflags;

  Log('Initializing SDL...');


  if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) then
  begin
    SDL_Quit;
    CritError('Couldn''t initialize SDL : '+SDL_GetError);
  end;

  Log('Checking mode '+IntToStr(Width)+'x'+IntToStr(Height)+'@'+IntToStr(BPP)+'bpp...');
  if BPP <> SDL_VideoModeOK( Width, Height, BPP, Flags ) then
  begin
    SDL_Quit;
    CritError( 'Mode '+IntToStr(Width)+'x'+IntToStr(Height)+'@'+IntToStr(BPP)+'bpp not available!' );
  end;

  SDL_WM_SetCaption('Valkyrie SDL Application','VSDL Application');

  if VSDLOPENGL     in vsdlflags then
  begin
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 8 );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
  end;

  Log('Mode found...');
  screen := SDL_SetVideoMode(Width, Height, BPP, Flags);
  if ( screen = nil ) then
  begin
    Writeln('Couldn''t set '+IntToStr(Width)+'x'+IntToStr(Height)+'@'+IntToStr(BPP)+'bpp video mode : ',SDL_GetError);
    SDL_Quit;
    exit;
  end;
  Log('Mode set.');
  if VSDLOPENGL in vsdlflags then SetupOpenGL;

  Log('Finished Init');
end;

procedure TValkyrieSDL.SetupOpenGL;
begin
  ratio := width / height;
  glShadeModel(GL_SMOOTH);
  glClearColor(0.0, 0.0, 0.0, 0.0);
  glClearDepth(1.0);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
  glEnable( GL_CULL_FACE );
  glEnable( GL_DEPTH_TEST );
  glDepthFunc( GL_LEQUAL );
  glCullFace( GL_BACK );
  glFrontFace( GL_CCW );
  glClearColor( 0, 0, 0, 0 );
  glViewport( 0, 0, width-1, height-1 );
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity;
  gluPerspective( 60.0, ratio, 1.0, 1024.0 );
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity;
end;

{ TColor }

procedure TColor.Init(nr, ng, nb : Byte; na : Byte = 255);
begin
  SDLColor.r := nr;
  SDLColor.g := ng;
  SDLColor.b := nb;
  SDLColor.unused := na;
end;

procedure TColor.Init(nSDLColor : TSDL_Color);
begin
  SDLColor := nSDLColor;
end;

function TColor.RGB : Uint32;
begin
  with SDLColor do
  Exit(SDL_MapRGB(SDL_GetVideoSurface^.format,r,g,b));
end;

function TColor.RGBA : Uint32;
begin
  with SDLColor do
  Exit(SDL_MapRGBA(SDL_GetVideoSurface^.format,r,g,b,unused));
end;


end.

