{$INCLUDE valkyrie.inc}
// @abstract(Output system interface for Valkyrie Output)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 7, 2004)
// @lastmod(Nov 28, 2006)
//
// Implements an basic output system interface for Valkyrie,
// to handle text-mode screens. System aims to be platform
// independent, tested on Linux, Win32 and GO32V2 targets.
// By itself @link(TOutput) doesn't implement any driver,
// to use standard text-mode use @link(TTextOutput).
//
// @preformatted(
// Additional Features:
//   -- Integration with Valkyrie Debug, GUI, and Systems 
//   -- Color coded strings, when ColorEnabled -- allows
//      strings to dynamically change color via the
//      @@ escape sequences.
//   -- Rectangle clearing -- used by Valkyrie GUI system
//  )
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit voutput;
interface
uses vsystem, vutil;

const
// Screen size default - columns. After @link(TOutput) initialization this
// variable should be treated as read-only.
      ScreenSizeX : Word = 80;
// Screen size default - rows. After @link(TOutput) initialization this
// variable should be treated as read-only.
      ScreenSizeY : Word = 25;
// Screen size default - number of bytes. After @link(TOutput) initialization this
// variable should be treated as read-only.
      ScreenSize : DWord = 80*25;

// Binary uncompressed screendump. Size is ScreenSize.
type TScreenDump = ^Word;

// The Picture type holds all data needed to represent something on screen.
// In case of an ASCII representation these are the ASCII char and lightning
// colors. Used by @link(TWgtMapField) to draw the output.
type TPictureRec = record
       // The ASCII character to represent on-screen
       Picture   : char;
       // The color, modified internaly for lighteffects.
       Color     : byte;
     end;
     
// A typecast to store @link(TPictureRec) in a word.
type TPicture = Word;

// Color for TOutput instances supporting true-color
type TTrueColorRec = array[0..3] of Byte;
// Color for TOutput instances supporting true-color
type TTrueColor    = DWord;

// Constant for defaults
const NOCOLOR : TTrueColor = 0;

// A interface for UI implementations
type IUIElement = interface
  procedure Draw;
end;


// The basic output class, published as the singleton @link(Output).
// Should be initialized and disposed via TSystems. @link(TOutput) by itself
// should not be used. Instead initialize the @link(Output) variable with
// an instance of @link(TOutput) that implements a driver (eg.
// @link(TTextOutput).
type TOutput = class(TSystem)
       // User interface root. Drawing of it should be handled by descendant classes.
       UI : IUIElement;
       // Initializes the Video system. Should be overriden
       constructor Create; override;
       // Updates (redraws) the screen.
       procedure Update; virtual; abstract;
       // Puts character char to screen coordinate x,y, with color atr.
       procedure DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0); virtual; abstract;
       // Draws a @link(TPicture).
       procedure DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0); virtual; abstract;
       // Draws str to screen at coordinate (x,y), with color atr, with no escape sequences.
       procedure PrintString(x,y : byte; atr : byte; const str : Ansistring; Color : TTrueColor = 0);
       // Draws str to screen at coordinate (x,y), with color atr, with
       // substitution. Substitutes @1..@9 for the corresponding element in the
       // param array. The param array can hold strings, Ansistrings and integers.
       procedure DrawString(x,y : byte; atr : byte; const str : Ansistring;
           const param : array of Const; Bold : Boolean = False; Color : TTrueColor = 0);
       // Draws str to screen at coordinate (x,y), with color atr.
       procedure DrawString(x,y : byte; atr : byte; const str : Ansistring;
           Bold : Boolean = False; Color : TTrueColor = 0);
       // Draws str to screen at coordinate (x,y), with color atr, in lines of MaxLen, spliting words.
       // Returns the number of lines written.
       function DrawString(x,y : byte; atr : byte; const str : Ansistring;
           MaxLen : Word; Color : TTrueColor = 0) : byte;
       // Draws str to screen at coordinate (x,y), with color atr, right aligned.
       procedure LeftDrawString(x,y : byte; atr : byte; const str : Ansistring;
           Color : TTrueColor = 0);
       // Draws str to screen. Coordinate (x,y) will be the center character.
       procedure CenterDrawString(x,y : byte; atr : byte; const str : Ansistring;
           Color : TTrueColor = 0);
       // Draws str to screen at coordinate (x,y), with color atr, right aligned,
       // with parameters.
       procedure LeftDrawString(x,y : byte; atr : byte; const str : Ansistring;
           const param : array of Const; Color : TTrueColor = 0);
       // Draws str to screen. Coordinate (x,y) will be the center character,
       // with parameters.
       procedure CenterDrawString(x,y : byte; atr : byte; const str : Ansistring;
           const param : array of Const; Color : TTrueColor = 0);
       // Returns string length. Takes into account color escape
       // sequences.
       function Length(const str : Ansistring; upto : word = 0) : Word;
       // Returns string length. Takes into account color escape
       // sequences and Params.
       function Length(const str : Ansistring; const param : array of Const) : Word;
       // Clears the whole screen memory.
       procedure Clear; virtual; abstract;
       // Clears a rectangle of screen memory.
       procedure ClearRect(x1,y1,x2,y2 : byte); virtual; abstract;
       // Clears a rectangle of screen memory.
       procedure ClearRect(Rect : TRect); virtual;
       // Clears a rectangle of screen memory with given color.
       procedure ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0); virtual; abstract;
       // Clears a rectangle of screen memory with given color.
       procedure ClearRectColor(Rect : TRect; atr : byte; Color : TTrueColor = 0); virtual;
       // Deinitializes the Video system.
       destructor Destroy; override;
       // Hides the cursor.
       procedure HideCursor; virtual;
       // Shows the cursor.
       procedure ShowCursor; virtual;
       // Moves the cursor to thr desired position.
       procedure MoveCursor(x,y : byte); virtual;
       // Strips escape sequences from a string.
       function Strip(const str : Ansistring) : Ansistring;
       // Converts color name to color number.
       function StringToColor(str : Ansistring) : Byte;
       // Creates a screenshot file at the given name
       function ScreenShot(const FileName : string; stype : byte = 0) : Boolean; virtual; abstract;
       // Returns the extension for the screenshots files.
       function ScreenShotExt(stype : byte = 0) : string; virtual; abstract;
       // Sets application title
       procedure SetTitle(Long,Short : Ansistring); virtual;
       // Toggles fullscreen (if supported)
       procedure ToggleFullScreen; virtual;
       // Converts color id to color VCode
       function ColorToVCode(Color : byte) : string;
       function BoldColor(color : byte) : byte;
       public
       LastCodedColor : Byte;
       LastCodedBold  : Boolean;
     end;

// The @link(TOutput) singleton.
const Output : TOutput = nil;

// Color constants for independence from Crt
const Black        = 0;    DarkGray     = 8;
      Blue         = 1;    LightBlue    = 9;
      Green        = 2;    LightGreen   = 10;
      Cyan         = 3;    LightCyan    = 11;
      Red          = 4;    LightRed     = 12;
      Magenta      = 5;    LightMagenta = 13;
      Brown        = 6;    Yellow       = 14;
      LightGray    = 7;    White        = 15;
      
const ColorNames : array[0..15] of PChar =
      ('BLACK',     'BLUE',     'GREEN',    'CYAN',        'RED',
       'MAGENTA',   'BROWN',    'LIGHTGRAY','DARKGRAY',    'LIGHTBLUE',
       'LIGHTGREEN','LIGHTCYAN','LIGHTRED', 'LIGHTMAGENTA','YELLOW',
       'WHITE');
const ColorCodes : array[0..15] of Char =
      ('D','b','g','c','r','v','n','l','d','B','G','C','R','V','y','L');

implementation

uses sysutils;

constructor TOutput.Create;
begin
  inherited Create;
  if UpCase(Self.ClassName) = 'TOUTPUT' then CritError('Plain TOutput system initialized!');
  UI := nil;
end;

function TOutput.Length(const str : Ansistring; upto : word = 0) : Word;
var Max, Len, Position : DWord;
begin
  Max := system.Length(str);
  Len := 0;
  Position := 0;
  if Max < 2 then Exit(Max);
  while Position < Max do
  begin
    Inc(Position);
    if (str[Position] = '@') then
      if Position < Max then
        if (str[Position+1] <> '@') and (str[Position+1] <> '_') then begin Inc(Position); Continue end else Inc(Position);
    Inc(Len);
    if (upto <> 0) and (len >= upto) then Exit(Position)
  end;
  if upto <> 0 then Exit(Position);
  Exit(Len);
end;

function TOutput.Length(const str : Ansistring; const param : array of Const) : Word;
begin
  Exit(Length(VFormat(str,param)));
end;

procedure TOutput.ClearRect(Rect: TRect);
begin
  ClearRect(Rect.x1,Rect.y1,Rect.x2,Rect.y2);
end;

procedure TOutput.ClearRectColor(Rect: TRect; atr : byte; Color : TTrueColor = 0);
begin
  ClearRectColor(Rect.x1,Rect.y1,Rect.x2,Rect.y2,atr,color);
end;


procedure TOutput.LeftDrawString(x,y : byte; atr : byte; const str : Ansistring; Color : TTrueColor = 0);
begin
  DrawString(x-length(str)+1,y,atr,str,False,Color)
end;

procedure TOutput.CenterDrawString(x,y : byte; atr : byte; const str : Ansistring; Color : TTrueColor = 0);
begin
  DrawString(x-(length(str) div 2)+1,y,atr,str,False,Color)
end;

procedure TOutput.LeftDrawString(x,y : byte; atr : byte; const str : Ansistring; const param : array of Const; Color : TTrueColor = 0);
var Temp : Ansistring;
begin
  Temp := VFormat(str,param);
  DrawString(x-Length(Temp)+1,y,atr,Temp,False,Color);
end;

procedure TOutput.CenterDrawString(x,y : byte; atr : byte; const str : Ansistring; const param : array of Const; Color : TTrueColor = 0);
var Temp : Ansistring;
begin
  Temp := VFormat(str,param);
  DrawString(x-(Length(Temp) div 2)+1,y,atr,Temp,False,Color)
end;

function TOutput.DrawString(x,y : byte; atr : byte; const str : Ansistring; MaxLen : Word; Color : TTrueColor = 0) : Byte;
var Line,Rest,Temp : Ansistring;
    YPos           : Word;
begin
  Line := '';
  Rest := Str;
  YPos := y;
  if Length(Rest) <= MaxLen then
  begin
    Inc(YPos);
    DrawString(x,y,atr,str,False,Color)
  end
  else
  while Rest <> '' do
  begin
    Temp := Rest;
    Split(Temp,Line,Rest,' ',Length(Temp,MaxLen) );
    DrawString(x,YPos,atr,Line,False,Color);
    Color := LastCodedColor;
    Inc(YPos);
  end;
  Exit(ypos-y);
end;

procedure TOutput.PrintString(x,y : byte; atr : byte; const str : Ansistring; Color : TTrueColor = 0);
var StrLength,PosString : dword;
begin
  StrLength := system.Length(str);
  PosString := 0;
  if StrLength > 0 then
  repeat
    Inc(PosString);
    DrawPicture(x,y,Ord(str[PosString])+(atr shl 8),Color);
    Inc(x);
  until (x = ScreenSizeX+1) or (PosString = StrLength);
end;

procedure TOutput.DrawString(x,y : byte; atr : byte; const str : Ansistring; Bold : Boolean = False; Color : TTrueColor = 0);
var PosX,PosY,MaxX,StrLength,PosString,BackColor,ForeColor : dword;
begin
  PosX := X;
  PosY := Y;
  MaxX := ScreenSizeX;
  StrLength := system.Length(str);
  if StrLength = 0 then Exit;
  PosString := 0;
  BackColor := (atr div 16) * 16;
  ForeColor := atr mod 16;
  repeat
    Inc(PosString);
    if Str[PosString] = '@' then
    begin
      Inc(PosString);
      if PosString > StrLength then Break;
      case Str[PosString] of
        '>' : begin ForeColor := atr mod 16; Continue end;
        '<' : begin ForeColor := BoldColor(atr mod 16); Continue end;
        'r' : ForeColor := Red;            'R' : ForeColor := LightRed;
        'b' : ForeColor := Blue;           'B' : ForeColor := LightBlue;
        'g' : ForeColor := Green;          'G' : ForeColor := LightGreen;
        'v' : ForeColor := Magenta;        'V' : ForeColor := LightMagenta;
        'c' : ForeColor := Cyan;           'C' : ForeColor := LightCyan;
        'l' : ForeColor := LightGray;      'L' : ForeColor := White;
        'd' : ForeColor := DarkGray;       'D' : ForeColor := Black;
    'n','N' : ForeColor := Brown;      'y','Y' : ForeColor := Yellow;
        '/' : begin PosX := X; Inc(PosY); Continue; end;
        '@' : begin DrawPicture(PosX,PosY,Ord('@')+((BackColor+ForeColor) shl 8),color); Inc(PosX); Continue; end;
        '_' : begin DrawPicture(PosX,PosY,Ord(' ')+((BackColor+ForeColor) shl 8),color); Inc(PosX); Continue; end;
        else begin Log('Unknown color key "'+str[PosString]+'" in "'+str+'".'); Continue; end;
      end;
      if Bold then ForeColor := BoldColor(ForeColor);
    end
    else
    begin
      DrawPicture(PosX,PosY,Ord(str[PosString])+((BackColor+ForeColor) shl 8));
      Inc(PosX);
    end;
  until (PosX = MaxX+1) or (PosString = StrLength);
  LastCodedColor := ForeColor;
  LastCodedBold  := Bold;
end;

procedure TOutput.DrawString(x,y : byte; atr : byte; const str : Ansistring; const param : array of Const; Bold : Boolean = False; Color : TTrueColor = 0);
begin
  DrawString(x,y,atr,VFormat(str,param),Bold,Color);
end;


function TOutput.BoldColor(color : byte) : byte;
begin
  if color <= 7 then Exit(color+8) else
  if color = 8  then Exit(7)
                else Exit(White);
end;

destructor TOutput.Destroy;
begin
//  if Assigned(UI) then FreeAndNil(UI);
  inherited Destroy;
end;

procedure TOutput.HideCursor; begin Log(ClassName+'.HideCursor unsupported!'); end;
procedure TOutput.ShowCursor; begin Log(ClassName+'.ShowCursor unsupported!'); end;
procedure TOutput.MoveCursor(x,y : byte); begin Log(ClassName+'.MoveCursor unsupported!'); end;

function TOutput.Strip(const str : Ansistring) : Ansistring;
var StrLength,PosString : DWord;
begin
  StrLength := system.Length(str);
  if StrLength = 0 then Exit('');
  PosString := 0;
  Strip := '';
  repeat
    Inc(PosString);
    if Str[PosString] = '@' then
    begin
      Inc(PosString);
      if PosString > StrLength then Exit(Strip);
      case Str[PosString] of
        '@' : Strip += '@';
      end;
    end
    else Strip += Str[PosString];
  until PosString = StrLength;
end;

function TOutput.StringToColor(str : Ansistring) : Byte;
var Count : Byte;
begin
  str := Trim(UpCase(str));
  for Count := 0 to 15 do
    if str = ColorNames[Count] then Exit(Count);
  Exit(0);
end;

procedure TOutput.SetTitle(Long, Short: Ansistring);
begin
  Log(ClassName+'.SetTitle unsupported!');
end;

procedure TOutput.ToggleFullScreen;
begin
  Log(ClassName+'.ToggleFullScreen unsupported!');
end;

function TOutput.ColorToVCode(Color : byte) : string;
begin
  if Color > 15 then Exit('') else Exit('@'+ColorCodes[Color]);
end;

end.

{
LOG:
2006-DEC-05 Added Bold option for DrawString
2006-DEC-05 Added ColorToVCode and StringToColor
2006-DEC-05 Added Strip (ToDo: make it work on Ansi)
2006-NOV-28 Added PrintString, and made ColorCoding always enabled
2006-NOV-09 Added @/ linebreak to color coded strings
2006-JAN-14 DrawString, Length, LeftDrawString and CenterDrawString with param support
2006-JAN-14 Rewrote DrawString
2006-JAN-14 AnsiString Support
2006-JAN-14 Ported to Valkyrie 2
2005-FEB-18 Fixed a few comments
2005-JAN-28 Changed to support other modes then 80x25
2004-NOV-10 Added support for GenRogue Movie files
2004-OCT-22 Added support for screen grabs
2004-MAY-18 Added ClearRectColor
}
