{$INCLUDE valkyrie.inc}
// @abstract(Node class for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 7, 2004)
// @cvs($Author: chaos-dev $)
// @cvs($Date: 2008-01-14 22:16:41 +0100 (Mon, 14 Jan 2008) $)
//
// @link(TNode) is core class from which other classes inherit.
// It implements a tree-like structure. Also, each node
// has an unique identifier, represented by @link(TUID).
//
// This unit also implements the two Valkyrie base classes :
// TVObject and TVClass.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>
//
// @preformatted(
// TODO: Check wether a Node can dispose of itself via Self.Done.
// TODO: Implement generic Save/Load via streams.
// )

unit vnode;
interface
uses vmsg, vutil, vdebug;

// The most generic of Valkyrie objects. Implements only the error
// handling functions. It is recommended that all Valkyrie classes
// inherit at least this class.
type

{ TVObject }

TVObject = class
     // TVObject Interface for @link(grdebug.CritError)
     procedure   CritError( const aCritErrorString : Ansistring ); virtual;
     // TVObject Interface for @link(grdebug.Warning).
     procedure   Warning  ( const aWarningString   : Ansistring ); virtual;
     // TVObject Interface for @link(grdebug.Log).
     procedure   Log      ( const aLogString       : Ansistring ); virtual;
     // TVObject Interface for @link(grdebug.Log).
     procedure   Log      ( Level : TLogLevel; const aLogString : Ansistring ); virtual;
     // TVObject Interface for @link(grdebug.CritError), VFormat version.
     procedure   CritError( const aCritErrorString : Ansistring; const aParam : array of Const );
     // TVObject Interface for @link(grdebug.Warning), VFormat version.
     procedure   Warning  ( const aWarningString   : Ansistring; const aParam : array of Const );
     // TVObject Interface for @link(grdebug.Log), VFormat version.
     procedure   Log      ( const aLogString       : Ansistring; const aParam : array of Const );
     // TVObject Interface for @link(grdebug.Log), VFormat version.
     procedure   Log      ( Level : TLogLevel; const aLogString       : Ansistring; const aParam : array of Const );
     // Returns wether the object has a parent -- in case of TVObject it's always false
     function hasParent : boolean; virtual;
     // Returns wether the object has a child -- in case of TVObject it's always false
     function hasChild : boolean; virtual;
     // Returns wether the object is a TVNode
     function isNode : boolean; virtual;
  end;

// The base Valkyrie class, implements data for object serialization and
// unique identification (see @link(TUIDStore)). One of the reasons for
// this serialization is to create a global Load/Save Mechanism.
type

{ TVClass }

TVClass = class(TVObject)
       // Unique IDentification number (@link(TUID))
       // Assigned by the @link(UIDs) singleton, unique.
       UID    : TUID;
       // Identification Number of this Class - may be shared among similar
       // Classes.
       ID     : TIDN;
       // TVClass Interface for @link(grdebug.CritError)
       // Calls vdebug.CritError, providing additional information on
       // the error-calling TVClass.
       procedure   CritError(const aCritErrorString : Ansistring); override; overload;
       // TVClass Interface for @link(grdebug.Warning).
       // Calls vdebug.Warning, providing additional information on
       // the warning-calling TVClass.
       procedure   Warning  (const aWarningString   : Ansistring); override; overload;
       // TVClass Interface for @link(grdebug.Log).
       // Calls vdebug.Log, providing additional information on
       // the log-calling TVClass.
       procedure   Log      (const aLogString       : Ansistring); override; overload;
       // TVClass Interface for @link(grdebug.Log).
       // Calls vdebug.Log, providing additional information on
       // the log-calling TVClass.
       procedure   Log      (Level : TLogLevel; const aLogString : Ansistring); override; overload;
     end;

// The basic node class. Implements a self-disposing tree-like structure. The
// base class of @link(TSystem), and considered a building block for the data
// structure of the program. At best, all the program nodes should be gathered
// in one tree -- that allows one-call disposal of all the allocated memory.
type TNode = class(TVClass)
       // Link to the parent node.
       Parent     : TNode;
       // Link to first child node.
       Child      : TNode;
       // Link to next node.
       Next       : TNode;
       // Link to previous node.
       Prev       : TNode;
       // Count of children nodes.
       ChildCount : DWord;
       // Standard constructor, zeroes all fields.
       constructor Create; virtual;
       // zeroes all fields.
       procedure Clean; virtual;
       // Adds theChild as a child of current node.
       // The child is added to the END of the Child list.
       procedure   Add(theChild : TNode); virtual;
       // Changes parent to Destination.
       // Error free.
       procedure   Move(Destination : TNode);
       // Basic recieveing method. Should be overriden.
       procedure   Receive(MSG : TMessage); virtual;
       // Removes self from Parent node.
       procedure   Detach;
       // Destroys all children.
       procedure DestroyChildren;
       // Standard destructor, frees @link(UID), and destroys
       // children and siblings.
       destructor  Destroy; override;
       // Returns wether the node has a parent.
       function hasParent : boolean; override;
       // Returns wether the object has a child.
       function hasChild : boolean; override;
       // Returns wether the object is a TNode
       function isNode : boolean; override;
       // Returns wether the object is a first child
       function isFirstChild : boolean; 
       // Returns wether the object is the last child
       function isLastChild : boolean;
     end;


implementation
uses vuid, sysutils;

//REM       procedure   Remove(childUID : TUID);

constructor TNode.Create;
begin
  Log(LTRACE,'Created.');
  Clean;
end;

procedure TNode.Clean;
begin
  Child      := nil;
  Parent     := nil;
  Next       := Self;
  Prev       := Self;
  ChildCount := 0;
end;

procedure TNode.Add(theChild : TNode);
begin
  if theChild.Parent <> nil then theChild.Detach;
  theChild.Parent := Self;
  if Child = nil then 
    Child := theChild
  else
  begin
    theChild.Prev := Child.Prev;
    theChild.Next := Child;
    Child.Prev.Next := theChild;
    Child.Prev      := theChild;
  end;  
  Inc(ChildCount);
end;

function TNode.hasParent : boolean;
begin
  Exit(Parent <> nil);
end;

function TNode.hasChild : boolean;
begin
  Exit(Child <> nil);
end;

function TNode.isNode : boolean; 
begin
  Exit(True);
end;

function TNode.isFirstChild : boolean; 
begin
  if Parent <> nil then 
    Exit(Parent.Child = Self) 
  else Exit(False);
end;

function TNode.isLastChild : boolean; 
begin
  if Parent <> nil then 
    if Parent.Child <> nil then 
      Exit(Parent.Child.Next = Self) 
    else Exit(False)
  else Exit(False);
end;


procedure TNode.Detach;
begin
  if Parent <> nil then
  begin
    if Parent.Child = Self then
    begin
      if Next <> Self then Parent.Child := Next
                      else Parent.Child := nil;
    end;
    Dec(Parent.ChildCount);
  end;

  Prev.Next := Next;
  Next.Prev := Prev;
  
  Prev := Self;
  Next := Self;
  
  Parent := nil;
end;

procedure TNode.DestroyChildren;
begin
  while Child <> nil do
  begin
    Child.Free;
  end;
end;


procedure TNode.Receive(MSG : TMessage);
begin
  case MSG.ID of
    0 :;
    //MSG_NODE_Destroy : begin Parent.Remove(Self); Self.Done; exit; end;
  else
    Self.Warning('Unknown message recieved (@1, ID: @2)',[Msg.ClassName,Msg.ID]);
  end;
  MSG.Free;
end;

procedure TNode.Move(Destination : TNode);
begin
  if Parent <> nil      then Detach;
  if Destination <> nil then Destination.Add(Self);
end;

destructor TNode.Destroy;
begin
  Detach;
  if UIDs <> nil then UIDs.Remove(UID);
  while Child <> nil do
  begin
    Child.Free;
  end;
  Log(LTRACE,'Destroyed.');
end;

procedure TVObject.CritError( const aCritErrorString : Ansistring);
begin
  vdebug.CritError('<'+classname+'> ' + aCritErrorString);
end;

procedure TVObject.Warning  ( const aWarningString   : Ansistring);
begin
  vdebug.Warning('<'+classname+'> '+aWarningString);
end;

procedure TVObject.Log      (const aLogString       : Ansistring);
begin
  vdebug.Log('<'+classname+'> '+aLogString);
end;

procedure TVObject.Log(Level: TLogLevel; const aLogString: Ansistring);
begin
  if Level > LogLevel then Exit;
  vdebug.Log(Level,'<'+classname+'> '+aLogString);
end;

procedure TVObject.CritError(const aCritErrorString: Ansistring; const aParam: array of const);
begin
  CritError(VFormat(aCritErrorString,aParam));
end;

procedure TVObject.Warning(const aWarningString: Ansistring; const aParam: array of const);
begin
  Warning(VFormat(aWarningString,aParam));
end;

procedure TVObject.Log(const aLogString: Ansistring; const aParam: array of const);
begin
  Log(VFormat(aLogString,aParam));
end;

procedure TVObject.Log(Level: TLogLevel; const aLogString: Ansistring; const aParam: array of const);
begin
  if Level > LogLevel then Exit;
  Log(Level, VFormat(aLogString,aParam));
end;

procedure TVClass.Log      (const aLogString       : Ansistring);
begin
  vdebug.Log('<'+classname+'/'+IntToStr(ID)+'/'+IntToStr(UID)+'> '+aLogString);
end;

procedure TVClass.Log(Level: TLogLevel; const aLogString: Ansistring);
begin
  if Level > LogLevel then Exit;
  vdebug.Log('<'+classname+'/'+IntToStr(ID)+'/'+IntToStr(UID)+'> '+aLogString);
end;

procedure TVClass.CritError(const aCritErrorString : Ansistring);
begin
  vdebug.CritError('<'+classname+'/'+IntToStr(ID)+'/'+IntToStr(UID)+'> '+aCritErrorString);
end;

procedure TVClass.Warning  (const aWarningString   : Ansistring);
begin
  vdebug.Warning('<'+classname+'/'+IntToStr(ID)+'/'+IntToStr(UID)+'> '+aWarningString);
end;

function TVObject.hasParent : boolean;
begin
  Exit(False);
end;

function TVObject.hasChild : boolean;
begin
  Exit(False);
end;

function TVObject.isNode : boolean; 
begin
  Exit(False);
end;

end.

// Modified      : $Date: 2008-01-14 22:16:41 +0100 (Mon, 14 Jan 2008) $
// Last revision : $Revision: 110 $
// Last author   : $Author: chaos-dev $
// Last commit   : $Log$
// Head URL      : $HeadURL: https://libvalkyrie.svn.sourceforge.net/svnroot/libvalkyrie/fp/src/vnode.pas $

