unit vluaconfig;
{$mode objfpc}
interface

uses vnode, vlua;

type TEntryCallback = procedure ( key, value : Variant ) of object;

type

{ TLuaConfig }

TLuaConfig = class(TVObject)
    constructor Create( const aFileName : Ansistring = '');
    procedure Load( const aFileName : Ansistring );
    procedure EntryFeed( const Table : AnsiString; const Callback : TEntryCallback );
    procedure RecEntryFeed( const Table : AnsiString; const Callback : TEntryCallback );
    function isPaused : Boolean;
    function Resume : Variant;
    destructor Destroy; override;
  protected
    function GetValue( const Key : AnsiString ) : Variant;
    function HasValue( const Key : AnsiString ) : Boolean;
    function Resolve( const Key : AnsiString ) : Boolean;
  protected
    FLua     : TLua;
    FThread  : PLua_State;
    FResult  : Variant;
  public
    property Entries[ const Key : AnsiString ] : Variant read GetValue; default;
    property Valid[ const Key : AnsiString ] : Boolean read HasValue;
    property Lua : TLua read FLua;
  end;

implementation

uses sysutils, strutils, lua;

constructor TLuaConfig.Create( const aFileName : Ansistring = '');
begin
  FLua := TLua.Create;
  if aFileName <> '' then Load( aFileName );
  FThread := nil;
  FResult := Null;
end;

procedure TLuaConfig.Load( const aFileName : Ansistring );
begin
  FLua.LoadFile( aFileName );
end;

procedure TLuaConfig.EntryFeed(const Table: AnsiString;
  const Callback: TEntryCallback);
begin
  if not Resolve( Table ) then raise ELuaException.Create('EntryFeed('+Table+') failed!');
  if not lua_istable( FLua.LuaState, -1 ) then raise ELuaException.Create('EntryFeed('+Table+') target not a table!');

  lua_pushnil( FLua.LuaState );  // first key */
  while (lua_next( FLua.LuaState, -2 ) <> 0) do
  begin
    // uses 'key' (at index -2) and 'value' (at index -1) */
    Callback( lua_tovariant( FLua.LuaState, -2 ), lua_tovariant( FLua.LuaState, -1 ) );
    lua_pop( FLua.LuaState, 1 );
  end;
  lua_pop( FLua.LuaState, 1 );
end;

procedure TLuaConfig.RecEntryFeed(const Table: AnsiString;
  const Callback: TEntryCallback);
  procedure Iterate( const KeyStart : AnsiString );
  begin
    lua_pushnil( FLua.LuaState );  // first key */
    while (lua_next( FLua.LuaState, -2 ) <> 0) do
    begin
      // uses 'key' (at index -2) and 'value' (at index -1) */
      if lua_istable( FLua.LuaState, -1 )
        then Iterate( KeyStart + lua_tovariant( FLua.LuaState, -2 ) + '.' )
        else Callback( KeyStart + lua_tovariant( FLua.LuaState, -2 ), lua_tovariant( FLua.LuaState, -1 ) );
      lua_pop( FLua.LuaState, 1 );
    end;
  end;
begin
  if not Resolve( Table ) then raise ELuaException.Create('EntryFeed('+Table+') failed!');
  if not lua_istable( FLua.LuaState, -1 ) then raise ELuaException.Create('EntryFeed('+Table+') target not a table!');

  Iterate('');
  lua_pop( FLua.LuaState, 1 );
end;

function TLuaConfig.isPaused: Boolean;
begin
  Exit( FThread <> nil );
end;

function TLuaConfig.Resume: Variant;
var Error : AnsiString;
    Res   : Integer;
begin
  if FThread = nil then Exit( false );
  lua_pushvariant( FThread, FResult );
  Res := lua_resume( FThread, 1 );
  if (Res <> 0) and (Res <> LUA_YIELD_) then
  begin
    Error := lua_tostring( FThread, -1 );
    lua_pop( FThread, 1 );
    lua_pop( FLua.LuaState, 1 );
    raise Exception.Create( Error );
  end;
  FResult := lua_tovariant( FThread, -1 );
  Resume := FResult;
  lua_pop( FThread, 1 );
  if Res <> LUA_YIELD_ then
  begin
    lua_pop( FLua.LuaState, 1 );
    FThread := nil;
  end;
end;

destructor TLuaConfig.Destroy;
begin
  FreeAndNil( FLua );
end;

function TLuaConfig.GetValue(const Key: AnsiString): Variant;
var Error : AnsiString;
    Res   : Integer;
begin
  if FThread <> nil then
  begin
    // Signal error?
    lua_pop( FLua.LuaState, 1 );
    FThread := nil;
  end;
  if not Resolve( Key ) then raise ELuaException.Create('GetValue('+Key+') failed!');
  if lua_isfunction( FLua.LuaState, -1 ) then
  begin
    FThread := lua_newthread( FLua.LuaState );
    lua_insert(FLua.LuaState, -2);
    lua_xmove(FLua.LuaState, FThread, 1);
    GetValue := Resume;
    Exit;
  end;
  GetValue := lua_tovariant( FLua.LuaState, -1 );
  lua_pop( FLua.LuaState, 1 );
end;

function TLuaConfig.HasValue(const Key: AnsiString): Boolean;
var Piece : AnsiString;
    Count : DWord;
begin
  if Resolve( Key ) then
  begin
    lua_pop( FLua.LuaState, 1 );
    Exit( True );
  end
  else
    Exit( False );
end;

function TLuaConfig.Resolve(const Key: AnsiString): Boolean;
var Piece : AnsiString;
    Count : DWord;
begin
  Count := 1;
  repeat
    Piece := ExtractDelimited( Count, Key, ['.'] );
    if Piece = '' then break;
    if Count = 1 then
      lua_getglobal( FLua.LuaState, PChar(Piece) )
    else
      if lua_istable( FLua.LuaState, -1 ) then
      begin
        lua_pushstring( FLua.LuaState, PChar(Piece) );
        lua_gettable( FLua.LuaState, -2);
        lua_insert( FLua.LuaState, -2);
        lua_pop( FLua.LuaState, 1);
      end
      else
      begin
        lua_pop( FLua.LuaState, 1 );
        Exit(False);
      end;
    Inc(Count);
  until false;
  if Count = 1 then Exit(False);
  Exit( True );
end;

end.

