{$INCLUDE valkyrie.inc}
// @abstract(Input system interface for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 19, 2004)
// @lastmod(Jan 15, 2006)
//
// Implements an basic input system interface for Valkyrie, based
// on FP Keyboard. Accessed via the @link(Input) singleton.
//
// @preformatted(
// TODO: Multiple Command set support
// TODO: Full command set handling.
// TODO: TInputLine wiget returns a value for special events (registered).
// TODO: Multiple command sets calling - GetCommand(Set) : Byte;
// TODO: CKeyBuffer shouldn't be hardcoded.
// )
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit vinput;
interface
uses sysutils, classes, vds, vsystem, vini;

// Hard coded Buffersize.
const CKeyBufferSize = 20;
      COMMAND_INVALID = 255;

type // Command set. Indexes are VKeyCodes, values are commandID's.
     TCommandSet  = array[0..255] of Byte;
     // Filter set. Used to restrict the amount of valid keys.
     TKeyFilter   = set of Byte;
     // Macro array. Holds Key VCodes.
     TKeyMacro    = array[0..CKeyBufferSize-1] of Byte;
     // Token reader for config file parsing.
     TTokenReader = function(const Token : string) : byte;

// The basic input class, published as the singleton @link(Output).
type

{ TInput }

TInput = class(TSystem)
       // Initializes the Keyboard system.
       constructor Create; override;
       // Reads a key from the keyboard and returns it's VKey value.
       function GetKey : byte;
       // Reads a key from the keyboard and returns it's Command value.
       function GetCommand : byte;
       // Reads a key from the keyboard and returns it's Command value.
       function GetCommand(const filter : TKeyFilter) : byte;
       // Repeats @link(GetKey), until key is in char set.
       function GetKey(const filter : TKeyFilter) : byte;
       // Registers an 'overdrive' key -- used for @link(TConsole).
       // Whenever the key is pressed after registering, the
       // Keyboard system calls @link(TSystem.Call). Currently only
       // one registration supported.
       procedure RegisterOverrideKey(Key : Byte; System : TSystem);
       // Feeds a key as the last key. Useful in breaking @link(RegisterKey).
       procedure FeedKey(Key : Byte);
       // Feeds a key to the buffer. Used for macros.
       procedure QueueKey(Key : Byte);
       // Returns a text representation of the given VKeyCode.
       function VCodeToString(VKeyCode : Byte) : string;
       // Returns a text representation of the given Commands VKeyCode.
       function CommandToString(Command : Byte) : string;
       // Returns the VKeyCode of the given command.
       function CommandToVCode(Command : Byte) : Byte;
       // Returns the VKeyCode of the given command.
       function VCodeToCommand(KeyCode: Byte) : byte;
       // Parses VKeyCode form a string.
       function StringToVCode(const KeyCode : string) : byte;
       // Registers a Key in the Command in the command set;
       procedure RegisterCommand(Key, Command : Byte);
       // Loads Keybindings from file
       procedure Load(var T : Text; CommandReader : TTokenReader);
       // Loads Keybindings from INI file (see vini.pas)
       procedure Load(INIFile : TINI; CommandReader : TTokenReader);
       // Returns wether a key was pressed.
       function KeyPressed : Boolean;
       // Resets all waiting keys (don't do it yourself -- or PlayBack wont work!)
       procedure ResetKeyBuffer;
       // Deinitializes the Keyboard system.
       destructor Destroy; override;
       // Starts recording keystrokes.
       procedure MacroStart;
       // Stops recording keystrokes.
       procedure MacroStop;
       // Saves the recorded macro to a TKeyMacro variable.
       function MacroSave : TKeyMacro;
       // Converts macro to string.
       function MacroToString(const theMacro : TKeyMacro) : string;
       // Converts string to macro.
       function StringToMacro(const theMacro : string) : TKeyMacro;
       // Plays the given macro.
       procedure MacroPlay(const theMacro : TKeyMacro);
       // Starts recording keys for replay.
       procedure RecorderStart;
       // Stops recording keys for replay.
       procedure RecorderStop;
       // Dumps raw keydtrokes into given stream.
       procedure RecorderDump(OSt : TStream);
       // Loads playback from given stream.
       procedure PlaybackLoad(ISt : TStream);
       // Starts playback.
       procedure PlaybackStart;
       // Stops/pauses playback.
       procedure PlaybackStop;
       // Resets recording/playback
       procedure ResetRecorder;
       // Returns wether the register callback is turned on
       function RegisterCalled : Boolean;
       // For key-Value mechanisms
       procedure RegisterCommandVariant( Key, Value : Variant );
       procedure ResetKeyRegistration();
       protected
       function PollKey : Boolean; virtual; abstract;
       function GetVCode : byte; virtual; abstract;
       public
       LastKey   : Byte;
       private
       function MacroTest(Key : byte) : byte;
       private
       RegKey    : Byte;
       RegSys    : TSystem;
       RegOn     : Boolean;
       CmdSet    : TCommandSet;
       Keys      : TByteArray;
       Macro     : TByteArray;
       KeyOutSize    : DWord;
       KeyOut        : TMemoryStream;
       KeyIn         : TMemoryStream;
       Macroing      : Boolean;
       Recording     : Boolean;
       Playback      : Boolean;
       PlaybackDelay : DWord;
       PlaybackSize  : LongInt;
       PlaybackCount : LongInt;
     end;

// The @link(TInput) singleton.
const Input : TInput = nil;

// VKey constants. VKey coding is a one byte coding for all the keys.
// It uses standard Keycodes for all the standard ASCII keyboard signs
// (that is A-Za-z0-9 and all the standard ASCII signs like
// !@#$%^&*()_+-=[]{}:";'<>,.?/~`\| that are achievable from the keyboard.
// The rest of the numbers are used as below:
const
  VKEY_NONE      = 0;
  VKEY_ESCAPE    = 1;
  VKEY_TAB       = 3;
  VKEY_BACKSPACE = 4;
  VKEY_ENTER     = 5;
  VKEY_INSERT    = 6;
  VKEY_DELETE    = 7;
  VKEY_HOME      = 8;
  VKEY_END       = 9;
  VKEY_PAGEUP    = 10;
  VKEY_PAGEDOWN  = 11;
  VKEY_UP        = 12;
  VKEY_DOWN      = 13;
  VKEY_LEFT      = 14;
  VKEY_RIGHT     = 15;
  VKEY_CENTER    = 16;
  VKEY_F1        = 128;
  VKEY_F2        = 129;
  VKEY_F3        = 130;
  VKEY_F4        = 131;
  VKEY_F5        = 132;
  VKEY_F6        = 133;
  VKEY_F7        = 134;
  VKEY_F8        = 135;
  VKEY_F9        = 136;
  VKEY_F10       = 137;
  VKEY_F11       = 138;
  VKEY_F12       = 139;
  VKEY_NUM_ENTER = 140;

const VKEY_ARROWSET = [VKEY_HOME,VKEY_END,VKEY_PAGEDOWN,VKEY_PAGEUP,
                       VKEY_LEFT,VKEY_RIGHT,VKEY_UP,VKEY_DOWN];

var QuitApplicationProc   : procedure = nil;
    QuitApplicationOOProc : procedure of object = nil;
procedure QuitApplication;

implementation

uses voutput, vutil, strutils, variants;

procedure QuitApplication;
begin
  if Assigned(QuitApplicationProc)   then QuitApplicationProc   else
  if Assigned(QuitApplicationOOProc) then QuitApplicationOOProc else
    Halt(0);
end;


constructor TInput.Create;
begin
  inherited Create;
  if UpCase(Self.ClassName) = 'TINPUT' then CritError('Plain TInput system initialized!');
  RegKey := 0;
  RegSys := nil;
  RegOn  := false;
  FillByte(CmdSet,256,0);
  Keys  := TByteArray.Create(CKeyBufferSize);
  Macro := TByteArray.Create(CKeyBufferSize);
  Macroing := False;
  KeyIn     := nil;
  KeyOut    := nil;
  KeyOutSize:= 0;
  Recording := False;
  Playback  := False;
  PlaybackSize  := 0;
  PlaybackCount := 0;
  PlaybackDelay := 50;
end;

procedure TInput.RegisterCommand(Key, Command : Byte);
begin
  if CmdSet[Key] <> 0 then Warning('Key '+VCodeToString(Key)+' rebinded!');
  CmdSet[Key] := Command;
end;

function TInput.GetCommand : byte;
begin
  Exit(CmdSet[GetKey]);
end;

// Reads a key from the keyboard and returns it's Command value.
function TInput.GetCommand(const filter : TKeyFilter) : byte;
var Command : Byte;
begin
  repeat Command := GetCommand() until Command in Filter;
  Exit(Command);
end;


// Feeds a key to the buffer. Used for macros.
procedure TInput.QueueKey(Key : Byte);
begin
  Keys.Push(Key);
end;

function TInput.GetKey : byte;
begin
  if Assigned(Output) then Output.Update;
  if (not Keys.isEmpty) then begin LastKey := Keys.Peek; Exit(MacroTest(Keys.Pop)); end;
  if Playback then
    if PlaybackCount+1 < PlaybackSize then
    begin
      Inc(PlaybackCount);
      Sleep(PlaybackDelay);
      Exit(KeyIn.readByte);
    end;
  repeat
    repeat
{      if InterruptProc <> nil then
        if InterruptTimer.Out then
          begin InterruptProc; InterruptTimer.Reset; end;}
    until PollKey;

    LastKey := GetVCode;
    if Assigned(RegSys) and (LastKey = RegKey) then
    begin
      RegOn := True;
      RegSys.Call;
      RegOn := False;
    end;
  until LastKey <> 0;
  Exit(MacroTest(LastKey));
end;

function TInput.GetKey(const filter : TKeyFilter) : Byte;
var Key : Byte;
begin
  repeat Key := Input.GetKey; until Key in filter;
  Exit(Key);
end;

procedure TInput.RegisterOverrideKey(Key : Byte; System : TSystem);
begin
  RegKey := Key;
  RegSys := System;
  Log('Key '+VCodeToString(Key)+' binded to '+System.ClassName+' system.');
end;

procedure TInput.FeedKey(Key : Byte);
begin
  LastKey := Key;
end;

procedure TInput.MacroStart;
begin
  Macro.Clear;
  Macroing := True;
end;

procedure TInput.MacroStop;
begin
  Macroing := False;
end;

procedure TInput.MacroPlay(const theMacro : TKeyMacro);
var cn : Byte;
begin
  cn := 0;
  while theMacro[cn] <> 0 do
  begin
    Keys.Push(theMacro[cn]);
    Inc(cn); if cn = CKeyBufferSize then Break;
  end;
end;

procedure TInput.RecorderStart;
begin
  if KeyOut = nil then KeyOut := TMemoryStream.Create;
  KeyOut.SetSize(1024);
  KeyOutSize := 0;
  Recording := True;
end;

procedure TInput.RecorderStop;
begin
  Recording := False;
end;

procedure TInput.RecorderDump(OSt : TStream);
begin
  OSt.WriteDWord(KeyOutSize);
  KeyOut.Position:= 0;
  OSt.CopyFrom(KeyOut,KeyOutSize);
end;

procedure TInput.PlaybackLoad(ISt : TStream);
begin
  PlaybackSize := ISt.ReadDWord;
  KeyIn := TMemoryStream.Create;
  KeyIn.SetSize(PlaybackSize);
  KeyIn.CopyFrom(ISt,PlaybackSize);
  KeyIn.Position:= 0;
end;

procedure TInput.PlaybackStart;
begin
  Playback := True;
end;

procedure TInput.PlaybackStop;
begin
  Playback := False;
end;

procedure TInput.ResetRecorder;
begin
  FreeAndNil(KeyOut);
  FreeAndNil(KeyIn);
  KeyOutSize := 0;
  Playback := False;
  Recording := False;
  PlaybackSize  := 0;
  PlaybackCount := 0;
  PlaybackDelay := 50;
end;

function TInput.RegisterCalled: Boolean;
begin
  Exit(RegOn);
end;

procedure TInput.RegisterCommandVariant(Key, Value: Variant);
var K,V : Byte;
begin
  if VarIsOrdinal(Value)
    then V := Value
    else V := COMMAND_INVALID;
  K := StringToVCode(Key);
  RegisterCommand(K,V);
end;

procedure TInput.ResetKeyRegistration();
begin
  FillByte(CmdSet,256,0);
end;

function TInput.MacroSave : TKeyMacro;
var cn : Byte;
begin
  MacroStop;
  for cn := 0 to CKeyBufferSize-1 do
    if (not Macro.isEmpty) then MacroSave[cn] := Macro.Pop else MacroSave[cn] := 0;
end;


destructor TInput.Destroy;
begin
  FreeAndNil(KeyOut);
  FreeAndNil(KeyIn);
  FreeAndNil(Keys);
  FreeAndNil(Macro);
  inherited Destroy;
end;

function TInput.VCodeToString(VKeyCode : Byte) : string;
begin
  if VKeyCode in [32..126] then Exit(Chr(VKeyCode));
  case VKeyCode of
    VKEY_NONE      : Exit('NONE');
    VKEY_ESCAPE    : Exit('ESCAPE');
    VKEY_TAB       : Exit('TAB');
    VKEY_BACKSPACE : Exit('BACKSPACE');
    VKEY_ENTER     : Exit('ENTER');
    VKEY_INSERT    : Exit('INSERT');
    VKEY_DELETE    : Exit('DELETE');
    VKEY_HOME      : Exit('HOME');
    VKEY_END       : Exit('END');
    VKEY_PAGEUP    : Exit('PAGEUP');
    VKEY_PAGEDOWN  : Exit('PAGEDOWN');
    VKEY_UP        : Exit('UP');
    VKEY_DOWN      : Exit('DOWN');
    VKEY_LEFT      : Exit('LEFT');
    VKEY_RIGHT     : Exit('RIGHT');
    VKEY_CENTER    : Exit('CENTER');
    VKEY_F1        : Exit('F1');
    VKEY_F2        : Exit('F2');
    VKEY_F3        : Exit('F3');
    VKEY_F4        : Exit('F4');
    VKEY_F5        : Exit('F5');
    VKEY_F6        : Exit('F6');
    VKEY_F7        : Exit('F7');
    VKEY_F8        : Exit('F8');
    VKEY_F9        : Exit('F9');
    VKEY_F10       : Exit('F10');
    VKEY_F11       : Exit('F11');
    VKEY_F12       : Exit('F12');
    VKEY_NUM_ENTER : EXIT('NUM_ENTER');
  end;
  Exit('ERROR');
end;

function TInput.CommandToString(Command : Byte) : string;
var Count : Word;
begin
  for Count := 0 to 255 do
    if CmdSet[Count] = Command then Exit(VCodeToString(Count));
  Exit('ERROR');
end;

function TInput.CommandToVCode(Command : Byte) : Byte;
var Count : Word;
begin
  for Count := 0 to 255 do
    if CmdSet[Count] = Command then Exit(Count);
  Exit(0);  
end;

function TInput.VCodeToCommand(KeyCode : Byte) : byte;
begin
  Exit(CmdSet[KeyCode]);
end;


function TInput.StringToVCode(const KeyCode : string) : byte;
var Code : string[10];
begin
  if (length(KeyCode) = 1) and
     (KeyCode[1] in [chr(32)..chr(126)]) then Exit(Ord(KeyCode[1]));
  Code := UpCase(Trim(KeyCode));
  if Code = 'NONE'      then Exit(VKEY_NONE);
  if Code = 'ESCAPE'    then Exit(VKEY_ESCAPE);
  if Code = 'TAB'       then Exit(VKEY_TAB);
  if Code = 'BACKSPACE' then Exit(VKEY_BACKSPACE);
  if Code = 'ENTER'     then Exit(VKEY_ENTER);
  if Code = 'INSERT'    then Exit(VKEY_INSERT);
  if Code = 'DELETE'    then Exit(VKEY_DELETE);
  if Code = 'HOME'      then Exit(VKEY_HOME);
  if Code = 'END'       then Exit(VKEY_END);
  if Code = 'PAGEUP'    then Exit(VKEY_PAGEUP);
  if Code = 'PAGEDOWN'  then Exit(VKEY_PAGEDOWN);
  if Code = 'UP'        then Exit(VKEY_UP);
  if Code = 'DOWN'      then Exit(VKEY_DOWN);
  if Code = 'LEFT'      then Exit(VKEY_LEFT);
  if Code = 'RIGHT'     then Exit(VKEY_RIGHT);
  if Code = 'CENTER'    then Exit(VKEY_CENTER);
  if Code = 'F1'        then Exit(VKEY_F1);
  if Code = 'F2'        then Exit(VKEY_F2);
  if Code = 'F3'        then Exit(VKEY_F3);
  if Code = 'F4'        then Exit(VKEY_F4);
  if Code = 'F5'        then Exit(VKEY_F5);
  if Code = 'F6'        then Exit(VKEY_F6);
  if Code = 'F7'        then Exit(VKEY_F7);
  if Code = 'F8'        then Exit(VKEY_F8);
  if Code = 'F9'        then Exit(VKEY_F9);
  if Code = 'F10'       then Exit(VKEY_F10);
  if Code = 'F11'       then Exit(VKEY_F11);
  if Code = 'F12'       then Exit(VKEY_F12);
  if Code = 'NUM_ENTER' then Exit(VKEY_NUM_ENTER);
  Exit(0);
end;

function TInput.MacroTest(Key : byte) : byte;
begin
  if Recording      then begin KeyOut.WriteByte(Key); Inc(KeyOutSize); end;
  if (not Macroing) then Exit(Key);
  if not (Macro.Count >= CKeyBufferSize) then Macro.Push(Key);
  Exit(Key);
end;

var INICallbackProcedureReader : TTokenReader;

procedure INICallbackProcedure(const nkey, nvalue : shortstring);
var cmdid,val : byte;
begin
  Input.Log(nkey+'->'+nvalue);
  cmdid := INICallbackProcedureReader(nkey);
  if cmdid = 0 then Input.Warning('Unknown command key "'+nkey+'" in INI file!');
  val   := Input.StringToVCode(nvalue);
  if val   = 0 then Input.Warning('Value NONE assigned to command "'+nkey+'" in INI file!');
  Input.cmdset[val] := cmdid;
end;

procedure TInput.Load(INIFile : TINI; CommandReader : TTokenReader);
begin
  INICallbackProcedureReader := CommandReader;
  INIFile.QuerySection(@INICallbackProcedure);
end;

function TInput.KeyPressed : Boolean;
begin
  if Playback then Exit(True);
  Exit(PollKey);
end;

procedure TInput.ResetKeyBuffer;
var RecordBack : Boolean;
begin
  if Playback then Exit;
  RecordBack := Recording;
  if Recording then RecorderStop;
  while PollKey do GetKey;
  if RecordBack then RecorderStart;
end;


procedure TInput.Load(var T : Text; CommandReader : TTokenReader);
var RS    : string[80];
    cmd   : string[40];
    cmdid : Byte;
    val   : Byte;
    
begin
  repeat
    Readln(T,RS);
    RS := Trim(RS);
    if RS = '' then Break;
    cmd   := Trim(Upcase(Copy2Symb(RS,'=')));
    cmdid := CommandReader(cmd);
    if cmdid = 0 then Warning('Unknown command "'+cmd+'" read from textfile');
    val   := StringToVCode(Trim(ExtractDelimited(2,RS,['"'])));
    if val   = 0 then Warning('Value NONE assigned to "'+cmd+'" in textfile');
    cmdset[val] := cmdid;
  until (RS = '') or EOF(T);
end;

function TInput.MacroToString(const theMacro : TKeyMacro) : string;
var pos : byte;
    ms  : string;
begin
  pos := 0;
  ms  := '';
  while (theMacro[pos] <> 0) do
  begin
    ms := ms + ' ' + VCodeToString(theMacro[pos]);
    Inc(pos);
  end;
  Delete(ms,1,1);
  Exit(ms);
end;

function TInput.StringToMacro(const theMacro : string) : TKeyMacro;
var pos : byte;
    mc  : TKeyMacro;
    ms  : string;
begin
  for pos := 0 to CKeyBufferSize-1 do mc[pos] := 0;
  pos := 1;
  ms := '';
  repeat
    ms := ExtractDelimited(pos,theMacro,[' ']);
    mc[pos-1] := StringToVCode(ms);
    Inc(pos);
  until (ms = '');
  Exit(mc);
end;

end.

{
LOG:
15-JAN-2006 INIFile support resupplied
14-JAN-2006 Ported to Valkyrie 2 (TODO: INIFile support and Interrupt)
}
