{$INCLUDE valkyrie.inc}
// @abstract(Systems management for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(June 6, 2004)
// @lastmod(Jan 15, 2006)
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>
unit vini;
interface
uses vnode, vds, vutil;

// Callback procedure for mass parsing ini entries.
type TINIFileCallbackProcedure = procedure (const nkey,nvalue : string);

// INI Entry Type type ;-).
type TINIEntryType = (typeString,typeNumber,typeBoolean,typeFlags);

// INI Entry type. EValue holds either a Integer, Boolean or Flags, or a AnsiString
type TINIEntry = record
                   a : AnsiString;
                   case EType : TINIEntryType of
                     typeNumber  : (n : LongInt;);
                     typeBoolean : (b : boolean;);
                     typeFlags   : (f : TFlags;);
                 end;
     PINIEntry = ^TINIEntry;

type TEntryAssocArray = specialize TAssocArray<TINIEntry>;
type TINIArray = specialize TAssocArray<TEntryAssocArray>;

     
// The INI file class.
type TINI = class(TINIArray)
       // Initializes a plain INI file data holder. Useful for constructing an
       // INI file in memory and then printing it to a file.
       constructor Create; reintroduce;
       // Loads a INI file.
       constructor Load(const FileName : string; Created : Boolean = False);
       // Creates a new section of the given name.
       procedure addSection(const Name : string); virtual;
       // Disposes properly of the allocated data.
       destructor Destroy; override;
       // Changes current section.
       procedure setSection(const Name : string);
       // Tries to change section. Returns wether change succeded (Section found).
       function trySetSection(const Name : string) : boolean;
       // Adds a string entry to given section.
       procedure add(const nKey : string; const nValue : Ansistring; Section : TEntryAssocArray = nil );virtual;
       // Adds a boolean entry to given section.
       procedure add(const nKey : string; const nValue : boolean; Section : TEntryAssocArray = nil );virtual;
       // Adds a integer entry to given section.
       procedure add(const nKey : string; const nValue : integer; Section : TEntryAssocArray = nil );virtual;
       // Adds a flags entry to given section.
       procedure add(const nKey : string; const nValue : TFlags; Section : TEntryAssocArray = nil );virtual;
       // Adds a raw INI file entry, type depending of first character. Can be
       // overriden to provide additional options.
       procedure addRaw(const nKey : string; const nValue : Ansistring); virtual;
       // Returns wether the key is defined in the current section.
       function Defined(const nkey : string) : boolean;
       // Returns a number from the current section.
       function getNumber(const nkey : string) : LongInt;
       // Returns a boolean from the current section.
       function getBoolean(const nkey : string) : Boolean;
       // Returns a String from the current section.
       function getString(const nkey : string) : AnsiString;
       // Returns Flags from the current section.
       function getFlags(const nkey : string) : TFlags;
       // Returns entry converted to string.
       function get(const nkey : string) : AnsiString;
       // Resolves a flagstring based on the [FLAGS] section.
       function ResolveFlags(const flagString : string) : TFlags;
       // Queries current section with given callback.
       procedure QuerySection(inicallback : TINIFileCallbackProcedure);
       // Converts entry of any type to string.
       function EntryToString(const Entry : TINIEntry) : Ansistring;
       // DisposeOf override
       procedure DisposeOf(Value : TEntryAssocArray); override;
       property Num    [index : string] : LongInt    read getNumber  write add;
       property Bool   [index : string] : Boolean    read getBoolean write add;
       property Str    [index : string] : AnsiString read getString  write add;
       property Flag   [index : string] : TFlags     read getFlags   write add;
       property General[index : string] : AnsiString read get; default;
       protected
       Selected  : TEntryAssocArray;
       Flags     : TEntryAssocArray;
       Constants : TEntryAssocArray;
     end;

implementation

uses sysutils, strutils;

constructor TINI.Create;
begin
  inherited Create( False );
  Selected := nil;
end;

constructor TINI.Load(const FileName : string; Created : Boolean = False);
var INI : Text;
    S,V : String;
    IP  : Byte;
begin
  if not Created then Create;
  if not FileExists(FileName) then CritError('File "'+Filename+'" not found!');
  Assign(INI,FileName);
  Reset(INI);
  while not EOF(INI) do
  begin
    Readln(INI,S);
    if S = '' then Continue;
    if S[1] = '#' then
    begin
      if Copy2Symb(s,' ') = '#include' then Load(ExtractDelimited(2,s,['"']),True);
      Continue;
    end;
    IP := pos('//',S);
    if IP <> 0 then Delete(S,IP,Length(S)-IP+1);
    S := Trim(S);
    if S = '' then Continue;
    if S[1] = '#' then Continue;
    if length(s) < 3 then Continue;
    if (S[1] = '[') and (S[length(S)] = ']') then
    begin
      AddSection(MidStr(S,2,Length(S)-2));
      Continue;
    end;
    if Selected = nil then CritError('Key without section in "'+Filename+'"!');
    S := Copy2Symb(S,';');
    V := S;
    S := UpCase(Trim(Copy2Symb(s,'=')));
    Delete(V,1,Pos('=',V));
    V := Trim(V);
    AddRaw(S,V);
  end;
  Close(INI);
  Selected := nil;
end;

procedure TINI.addSection(const Name : string);
var SectName : string[40];
begin
  SectName := UpCase(Name);
  if Exists(SectName) then CritError('Trying to re-create section '+SectName+'!');
  addEntry(SectName,TEntryAssocArray.Create(True));
  Selected := TEntryAssocArray(FLastEntry^.Value);
  if SectName = 'FLAGS'     then Flags  := Selected;
  if SectName = 'CONSTANTS' then Constants := Selected;
end;

procedure TINI.setSection(const Name : string);
var SectName : string[40];
begin
  SectName := UpCase(Name);
  if not TrySetSection(SectName) then CritError('Section ['+SectName+'] not found!');
end;

function TINI.trySetSection(const Name : string) : boolean;
var SectName : string[40];
begin
  SectName := UpCase(Name);
  trySetSection := Exists(SectName);
  if trySetSection then Selected := TEntryAssocArray(FLastEntry^.Value);
end;

function TINI.Defined(const nKey : string) : boolean;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Exit(Selected.Exists(UpCase(nKey)));
end;

procedure TINI.add(const nKey : string; const nValue : Ansistring; Section : TEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeString; Entry.a := nValue;
  Section[UpCase(nKey)] := Entry;
end;

procedure TINI.add(const nKey : string; const nValue : Boolean; Section : TEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeBoolean; Entry.b := nValue;
  Section[UpCase(nKey)] := Entry;
end;

procedure TINI.add(const nKey : string; const nValue : Integer; Section : TEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeNumber; Entry.n := nValue;
  Section[UpCase(nKey)] := Entry;
end;

procedure TINI.add(const nKey : string; const nValue : TFlags; Section : TEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeFlags; Entry.f := nValue;
  Section[UpCase(nKey)] := Entry;
end;

procedure TINI.addRaw(const nKey : string; const nValue : Ansistring);
var V     : Ansistring;
    Entry : TINIEntry;
begin
 V := nValue;
 case V[1] of
  '#' : begin
          if length(V) < 2 then CritError('INI file error: '''+nKey+' => '+V+''' - const value error!');
          Delete(V,1,1);
          Entry := Constants[UpCase(V)];
          if Entry.EType <> typeNumber then CritError('Constant "'+V+'" not a number!');
          Add(nKey,Entry.n);
        end;
  't','T','f','F' : Add(nKey,StrToBool(V));
  '0'..'9', '-' : Add(nKey,StrToInt(V));
  '"' : Add(nKey,ExtractDelimited(2,V,['"']));
  '[' : Add(nKey,ResolveFlags(ExtractDelimited(2,V,['[',']'])));
 end;
end;

function TINI.getNumber(const nkey : string) : LongInt;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected[UpCase(nkey)];
  if Entry.EType <> typeNumber then CritError('Key "'+nkey+'" should be of type "Number"!');
  Exit(Entry.n);
end;

function TINI.getBoolean(const nkey : string) : Boolean;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected[UpCase(nkey)];
  if Entry.EType <> typeBoolean then CritError('Key "'+nkey+'" should be of type "Boolean"!');
  Exit(Entry.b);
end;

function TINI.getString(const nkey : string) : AnsiString;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected[UpCase(nkey)];
  if Entry.EType <> typeString then CritError('Key "'+nkey+'" should be of type "String"!');
  Exit(Entry.a);
end;

function TINI.getFlags(const nkey : string) : TFlags;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected[UpCase(nkey)];
  if Entry.EType <> typeFlags then CritError('Key "'+nkey+'" should be of type "Flags"!');
  Exit(Entry.f);
end;

function TINI.ResolveFlags(const flagString : string) : TFlags;
var FFlag      : string[40];
    FFlags     : TFlags;
    FCount     : Word;
    FlagStr    : string;
    FEntry     : TINIEntry;
begin
  if Flags = nil then CritError('No Flags Section selected when quering flagstring "'+flagstring+'"!');
  FFlags := [];
  if FlagString = '' then Exit(FFlags);
  FlagStr := UpCase(FlagString);
  FCount := 0;
  repeat
    Inc(FCount);
    FFlag := Trim(ExtractDelimited(FCount,FlagStr,['|']));
    if FFlag = '' then Break;
    FEntry := Flags[FFlag];
    if FEntry.EType <> typeNumber then CritError('Flag "'+FFlag+'" not a number!');
    Include(FFlags,FEntry.n);
  until FFlag = '';
  Exit(FFlags);
end;


procedure TINI.QuerySection(inicallback : TINIFileCallbackProcedure);
var Count : Word;
begin
  if Selected = nil then CritError('No Section selected!');
  with Selected do
  for Count := 0 to 95 do
  begin
    FLastEntry := FEntries[Count];
    while FLastEntry <> nil do
    begin
      inicallback(FLastEntry^.Key,EntryToString(FLastEntry^.Value));
      FLastEntry := FLastEntry^.Next;
    end;
  end;
end;

// Converts entry of any type to string.
function TINI.EntryToString(const Entry : TINIEntry) : Ansistring;
var c : byte;
begin
  case Entry.EType of
    typeString  : Exit(Entry.a);
    typeNumber  : Exit(IntToStr(Entry.n));
    typeBoolean : Exit(BoolToStr(Entry.b));
    typeFlags   : begin
                    EntryToString := '';
                    for c := 0 to 255 do
                      if c in Entry.f then
                        EntryToString += IntToStr(c)+ ' ';
                    Delete(EntryToString,Length(EntryToString),1);
                  end;
  end;
end;

procedure TINI.DisposeOf(Value: TEntryAssocArray);
begin
  FreeAndNil(Value);
end;

// Returns entry converted to string.
function TINI.get(const nkey : string) : AnsiString;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Exit(EntryToString(Selected[UpCase(nkey)]));
end;


destructor TINI.Destroy;
begin
  inherited Destroy;
end;

end.

